# Specific

Generate test doubles using clojure.spec

## Why?

Testing code that has side effects can be painful. Mocking out those interactions is a great way to keep your tests fast and reliable. Using clojure.spec, we can automatically generate those mocks.

## Dependencies

_Specific_ depends on Clojure 1.9 (or 1.8 with the clojure.spec backport). To add the following to your project.clj file to add it to your project.

```clojure
(defproject sample
  :dependencies [[org.clojure/clojure "1.9.0"]]
  :profiles {:dev {:plugins [[com.benrady/specific "0.0.1-SNAPSHOT"]]}})
```

## Usage

### Mock Functions

Mocking a function prevents the original function from being called, which is useful when you want to prevent side effects in a test, but still want to ensure it was invoked properly. Mocked functions validate their arguments against any specs defined for the original function. You can also add more constraints within the scope of the test.

Additionally, you can specify a subset of data as a return value, and let _Specific_ generate the rest of the data using the function's spec.

### Stub Functions

Stubbing a function also prevents the original function from being invoked, but is much more liberal in what it accepts, and conservative in what it sends. Only the specs defined on the original function will be enforced. 

The data generated by a stub or mock will only change when the spec itself changes. 

### Spy Functions

Spy functions call through to the original function, but enforce still the constraints in the function's spec. You can make assertions about how the spy was invoked after it is called.

```clojure
  (testing "spy functions"
    (with-spy [some-fun]

      (testing "tracks the arguments of each call"
        (some-fun "hello")
        (is (= [["hello"]] (calls some-fun))))))
```

## License

Copyright (C) 2016 Ben Rady <benrady@gmail.com>

This program is free software; you can redistribute it and/or modify it under the terms of the [GNU General Public License version 2](https://www.gnu.org/licenses/old-licenses/gpl-2.0.en.html) as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
