(ns com.beardandcode.forms
  (:require [dommy.core :as dommy :refer-macros [sel1 sel]]))

(extend-type js/HTMLCollection
  ISeqable
  (-seq [array] (array-seq array 0)))

(defn- make-link [class text]
  (let [element (dommy/create-element :a)
        label (dommy/create-text-node text)]
    (dommy/set-class! element class)
    (dommy/set-attr! element "href" "#")
    (dommy/append! element label)))

(defn- prevent-default [wrapped-fn]
  (fn [evt]
    (.preventDefault evt)
    (wrapped-fn evt)))

(defn- adjust-list! [list num-items template]
  (let [current-items (sel list "li")
        difference (- (count current-items) num-items)]
    (cond (pos? difference) ;; too many, remove some
          (doseq [elem (take difference current-items)]
            (dommy/remove! elem))

          (neg? difference) ;; too few, add some
          (dotimes [n (Math/abs difference)]
            (dommy/append! list (.cloneNode template true))))))

(defmulti make-item-template (fn [_ details] (:type details)))
(defmethod make-item-template "string" [list item-details]
  (let [cloned (-> list dommy/children first (.cloneNode true))
        remove-link (make-link "remove-item" (str "Remove this " (:title item-details)))
        input (sel1 cloned "input[type=text]")]
    (dommy/remove-attr! input "value")
    (dommy/remove-attr! input "name")
    (dommy/append! cloned remove-link)
    cloned))

(defmulti update-item! (fn [_ _ _ details] (:type details)))
(defmethod update-item! "string" [elem index value item-details]
  (dommy/set-attr! elem "data-index" (str index))
  (let [input (sel1 elem "input[type=text]")]
    (aset input "value" value)
    (dommy/set-attr! input "name" (str (:path item-details) "_" index))))

(defmulti refresh-state (fn [_ details] (:type details)))
(defmethod refresh-state "string" [list _]
  (mapv #(.-value %) (sel list "input[type=text]")))

(defmulti refresh-ui (fn [_ _ _ details] (:type details)))
(defmethod refresh-ui "string" [list values template item-details]
  (adjust-list! list (count values) template)
  (doall (map-indexed (fn [index [elem value]]
                        (update-item! elem index value item-details))
                      (map vector (sel list "li") values))))

(defn ^:export array-entry [selector]
  (let [list (sel1 selector)
        item-details {:title (or (dommy/attr list "data-item-title") "item")
                      :path (dommy/attr list "data-item-path")
                      :type (dommy/attr list "data-item-type")
                      :min (or (dommy/attr list "data-min-items") (.-NEGATIVE_INFINITY js/Number))
                      :max (or (dommy/attr list "data-max-items") (.-POSITIVE_INFINITY js/Number))}
        item-template (make-item-template list item-details)
        add-link (make-link "add-item" (str "Add another " (:title item-details)))
        state (atom)
        current-state (refresh-state list item-details)]
    (dommy/clear! list)
    (dommy/append! (dommy/parent list) add-link)
    (dommy/listen! list :change (fn [_] (reset! state (refresh-state list item-details))))
    (dommy/listen! list :click (prevent-default
                                (fn [evt]
                                  (let [target (.-target evt)
                                        index (int (dommy/attr (dommy/parent target) "data-index"))]
                                    (if (and (= (.-tagName target) "A") index)
                                      (swap! state #(if (> (count %) (:min item-details))
                                                      (into [] (concat (subvec % 0 index)
                                                                       (subvec % (inc index))))
                                                      %)))))))
    (dommy/listen! add-link :click (prevent-default
                                    (fn [_]
                                      (swap! state #(if (< (count %) (:max item-details))
                                                               (conj % nil) %)))))
    (add-watch state :refresh (fn [_ _ _ new-state]
                                (refresh-ui list new-state item-template item-details)))
    (reset! state current-state)))
