(ns com.beardandcode.forms.schema
  (:import [com.beardandcode.forms Schema])
  (:require [clojure.java.io :as io]
            [cheshire.core :as json]))

(defprotocol ISchema
  (as-map [_])
  (validate [_ instance enum-fns]))

(defn- slashify [parts]
  (str "/" (clojure.string/join "/" parts)))

(defn- pointer [error & suffixes]
  (let [parts (filter #(not (or (empty? %) (= "properties" %)))
                      (clojure.string/split ((error "instance") "pointer") #"/"))]
    (slashify (concat parts suffixes))))

(defn- parse-raw-error [error]
  (case (error "keyword")
    "required" (reduce #(assoc %1 (pointer error %2) #{:required}) {} (error "missing"))
    "format" {(pointer error) #{(keyword (str "invalid-" (error "attribute")))}}
    "enum" {(pointer error) #{:value-not-in-enum}}
    "pattern" {(pointer error) #{:does-not-match-pattern}}
    :default {}))

(defn- all-map-leaves
  ([root] (all-map-leaves root []))
  ([root prefix]
   (mapcat (fn [[k v]]
             (if (map? v) (all-map-leaves v (conj prefix k))
                 [[(conj prefix k) v]]))
           root)))

(defn- is-in? [coll v]
  (or (and (sequential? coll) (some #{v} coll))
      (contains? coll v)))

(extend Schema
  ISchema
  {:as-map #(.asMap %)
   :validate (fn [s instance enum-fns]
               (let [schema-errors (->> (json/generate-string instance)
                                        (.validate s)
                                        (json/parse-string)
                                        (map parse-raw-error)
                                        (apply merge-with clojure.set/union))
                     enum-errors (->> enum-fns
                                      all-map-leaves
                                      (reduce (fn [errors [k enum-fn]]
                                                (if-let [v (get-in instance k)]
                                                  (when-not (is-in? (enum-fn) v)
                                                    (assoc errors (slashify k) #{:value-not-in-enum}))))
                                              {}))]
                 (when-not (and (empty? schema-errors) (empty? enum-errors))
                   (merge-with clojure.set/union schema-errors enum-errors))))})

(defn new [path]
  (if-let [schema-url (io/resource path)]
    (let [schema (Schema. (slurp schema-url))]
      (if (.isValid schema) schema))))
