(ns utils.heal
  (:require [schema.core :as s]))

(defmulti heal!
  "Provides a solution for a group of users
  suffering from the same problem. Important to deal
  with them as a group because some need, for example, to
  be combined into a single email notification."
  (fn [ailment system users] ailment))

(defmulti sick?
  "Examination function takes an ailment by keyword and a
  record. Returns true if the record is in fact sick."
  (fn [ailment _] ailment))

(def Ailment s/Keyword)
 
(def PotentiallySickRecord {s/Any s/Any})

(def Patient {:record PotentiallySickRecord
              (optional-key :diagnosis) [Ailment]})

(def Diagnosis s/Keyword)

(defn- diagnose* [{:keys [record] :as patient} ailment]
  (if (sick? ailment record)
    (update-in patient [:diagnosis] conj ailment)
    patient))

(s/defn diagnose
  [diagnoses :- [Diagnosis]
   record :- PotentiallySickRecord] :- [Patient]
  (reduce (fn [memo ailment]
            (merge memo (diagnose* memo ailment)))
    {:record record :diagnosis []}
    diagnoses))

(defn- filter-sick [patients]
  (filter (comp not empty? :diagnosis) patients))

(s/defn ^:private group-by-ailments [patients :- [Patient]] :- {Ailment [s/Any]}
  (->> (map :diagnosis patients)
       (flatten)
       (into #{}) ; get distinct ailments
       (reduce (fn [memo ailment]
                 (->> (filter #(some #{ailment} (:diagnosis %)) patients)
                      (map :user)
                      (assoc memo ailment)))
         {})))

(s/defn heal-records!
  "Find records with potential problems and attempt to resolve them."
  [system records diagnoses :- [Diagnosis]]
  (let [groups (->> (map (partial diagnose diagnoses) records)
                    (filter-sick)
                    (group-by-ailments))]
    (doseq [[ailment users] groups]
      (heal! ailment system users))))
