(ns platform-order.storage.adapters.redis
  (:refer-clojure :exclude [pop!])
  (:require [platform-order.storage :as store]
            [platform-order.queue :as q]
            [taoensso.carmine :as car :refer [wcar]]))

;; Manages a queue on a FIFO, left to right, strategy.

(defn- add*! [conn list-name item]
  (->> (car/lpush list-name item)
       (wcar conn)))

(defrecord RedisAdapter [conn listener list-name failures-list-name]
  store/Storage
  (connect! [this] this)

  (disconnect! [this]
    (when-let [l @listener]
      (car/close-listener @listener)
      (reset! listener nil))
    this)

  (add! [_ item]
    (add*! conn list-name item))
   
  (pop! [_]
    (->> (car/rpop list-name)
         (wcar conn)))

  (length [_]
    (->> (car/llen list-name)
         (wcar conn)))
  
  (fetch [this]
    (->> (store/length this)
         (car/lrange list-name 0)
         (wcar conn)))
  
  (fail! [_ item]
    (add*! conn failures-list-name item)))

(defn redis-adapter
  [{:keys [connection-url list-name failures-list-name]}]
  {:pre [(not (nil? connection-url))]}
  (map->RedisAdapter {:list-name list-name
                      :failures-list-name failures-list-name
                      :conn {:pool {}
                             :spec {:uri connection-url}}
                      :listener (atom nil)}))
