(ns com.avisi-apps.gaps.rollbar
  (:require
    ["rollbar" :as Rollbar]
    ["@rollbar/react" :refer [LEVEL_CRITICAL LEVEL_DEBUG LEVEL_ERROR LEVEL_INFO LEVEL_WARN Provider ErrorBoundary]]
    [clojure.string :as str]
    [com.fulcrologic.fulcro.algorithms.react-interop :as react-interop]
    [com.fulcrologic.fulcro.components :refer [defsc] :as comp]
    [com.fulcrologic.fulcro.routing.dynamic-routing :as dr]))

;;
;; Rollbar config
;;
(defonce config (atom nil))
(defn set-config! [m] (reset! config m))

(defn notifier "Creates a new Rollbar notifier" []
  (when-let [c @config]
    (new ^js Rollbar (clj->js c))))

;;
;; Rollbar error levels
;;
(def critical LEVEL_CRITICAL)
(def debug LEVEL_DEBUG)
(def error LEVEL_ERROR)
(def info LEVEL_INFO)
(def warn LEVEL_WARN)

;;
;; Fulcro UI components
;;
(def ui-provider (react-interop/react-factory Provider))
(def ui-error-boundary (react-interop/react-factory ErrorBoundary))

(def ^:dynamic *render-error*
  "A `(fn [js-obj] what-to-render)`. Called in order to render an alternate for UI segments that have crashed.
   Defaults to the simple text message: \"There was an error\""
  (fn [_] "There was an error."))

;; From: com.fulcrologic.fulcro.react.error-boundaries
(defsc BodyContainer
  [_ {:keys [parent render]}]
  {:use-hooks? true}
  (comp/with-parent-context parent
    (render)))

(def ui-body-container (comp/factory BodyContainer))

(defn error-boundary [{:keys [level error-message parent render]}]
  (ui-error-boundary
    {:level (or level error)
     :errorMessage error-message
     :fallbackUI *render-error*}
    (ui-body-container {:parent parent :render render})))

;;
;; Utilities
;;
(defn with-transformers
  "Composes the functions `t` from right to left
  and assigns that to the Rollbar config"
  [c & t] (assoc c :transform (apply comp t)))

(defn fulcro-route-to-context-transformer
  "Given an app atom, returns a composable Rollbar transformer
  that sets the current route as the context"
  [app]
  (fn [^js payload]
    (set! (.-context payload) (str/join "/" (dr/current-route @app)))
    payload))
