(ns com.augustl.data-validation
  (:require clojure.set))

(defn base-err
  [err]
  {:base #{err}})

(defn attr-err
  [attr err]
  {:attrs {attr #{err}}})

(defn run-fn-validator
  [v data args]
  (apply (partial v data) args))

(defn run-sequential-validator
  [v data args]
  (let [attr (first v)
        v-fns (rest v)]
    (some
     (fn [fn-or-set]
       (cond
        (fn? fn-or-set) (if-let [err (apply fn-or-set data attr (attr data) args)]
                          (attr-err attr err))
        (set? fn-or-set) (let [errs (->> (map
                                          #(apply % data attr (attr data) args)
                                          fn-or-set)
                                         (remove nil?))]
                           (if (not (empty? errs))
                             {:attrs {attr (set errs)}}))
        :default (throw (Exception. "Validation must be function or set of functions"))))
     v-fns)))

(defn run-validator
  [v data args]
  (cond
   (fn? v) (run-fn-validator v data args)
   (sequential? v) (run-sequential-validator v data args)
   :default (throw (Exception. "Validator must be function or list."))))

(defn- merge-base-errors
  [err base-errs]
  (update-in err [:base] clojure.set/union base-errs))

(defn- merge-attr-errors
  [err attr-errs]
  (if (empty? attr-errs)
    err
    (let [attr (first (keys attr-errs))]
      (recur
       (update-in err [:attrs attr] clojure.set/union (attr attr-errs))
       (dissoc attr-errs attr)))))

(defn- merge-errors
  ([] nil)
  ([err-a err-b]
     (cond
      (contains? err-a :base) (merge-errors
                               (dissoc err-a :base)
                               (merge-base-errors err-b (:base err-a)))
      (contains? err-a :attrs) (merge-errors
                                (dissoc err-a :attrs)
                                (merge-attr-errors err-b (:attrs err-a)))
      (empty? err-a) err-b)))


(defn validate
  [vs data & args]
  (reduce merge-errors (remove nil? (map #(run-validator % data args) vs))))

(defn validate-partial
  "Only runs validations for attributes present in the data"
  [vs data & args]
  (let [vs-with-data (remove #(and (sequential? %) (not (contains? data (first %)))) vs)]
    (apply (partial validate vs-with-data data) args)))

(defn- call-error-fn
  "Assumes err-fn is a String if it's not a function."
  [err-fn & args]
  (if (fn? err-fn)
    (apply err-fn args)
    err-fn))

(defn validate-non-empty-string
  ([] (validate-non-empty-string "can't be blank"))
  ([err-fn]
     (fn [data attr value & args]
       (if (or (not (= (class value) String))
               (empty? (.trim value)))
         (call-error-fn err-fn)))))

(defn validate-presence
  ([] (validate-presence "must be set"))
  ([error-fn]
     (fn [data attr value & args]
       (if (not (contains? data attr))
         (call-error-fn error-fn)))))

(defn- humanized-list
  [list]
  (apply str (interpose ", " list)))

(defn validate-inclusion
  ([accepted-values]
     (validate-inclusion
      accepted-values
      (fn [accepted-values]
        (str "must be any of " (humanized-list accepted-values)))))
  ([accepted-values error-fn]
     (fn [data attr value & args]
       (if (not (contains? accepted-values value))
         (call-error-fn error-fn accepted-values)))))

(defn validate-only-accept
  ([attrs]
     (validate-only-accept
      attrs
      (fn [extraneous-attrs]
        (str "Unknown attributes: "
             (humanized-list (map name extraneous-attrs))))))
  ([attrs error-fn]
     (let [attrs (set attrs)]
       (fn [data & args]
         (let [provided-attrs (set (keys data))
               extraneous-attrs (clojure.set/difference provided-attrs attrs)]
           (if (not (empty? extraneous-attrs))
             (base-err (call-error-fn error-fn extraneous-attrs))))))))

(defn- assoc-record-list-errors-internal
  [record-list-errors attr-error]
  (cond
   (set? attr-error) (into {} (for [[i errs] record-list-errors]
                                [i (merge-base-errors errs attr-error)]))
   (map? attr-error) (let [all-indices (clojure.set/union (set (keys attr-error))
                                                          (set (keys record-list-errors)))]
                       (into {} (map
                                 (fn [i]
                                   (let [error-a (get record-list-errors i)
                                         error-b (get attr-error i)]
                                     (if (and error-a error-b)
                                       [i (merge-errors error-a error-b)]
                                       [i (or error-a error-b)])))
                                 all-indices)))
   (nil? attr-error) record-list-errors
   :else (throw (Exception. (str "Expected set or map or nil, got " (class attr-error))))))

(defn- list-to-indexed-map
  "Transforms [null 123 null null \"test\"] into {1 123 4 \"test\"}."
  [list]
  (into {} (map-indexed #(if %2 [%1 %2]) list)))

(defn assoc-record-list-errors
  [errors attr record-list-errors-list]
  (update-in errors [:attrs attr] (partial
                                   assoc-record-list-errors-internal
                                   (list-to-indexed-map record-list-errors-list))))

(defn validate-record-list
  [errors data attr validator]
  (let [record-errors (map validator (attr data))]
    (if (every? nil? record-errors)
      errors
      (assoc-record-list-errors errors attr record-errors))))