(ns atomist.json-editor
  (:require #?@(:cljs [[cljs-node-io.core :as io :refer [slurp spit file-seq]]]
                :clj [[clojure.java.io :as io]])
            [clojure.tools.cli :as cli]
            [clojure.edn :as edn]
            [cheshire.core :as json]
            [atomist.log :as log]
            [clojure.string :as string]))

(defn remove-first-v [& args]
  (apply str (drop 1 (first args))))

(defn ->obj [s & {:keys [keywordize-keys] :or {keywordize-keys true}}]
  #?(:cljs (js->clj (js/JSON.parse s) :keywordize-keys keywordize-keys)
     :clj (json/parse-string s keyword)))

(defn ->str [obj]
  #?(:cljs (js/JSON.stringify (clj->js obj) nil 2)
     :clj (json/generate-string obj {:pretty true})))

(defn exit-code [n]
  #?(:clj (System/exit n)
     :cljs (.exit js/process n)))

(defn editor [f path v]
  (-> (slurp f)
      (->obj)
      (assoc-in path v)
      (->str)
      (as-> content (spit f content))))

(defn run-replace [sed s]
  (let [[_ p r] (re-find #"s/(.*)/(.*)/g" sed)]
    (string/replace s (re-pattern p) r)))

(defn -main [& args]
  (try
    (let [{:keys [options arguments summary errors]} (cli/parse-opts args
                                                                     [["-f" "--file FILE"]
                                                                      ["-p" "--path PATH"]
                                                                      [nil "--sed CAPTURE"]])]
      (if (not (empty? errors))
        (do
          (println summary)
          (exit-code 1))
        (let [{:keys [file path sed]} options
              f (io/file file)]
          (if (.exists f)
            (do
              (editor f (edn/read-string path) (run-replace sed (first arguments)))
              (exit-code 0))))))
    (catch #?(:clj Throwable :cljs :default) ex
      (log/error ex "error ")
      (exit-code 1))))

