(ns atomist.cli
  (:require #?(:cljs [cljs.core.async :refer [<! >! chan] :refer-macros [go]]
               :clj [clojure.core.async :refer [go >! <! chan]])
            #?@(:cljs [["js-yaml" :as yaml]
                       [cljs-node-io.core :as io :refer [slurp spit file-seq]]
                       [cljs.spec.alpha :as spec]
                       [goog.crypt.base64 :as b64]]
                :clj [[clj-yaml.core :as yaml]
                      [clojure.java.io :as io]
                      [clojure.spec.alpha :as spec]
                      [base64-clj.core :as b64]])
            [atomist.zip :as zip]
            [clojure.string :as s]
            [clojure.edn :as edn]
            [atomist.log :as log]
            [atomist.skillspec]
            [atomist.promise :as promise])
  #?@(:cljs []
      :clj [(:import (org.apache.commons.io FilenameUtils))]))

(defn yaml-load [f]
  #?(:cljs (js->clj (.safeLoad yaml (slurp f)) :keywordize-keys true)
     :clj (yaml/parse-string (slurp f) :keywords true)))

(defn yaml-dump [obj f]
  #?(:cljs (spit f (.safeDump yaml (clj->js obj)))
     :clj (spit f (yaml/generate-string obj))))

(defn get-extension [f]
  #?(:clj (str "." (FilenameUtils/getExtension (.getName f)))
     :cljs (.getExt f)))

(defn b64-encode [s]
  #?(:cljs (b64/encodeString s)
     :clj (b64/encode s)))

(defn process-skill-metadata [root]
  (go
    (let [dest (io/file root ".atomist" "skill.yaml")
          skill (or
                 (let [skill (io/file root "skill.yaml")]
                   (if (.exists skill)
                     (-> skill
                         (yaml-load))))
                 (let [skill (io/file root "skill.edn")]
                   (if (.exists skill)
                     (-> skill
                         (slurp)
                         (edn/read-string)))))]

      (log/info "... add subscriptions into " dest)
      (let [subscriptions (->> (for [f #?(:cljs (file-seq (.getAbsolutePath (io/file root "graphql/subscription")))
                                          :clj (file-seq (io/file root "graphql/subscription")))
                                     :when (= (get-extension (io/file f)) ".graphql")]
                                 (do
                                   (log/info "add " f)
                                   (slurp f)))
                               (into []))]
        (log/infof "... found %d subscriptions" (count subscriptions))
        (-> skill
            (update :subscriptions (fnil concat []) subscriptions)
            (update :longDescription (fn [s] (or s (slurp (io/file root "description.md")))))
            (merge (let [readme (io/file root "README.md")]
                     (if (.exists readme)
                       (-> {:readme (-> (slurp readme)
                                        (b64-encode))}))))
            (as-> skill (do
                          (log/info "... validate skill metadata")
                          (if (not (spec/valid? :skill/spec skill))
                            (log/warn (with-out-str (spec/explain :skill/spec skill))))
                          skill))
            (yaml-dump dest))))))

(defn bundle [path]
  (promise/chan->promise
   (go
     (let [root (io/file path)]
       (try

         (log/info "... in " (.getAbsolutePath root))
         (.mkdir (io/file root ".atomist"))

         (log/info "... create skill metadata in .atomist")
         (<! (process-skill-metadata root))

         (log/info "... build archive of skill.zip")
         #?(:cljs (<! (zip/archive (io/file root ".atomist" "skill.zip")))
            :clj {:result {".atomist/skill.zip"
                           (zip/zip-root {:git-root root
                                          :zip-file (io/output-stream (io/file root ".atomist" "skill.zip"))})}})
         (catch #?(:clj Throwable :cljs :default) ex
           (log/error ex "failed to bundle")))))))

(comment
  (println "result:  " @(bundle "/Users/slim/atmhq/internal-skill")))

(defn register []
  (promise/chan->promise
   (go
     (try
       (let [])
       (catch #?(:clj Throwable :cljs :default) ex
         (log/error ex ": failed to register"))))))
