(ns atomist.cli
  (:require #?(:cljs [cljs.core.async :refer [<! >! chan] :refer-macros [go]]
               :clj [clojure.core.async :refer [go >! <! chan]])
            #?@(:cljs [["js-yaml" :as yaml]
                       [cljs-node-io.core :as io :refer [slurp spit file-seq]]
                       [cljs.spec.alpha :as spec]
                       [goog.crypt.base64 :as b64]
                       [cljs.pprint :refer [pprint]]
                       [goog.string :as gstring :refer [format]]
                       [goog.string.format]
                       [cljs-node-io.file :as file]]
                :clj [[clj-yaml.core :as yaml]
                      [clojure.java.io :as io]
                      [clojure.spec.alpha :as spec]
                      [base64-clj.core :as b64]
                      [clojure.pprint :refer [pprint]]])
            [atomist.zip :as zip]
            [clojure.string :as s]
            [clojure.edn :as edn]
            [atomist.log :as log]
            [atomist.json :as json]
            [atomist.skillspec]
            [atomist.promise :as promise]
            [clojure.tools.cli :as cli]
            [clojure.string :as string])
  #?@(:cljs []
      :clj [(:import (org.apache.commons.io FilenameUtils))]))

(defn yaml-load [f]
  #?(:cljs (js->clj (.safeLoad yaml (slurp f)) :keywordize-keys true)
     :clj (yaml/parse-string (slurp f) :keywords true)))

(defn yaml-dump [obj f]
  #?(:cljs (spit f (.safeDump yaml (clj->js obj)))
     :clj (spit f (yaml/generate-string obj))))

(defn get-extension [f]
  #?(:clj (str "." (FilenameUtils/getExtension (.getName f)))
     :cljs (.getExt f)))

(defn get-name [f]
  #?(:clj (.getName f)
     :cljs (.getName f)))

(defn b64-encode [s]
  #?(:cljs (b64/encodeString s)
     :clj (b64/encode s)))

(defn filter-tags-from-readme [s]
  (let [[_ content] (re-find #"<!---atomist-skill-readme:start--->([\s\S]*)<!---atomist-skill-readme:end--->" s)]
    (or content s)))

(defn cljc-file-seq
  [parent child]
  (let [f (io/file parent child)]
    (if (.exists f)
      #?(:cljs (file-seq (.getAbsolutePath f))
         :clj (file-seq f))
      [])))

(defn graphql-files
  [parent & kinds]
  (if-let [kind (some #(and (.exists (io/file parent))
                            (.exists (io/file parent (str "graphql/" %))) %) kinds)]
    (->> (for [f (cljc-file-seq parent (str "graphql/" kind))
               :when (= (get-extension (io/file f)) ".graphql")]
           (do
             (log/infof "adding %s: %s" kind f)
             (slurp f)))
         (into []))
    []))

(defn lookup-triggers [triggers]
  (if (seq triggers)
    (->> (for [trigger triggers]
           #?(:cljs (io/slurp (gstring/format "node_modules/@atomist/skill-bundler/graphql/subscription/%s.graphql" trigger))
              :clj (slurp (format "graphql/subscription/%s.graphql" trigger))))
         (into []))
    []))

(defn find-datalog-subscriptions [dir]
  (if (.exists dir)
    (->> (for [f (cljc-file-seq dir "datalog/subscription")
               :when (= (get-extension (io/file f)) ".edn")]
           [(get-name (io/file f)) #?(:cljs (io/slurp f)
                                      :clj (slurp f))])
         (into []))))

(defn process-skill-metadata [root]
  (go
    (let [dest (io/file root ".atomist" "skill.yaml")
          skill (or
                  (let [skill (io/file root "skill.yaml")]
                    (if (.exists skill)
                      (-> skill
                          (yaml-load))))
                  (let [skill (io/file root "skill.edn")]
                    (if (.exists skill)
                      (-> skill
                          (slurp)
                          (edn/read-string)))))]

      (log/info "... add subscriptions into " dest)
      (let [subscriptions (graphql-files root "subscription")
            signals (graphql-files root "signals" "signal")]
        (log/infof "... found %d subscriptions" (count subscriptions))
        (log/infof "... found %d signals" (count signals))
        (-> skill
            (update :subscriptions (fnil concat []) (concat subscriptions (lookup-triggers (:triggers skill))))
            (update :signals (fnil concat []) signals)
            (merge (if-let [datalog-subscriptions (find-datalog-subscriptions root)]
                     {:datalogSubscriptions (->> datalog-subscriptions
                                                 (map (fn [[name query]]
                                                        {:name name :query query}))
                                                 (into []))}))
            (merge (if (.exists (io/file root "resources/schema.edn"))
                     {:schemata [{:name "resources/schema.edn"
                                  :schema #?(:cljs (io/slurp "resources/schema.edn")
                                             :clj (slurp "resources/schema.edn"))}]}))
            (update :namespace (fn [s] s))
            (update :longDescription (fn [s] (or s
                                                 (let [f (io/file root "long-description.md")
                                                       f1 (io/file root "description.md")]
                                                   (if-let [f (or (and (.exists f) f)
                                                                  (and (.exists f1) f1))]
                                                     (slurp f)))
                                                 "please create a long-description.md file in the Repo")))
            (merge (let [readme (io/file root "README.md")]
                     (if (.exists readme)
                       (-> {:readme (-> (slurp readme)
                                        (filter-tags-from-readme)
                                        (b64-encode))}))))
            (as-> skill (do
                          (log/info "... validate skill metadata")
                          (if (not (spec/valid? :skill/spec skill))
                            (log/warn (with-out-str (spec/explain :skill/spec skill))))
                          skill))
            (yaml-dump dest))))))

(defn bundle [path]
  (promise/chan->promise
   (go
     (let [root (io/file path)]
       (try

         (log/info "... in " (.getAbsolutePath root))
         (.mkdir (io/file root ".atomist"))

         (log/info "... create skill metadata in .atomist")
         (<! (process-skill-metadata root))

         (log/info "... build archive of skill.zip")
         #?(:cljs (<! (zip/archive (io/file root ".atomist" "skill.zip")))
            :clj {:result {".atomist/skill.zip"
                           (zip/zip-root {:git-root root
                                          :zip-file (io/output-stream (io/file root ".atomist" "skill.zip"))})}})
         (catch #?(:clj Throwable :cljs :default) ex
           (log/error ex "failed to bundle")))))))

(comment
 (println "result:  " @(bundle "/Users/slim/atmhq/internal-skill")))

(defn register []
  (promise/chan->promise
   (go
     (try
       (let [])
       (catch #?(:clj Throwable :cljs :default) ex
         (log/error ex ": failed to register"))))))

(defn- extension [f ext]
  (io/file (.getParentFile f) (string/replace (.getName f) #"(.*)\.(.*)" (format "$1.%s" ext))))

(defn ->edn-str [obj]
  (with-out-str (pprint obj)))

(defn- edn->yaml [f]
  (-> (slurp f)
      (edn/read-string)
      (yaml-dump (io/file (extension f "yaml")))))

(defn- edn->json [f]
  (-> (slurp f)
      (edn/read-string)
      (json/->str)
      (as-> content (spit (io/file (extension f "json")) content))))

(defn- yaml->edn [f]
  (-> (yaml-load f)
      (->edn-str)
      (as-> content (spit (io/file (extension f "edn")) content))))

(defn- json->edn [f]
  (-> (slurp f)
      (json/->obj)
      (->edn-str)
      (as-> content (spit (io/file (extension f "edn")) content))))

(defn tr [argv]
  (promise/chan->promise
   (go
     (try
       (let [{:keys [options summary errors]} (cli/parse-opts (vec (#?(:cljs array-seq
                                                                       :clj seq) argv)) [[nil "--transform TRANSFORM"]
                                                                                         [nil "--file FILE"]])]
         (if (not (empty? errors))
           (log/info summary)
           (condp = (edn/read-string (:transform options))
             :edn->yaml (edn->yaml (io/file (:file options)))
             :yaml->edn (yaml->edn (io/file (:file options)))
             :json->edn (json->edn (io/file (:file options)))
             :edn->json (edn->json (io/file (:file options))))))
       (catch #?(:clj Throwable :cljs :default) ex
         (log/error ex ": failed to register"))))))

(defn validate [argv]
  (promise/chan->promise
   (go
     (try
       (let [{:keys [options summary errors]} (cli/parse-opts (vec (#?(:cljs array-seq
                                                                       :clj seq) argv)) [])]
         (if (empty? errors)
           (do
             (let [f (io/file "skill.yaml")]
               (log/info "spec " (spec/valid? :skill/spec (yaml-load f)))
               (when-let [s (and (.exists f)
                                 (not (spec/valid? :skill/spec (yaml-load f)))
                                 (with-out-str (spec/explain :skill/spec (yaml-load f))))]
                 (log/info s)
                 (throw (ex-info "failed to validate skill.yaml" {:summary :failure}))))
             (let [f (io/file "skill.edn")]
               (when-let [s (and (.exists f)
                                 (not (spec/valid? :skill/spec (-> (slurp f) (edn/read-string))))
                                 (with-out-str (spec/explain :skill/spec (-> (slurp f) (edn/read-string)))))]
                 (log/info s)
                 (throw (ex-info "failed to validate skill.edn" {:summary :failure})))))
           (do
             (log/info summary)
             (throw (ex-info "failed to run" {:summary summary
                                              :errors errors}))))
         (clj->js {:summary :okay}))
       (catch #?(:clj Throwable :cljs :default) ex
         (log/error ex (ex-data ex))
         (clj->js (ex-data ex)))))))
