(ns atomist.functions.package
  (:require [atomist.api :as api]
            [atomist.graphql :as graphql]
            [atomist.functions.manage :as manage]
            [clojure.pprint :refer [pprint]]
            [clojure.java.io :as io]
            [atomist.buckets :as buckets]
            [atomist.zip :as zip]
            [clojure.tools.logging :as log]
            [atomist.json :as json]
            [clojure.tools.cli :refer [parse-opts]]
            [clj-yaml.core :as yaml]
            [clojure.java.shell :as sh]
            [clojure.string :as s]
            [atomist.utils :as utils]
            [atomist.topics :as topics]
            [clojure.edn :as edn])
  (:refer-clojure :exclude [parse-opts]))

(defn add-atomist [git-root]
  (spit (io/file git-root "app.js") (slurp (io/resource "app.js")))
  (let [f (io/file git-root "package.json")]
    (spit f (-> (slurp f)
                (json/->obj)
                (assoc :main "app.js")
                (assoc-in [:dependencies "@google-cloud/pubsub"] "^1.0.0")
                (assoc-in [:dependencies "@google-cloud/storage"] "^4.1.3")
                (json/->str)))))

(defn- pr-edn [x]
  (with-out-str (pprint x)))

(defn- add-distinct [coll x]
  (if (some #(= x %) coll)
    coll
    (conj coll x)))

(defn- check-deps-edn [git-root]
  (letfn [(check-dev-alias [x] (if (not (and (contains? (:aliases x) :dev)
                                             (map? (-> x :aliases :dev))))
                                 (throw (ex-info "missing :dev alias in deps.edn" {:git-root git-root}))
                                 x))
          (add-jvm-opts [x] (update-in x [:aliases :dev :jvm-opts] (fnil add-distinct []) "-Xmx512m"))]
    (let [f (io/file git-root "deps.edn")]
      (spit f (-> (slurp f)
                  (read-string)
                  (check-dev-alias)
                  (add-jvm-opts)
                  (pr-edn))))))

(defn- check-shadow-cljs [git-root]
  (letfn [(check-release-build [x] (if (not (and (map? (-> x :builds :release))
                                                 (= :node-library (-> x :builds :release :target))
                                                 (= "index.js" (-> x :builds :release :output-to))
                                                 (= 'atomist.main/handler (-> x :builds :release :exports :handler))
                                                 (= :advanced (-> x :builds :release :compiler-options :optimizations))))
                                     (throw (ex-info ":release build in shadow-cljs.edn not configured correctly" {:git-root git-root
                                                                                                                   :config x}))
                                     x))
          (check-deps-aliases [x] (if (not (and (-> x :deps)
                                                (some #{:dev} (-> x :deps :aliases))))
                                    (throw (ex-info "shadow-cljs :deps must include the :dev alias from deps.edn" {:git-root git-root
                                                                                                                   :config x}))
                                    x))]
    (let [f (io/file git-root "shadow-cljs.edn")]
      (spit f (-> (slurp f)
                  (read-string)
                  (check-deps-aliases)
                  (check-release-build)
                  (pr-edn))))))

(defn run-npm [dir & args]
  (sh/with-sh-dir
    dir
    (let [{:keys [out err exit]} (apply sh/sh args)]
      (log/info (format "run %s" (apply str args)))
      (when (not (= 0 exit))
        (log/errorf "error running %s" exit)
        (log/info out)
        (log/error err)
        (throw "failed to run npm")))))

(defn package-skill
  [{:keys [git-root] :as request}]
  (log/info "-----> discover subscriptions")
  (let [subscriptions (->> (manage/discover-subscription git-root) :subscriptions (map :subscription) (into []))]

    (log/info "-----> update package.json")
    (add-atomist git-root)
    (log/info "-----> check deps.edn")
    (check-deps-edn git-root)
    (log/info "-----> check shadow-cljs.edn")
    (check-shadow-cljs git-root)

    (log/info "-----> add subscriptions to atomist.yaml or atomist.edn")
    (let [f (io/file git-root "atomist.yaml")]
      (when (.exists f)
        (spit
         f
         (-> (slurp f)
             (yaml/parse-string)
             (assoc :subscriptions subscriptions)
             (update :parameters #(remove empty? %))
             (yaml/generate-string :dumper-options {:flow-style :block})))))
    (let [f (io/file git-root "atomist.edn")]
      (when (.exists f)
        (spit
         f
         (-> (slurp f)
             (edn/read-string)
             (assoc :subscriptions subscriptions)
             (update :parameters #(remove empty? %))
             (pr-edn)))))

    (api/progress request "build index.js using npm run build")
    (run-npm git-root "npm" "ci")
    (run-npm git-root "npm" "run" "build")))

(defn process-namespace-in-name [m n]
  (assoc m
         :name (or (nth (re-find #"(.*)/(.*)" n) 2) n)
         :namespace (or (nth (re-find #"(.*)/(.*)" n) 1) "atomist")))

(defn filter-tags-from-readme [s]
  (let [[_ content] (re-find #"<!---atomist-skill-readme:start--->([\s\S]*)<!---atomist-skill-readme:end--->" s)]
    (or content s)))

(defn request->atomist-skill-input
  "request
     :subscriptions have already been pulled in from filesystem
     :skill, :parameters, and :commands have been pulled in from atomist.yaml
     :artifacts are ignored - we build the artifact section here"
  [request]
  (let [yaml (-> (:skill-metadata request)
                 (assoc-in [:skill :version] (or (:version request) (-> request :skill-metadata :skill :version))))
        skill (process-namespace-in-name (:skill yaml) (-> yaml :skill :name))
        atomist-skill-input (-> skill
                                (assoc :artifacts {:gcf [{:entryPoint (or (-> skill :runtime :entryPoint)
                                                                          (-> skill :runtime :entry_point)
                                                                          "eventhandler")
                                                          :url (:archive-url request)
                                                          :memory (or (-> skill :runtime :memory int) 256)
                                                          :timeout (or (-> skill :runtime :timeout int) 60)
                                                          :runtime (or (-> skill :runtime :runtime) "nodejs10")
                                                          :name (format "%s/%s" (:namespace skill) (:name skill))}]})
                                ;; author
                                (assoc :branchId (:branchId request))
                                (assoc :categories (->> (or (:categories skill) (:category skill))
                                                        (map s/upper-case)))
                                (assoc :commands (-> yaml :commands))
                                (assoc :commitSha (:sha request))
                                ;; description
                                (assoc :dispatchStyle (or (-> skill :dispatchStyle) (-> skill :dispatch)))
                                (assoc :displayName (or (-> skill :displayName) (-> skill :title) (-> request :skill-metadata :name)))
                                (assoc :iconUrl (or (-> skill :iconUrl) (-> skill :icon)))
                                ;; ingesters
                                ;; license
                                (assoc :homepageUrl (or (if (and (:owner request) (:repo request))
                                                          (format "https://github.com/%s/%s" (:owner request) (:repo request)))
                                                        (-> skill :homepageUrl) (-> skill :homepage)))
                                (assoc :longDescription (or
                                                         (let [f (io/file (:git-root request) "description.md")]
                                                           (when (.exists f)
                                                             (slurp f)))
                                                         (-> skill :documentation)
                                                         (-> skill :longDescription)
                                                         (-> skill :description)
                                                         (-> skill :long_description)))
                                ;; name
                                ;; namespace
                                (assoc :parameters (-> yaml :parameters))
                                (merge
                                 (when-let [integrations (-> yaml :integrations)]
                                   {:integrations integrations}))
                                (merge
                                 (when-let [resource-providers (-> yaml :resourceProviders)]
                                   {:resourceProviders resource-providers}))
                                (assoc :readme (-> (slurp (io/file (:git-root request) "README.md"))
                                                   (filter-tags-from-readme)
                                                   (utils/base64)))
                                (assoc :repoId (:repoId request))
                                (assoc :subscriptions (-> yaml :subscriptions))
                                (assoc :technologies (->> (or (-> skill :technologies) (-> skill :technology))
                                                          (map s/upper-case)))
                                ;; version
                                (assoc :videoUrl (or (-> skill :videoUrl) (-> skill :video))))]
    atomist-skill-input))

(defn register-skill [handler]
  (fn [request]
    (try
      (let [response (graphql/run-query
                      request
                      (graphql/register-skill)
                      {:skill (request->atomist-skill-input request)})]
        (log/infof "register-skill response body %s" response)
        (handler
         (assoc request
                :register-response
                response)))
      (catch Throwable t
        (log/error "register-response" (ex-data t))
        (api/status-message request :failure (format "unable to execute skill registration mutation - %s"
                                                     (-> (ex-data t)
                                                         :response
                                                         :body
                                                         :errors)))))))

(defn add-metadata [handler]
  (fn [request]
    (try
      (let [skill-metadata (merge
                            {:subscriptions (->> (manage/discover-subscription (:git-root request))
                                                 :subscriptions
                                                 (map :subscription))}
                            (manage/discover-skill-input (:git-root request))
                            (let [f (io/file (:git-root request) "atomist.edn")]
                              (when (.exists f)
                                (-> (slurp f)
                                    (edn/read-string)))))]
        (handler (assoc request :skill-metadata skill-metadata)))
      (catch Throwable t
        (api/status-message request :failure (format "error adding skill-metadata to registration:  %s" (.getMessage t)))))))

(defn insert-object-in-bucket [handler]
  (fn [request]
    (let [object-name (buckets/gcf-archive-object
                       (:team-id request)
                       (:owner request)
                       (:repo request)
                       (:sha request))
          archive-url (buckets/gcf-archive-url
                       (:bucket-name request)
                       (:team-id request)
                       (:owner request)
                       (:repo request)
                       (:sha request))]
      (try
        (api/progress request (format "store gs archive %s in bucket %s at %s" (:zip-file request) (:bucket-name request) object-name))
        (buckets/insert-object
         (:bucket-name request)
         object-name
         (:dir request)
         (:zip-file request))
        (handler (assoc request :archive-url archive-url))
        (catch Throwable t
          (log/errorf "unable to insert %s into bucket %s %s" (:zip-file request) (:bucket-name request) object-name)
          (.printStackTrace t)
          (api/status-message request :failure (.getMessage t)))))))

(defn create-zip [handler]
  (fn [request]
    (api/progress request (format "zip package %s" (:zip-file request)))
    (zip/zip-root (assoc request :zip-file (io/output-stream (io/file (:git-root request) (:zip-file request)))))
    (handler request)))

(defn metadata [git-root]
  (merge
   {}
   (let [f (io/file git-root "atomist.yaml")]
     (when (.exists f)
       (manage/parse-atomist-yaml (slurp f))))
   (let [f (io/file git-root "atomist.edn")]
     (when (.exists f)
       (-> (slurp f)
           (edn/read-string))))))

(defn check-atomist-yaml-for-skill-package [handler]
  (fn [request]
    (try
      (if (and (or
                (.exists (io/file (:git-root request) "atomist.yaml"))
                (.exists (io/file (:git-root request) "atomist.edn")))
               (#{"atomist/package-cljs-skill"
                  "atomist/package-python-skill"} (-> (metadata (:git-root request))
                                                      :skill
                                                      :package
                                                      :use)))
        (handler request)
        (do
          (log/info "atomist.yaml not set to use skill packaging atomist/package-cljs-skill")
          (api/status-message request :skipped "atomist.yaml not set to use skill packaging atomist/package-cljs-skill")))
      (catch Throwable t
        (api/status-message request :failure (format "bad atomist.yaml: %s" (.getMessage t)))))))

(defn parse-args [handler]
  (let [cli-options [[nil "--dir DIR" "cloned repo for skill"]
                     [nil "--owner OWNER" "GitHub Org"]
                     [nil "--repo REPO" "GitHub Repo name"]
                     [nil "--description DESCRIPTION" "skill description"]]]
    (fn [{:keys [args] :as request}]
      (let [{{:keys [owner repo dir description]} :options errors :errors} (parse-opts args cli-options)]
        (log/infof "%s/%s at %s" owner repo dir)
        (if (not (empty? errors))
          (do
            (log/error "errors: \n\t" (apply str (interpose "\n\t" errors)))
            (api/status-message request :failure "errors parsing skill args %s" (apply str (interpose "\n\t" errors))))
          (handler (assoc request :owner owner :repo repo :dir dir :description description)))))))

(defn read-atomist-payload [handler]
  (letfn [(payload->owner [{:keys [data]}]
            (or (-> data :Push first :repo :owner)
                (-> data :Tag first :commit :repo :owner)))
          (payload->repo [{:keys [data]}]
            (or (-> data :Push first :repo :name)
                (-> data :Tag first :commit :repo :name)))]
    (fn [request]
      (let [payload (-> (io/file (:payload request))
                        (slurp)
                        (json/->obj)
                        (api/event->request))]
        (log/info "extensions " (:extensions payload))
        (log/info "secrets " (map :uri (:secrets payload)))
        (log/info "data " (:data payload))
        (cond
          (contains? (:data payload) :Push)
          (api/status-message request :success "only building Tag events (skip Push)")
          #_(handler
             (-> request
                 (assoc :owner (payload->owner payload) :repo (payload->repo payload))
                 (merge payload {:api-key (->> payload :secrets (filter #(= "atomist://api-key" (:uri %))) first :value)})))
          (contains? (:data payload) :Tag)
          (handler
           (-> request
               (assoc :owner (payload->owner payload) :repo (payload->repo payload))
               (merge payload {:api-key (->> payload :secrets (filter #(= "atomist://api-key" (:uri %))) first :value)})))
          :else
          (do
            (log/info "dir " (:dir request))
            (when-let [f (io/file (:dir request))]
              (log/infof "File %s - %s" (.getPath f) (.exists f)))
            (api/status-message request :success "not building (unrecognized event)")))))))

(defn run-build [handler]
  (fn [request]
    (try
      (api/progress request (format "package cljs skill in %s" (:git-root request)))
      (if (= "atomist/package-cljs-skill" (-> (metadata (:git-root request)) :skill :package :use))
        (package-skill request))
      (handler request)
      (catch Throwable t
        (log/error t "Unable to package the skill")
        (api/status-message request :failure "skill packaging failed")))))

(defn add-tag-version [handler]
  (fn [request]
    (if-let [[_ version] (re-find #"v?(.*)" (-> request :data :Tag first :name))]
      (handler (assoc request :version version))
      (handler request))))

(defn check-environment [handler]
  (letfn [(env [& envs]
            (reduce (fn [agg x]
                      (or agg (System/getenv x))) false envs))]
    (fn [request]
      (let [request
            (cond-> request
              (env "ATOMIST_WORKSPACE_ID" "WORKSPACE_ID") (assoc :team-id (env "ATOMIST_WORKSPACE_ID" "WORKSPACE_ID"))
              (env "ATOMIST_GRAPHQL_ENDPOINT" "GRAPHQL_ENDPOINT") (assoc :graphql-endpoint (env "ATOMIST_GRAPHQL_ENDPOINT" "GRAPHQL_ENDPOINT"))
              (env "ATOMIST_PAYLOAD" "PAYLOAD") (assoc :payload (env "ATOMIST_PAYLOAD"))
              (env "ATOMIST_STORAGE" "STORAGE") (assoc :bucket-name (nth (re-find #"gs://(.*)" (env "ATOMIST_STORAGE" "STORAGE")) 1))
              (env "ATOMIST_TOPIC" "TOPIC") (assoc :topic (env "ATOMIST_TOPIC" "TOPIC")))]
        (if (and (:team-id request) (:topic request) (:graphql-endpoint request) (:payload request) (:bucket-name request))
          (try
            (log/info "environment:  " (select-keys request [:team-id :topic :graphql-endpoint :payload :bucket-name]))
            (topics/init-publisher (:topic request))
            (handler request)
            (finally
              (topics/shutdown-publisher)))
          (do
            (log/warnf "environment is missing some of WORKSPACE_ID GRAPHQL_ENDPOINT ATOMIST_PAYLOAD or ATOMIST_STORAGE %s" (str request))
            (api/status-message request :failure "environment is missing some of WORKSPACE_ID GRAPHQL_ENDPOINT ATOMIST_PAYLOAD or ATOMIST_STORAGE")))))))

(defn add-repo-details [handler]
  (fn [r]
    (let [request (assoc r
                         :git-root (io/file (:dir r))
                         :zip-file "archive.zip"
                         :branch (or (-> r :data :Push first :branch) "master"))]
      (if (and (contains? (:data request) :Push) (not (= "master" (:branch request))))
        (api/status-message request :success "not building non-default branches")
        (try
          (handler (merge request (api/get-repo-details request)))
          (catch Throwable t
            (log/error "unable to query graphql for repo details " t (ex-data t))
            (api/status-message request :failure "stopping because of graphql repo detail query failure")))))))

(defn -main [& args]
  (try
    (log/infof "correlation-id: %s" (or (System/getenv "CORRELATION_ID") (System/getenv "ATOMIST_CORRELATION_ID")))
    ((-> (fn [request]
           (api/status-message request :success
                               (format "registered new version %s of %s"
                                       (or (:version request) (-> request :skill-metadata :skill :version))
                                       (-> request :skill-metadata :skill :name))))
         (register-skill)
         (add-tag-version)
         (add-metadata)
         (insert-object-in-bucket)
         (create-zip)
         (run-build)
         (check-atomist-yaml-for-skill-package)
         (add-repo-details)
         (read-atomist-payload)
         (parse-args)
         (check-environment)) {:args (into [] args)
                               :api_version "1"
                               :correlation-id (or (System/getenv "CORRELATION_ID") (System/getenv "ATOMIST_CORRELATION_ID"))})
    (System/exit 0)
    (catch Throwable t
      (log/error t)
      (System/exit 1))))
