;; Copyright © 2021 Atomist, Inc.
;;
;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at
;;
;;     http://www.apache.org/licenses/LICENSE-2.0
;;
;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns atomist.docker
  (:require [atomist.async :refer-macros [<? go-safe] :refer [map-reduce]]
            [atomist.cljs-log :as log]
            [atomist.json :as json]
            [clojure.data :as data]
            [clojure.string :as str]
            [cljs.core.async :as async]
            ["parse-docker-image-name" :as parse-docker-image-name]
            [goog.string :as gstring]
            [http.client :as client]
            [clojure.set :as set]))

(def timeout 5000)

(defn- bearer-or-basic
  "most of the docker v2 apis use 'Bearer' in the Authorization header but ECR uses 'Basic'"
  [domain]
  (cond
    (str/includes? domain "amazonaws.com") "Basic"
    :else "Bearer"))

(defn- get-image-details-location
  "Weird re-direct thingy"
  [domain access-token repository image-digest]
  (go-safe
   (log/infof "Fetching image-details location for %s/%s (image id %s)" domain repository image-digest)
   (let [image-details (<? (client/get (gstring/format "https://%s/v2/%s/blobs/%s" domain repository image-digest)
                                       {:timeout timeout
                                        :headers {"Authorization" (gstring/format "%s %s" (bearer-or-basic domain) access-token)}}))
         location (-> image-details :headers :location)]
     (if (and location
              (#{307 302 200} (:status image-details)))
       location
       (log/warnf "Could not find image-details location for %s/%s (image id %s)" domain repository image-digest)))))

(defn- get-image-details
  "Fetch labels, ports etc given an image-id"
  [domain access-token repository image-id]
  (go-safe
   (log/infof "Fetching image details for %s/%s@%s" domain repository image-id)
   (if-let [location (<? (get-image-details-location domain access-token repository image-id))]
     (let [metadata (<? (client/get location {:timeout timeout}))]
       (if (= 200 (:status metadata))
         (-> metadata :body (str) (json/->obj))
         (log/warnf "Error fetching details for %s/%s (image id %s) -> %s" domain repository image-id (:status metadata))))
     (log/warnf "Could not find image details for %s/%s (image id %s)" domain repository image-id))))

(defn- get-manifest
  "Fetch a manifest by digest or tag. Can return a manifest or a manifest-list"
  [domain access-token repository tag-or-digest]
  (go-safe
   (log/infof "Fetching manifest %s/%s:%s" domain repository tag-or-digest)
   (let [url (gstring/format "https://%s/v2/%s/manifests/%s" domain repository tag-or-digest)
         request-opts {:timeout timeout
                       :headers {"Authorization" (gstring/format "%s %s" (bearer-or-basic domain) access-token)
                                 "Accept" "application/vnd.docker.distribution.manifest.v2+json,application/vnd.docker.distribution.manifest.list.v2+json"}}
         ;; for ECR, must also do a HEAD request - GET response does not have the docker-content-digest
         ecr-head-response (when (str/includes? domain "amazonaws.com")
                             (<? (client/head url request-opts)))
         response (<? (client/get url request-opts))]
     (if (= 200 (:status response))
       (let [manifest (-> response :body (str) (json/->obj))]
         ;; sometimes we get a list of manifests if the image is multi-platform
         (if (->> response :headers :content-type (= "application/vnd.docker.distribution.manifest.list.v2+json"))
           (log/infof "Found a manifest list for %s/%s:%s" domain repository tag-or-digest)
           (log/infof "Fetched normal manifest %s/%s:%s" domain repository tag-or-digest))
         (assoc manifest :digest (or
                                  (-> response :headers :docker-content-digest)
                                  (-> ecr-head-response :headers :docker-content-digest))))
       (log/warnf "error fetching manifest for %s/%s:%s status %s" domain repository tag-or-digest (:status response))))))

(defn private-repo?
  "Is repository a private repo?"
  [domain repository tag-or-digest]
  (go-safe
   (boolean
    (let [url (gstring/format "https://%s/v2/%s/manifests/%s" domain repository tag-or-digest)
          request-opts {:timeout timeout :headers {"Accept" "application/vnd.docker.distribution.manifest.v2+json,application/vnd.docker.distribution.manifest.list.v2+json"}}
          response (<? (client/head url request-opts))]
      (and (or
            (some-> response :headers :www-authenticate (str/includes? "Bearer"))
            (some-> response :headers :www-authenticate (str/includes? "Basic")))
           (or
            (= 401 (:status response))
               ;; gcr responds with 200s when call from within google cloud for some reason
            (and (= 200 (:status response))
                 (str/ends-with? domain "gcr.io"))
            (log/warnf "is-private-repo? %s -> false. Response: %s" (pr-str response))))))))

(defn- resolve-platform-manifest
  "Grab a manifest and associate it with its platform"
  [domain access-token repository platform-manifest]
  (go-safe
   (try
     (let [manifest (<? (get-manifest domain access-token repository (:digest platform-manifest)))]
       (assoc manifest :platform (:platform platform-manifest)))
     (catch :default _))))

(defn- get-manifests-for-list
  "Resolve all manifests in a given list"
  [domain access-token repository manifest-list]
  (go-safe
   (log/infof "Resolving all manifests for list %s/%s" domain repository)
   (<? (->>
        manifest-list
        (filter #(= (:mediaType %) "application/vnd.docker.distribution.manifest.v2+json"))
        (map (partial resolve-platform-manifest domain access-token repository))
        (async/merge)
        (async/reduce (fn [acc i]
                        (if (instance? js/Error i)
                          (do
                            (log/errorf i "Failed to resolve a manifest from manifest-list")
                            acc)
                          (conj acc i))) [])))))

(defn get-manifests
  "Get all manifests for a tag or digest. Will contain platform for fat-manifests"
  [domain access-token repository tag-or-digest]
  (go-safe
   (if-let [manifest (<? (get-manifest domain access-token repository tag-or-digest))]
     (if-let [manifest-list (:manifests manifest)]
       (map
        #(with-meta % {:manifest-list manifest})
        (<? (get-manifests-for-list domain access-token repository manifest-list)))
       ;; TODO - can we resolve platform via some API?
       [manifest])
     (log/warnf "Could not find manifests for %s/%s:%s" domain repository tag-or-digest))))

(defn get-labelled-manifests
  "Add labels, ports, env etc for each image to its manifest"
  [domain access-token repository tag-or-digest]
  (go-safe
   (log/infof "Fetching labelled manifests for %s/%s:%s" domain repository tag-or-digest)
   (if-let [manifests (<? (get-manifests domain access-token repository tag-or-digest))]
     (<? (map-reduce (fn [manifest]
                       (go-safe
                        (let [digest (-> manifest :config :digest)]
                          (if-let [metadata (<? (get-image-details domain access-token repository digest))]
                            (let [labels (-> metadata :config :Labels not-empty)
                                  env (-> metadata :config :Env not-empty)
                                  ports (-> metadata :config :ExposedPorts not-empty)]
                              (log/debugf "Found image metadata... %s" metadata)
                              (cond-> manifest
                                labels
                                (assoc :labels labels)

                                env
                                (assoc :env env)

                                ports
                                (assoc :ports ports)))
                            (do
                              (log/warnf "Could not find metadata for %s/%s:%s" domain repository digest)
                              manifest)))))
                     manifests))
     (log/warnf "Could not find manifests for %s/%s:%s" domain repository tag-or-digest))))

(defn ->platform
  [platform parent-image]
  (when platform
    [(merge {:schema/entity-type :docker/platform
             :docker.platform/image parent-image
             :docker.platform/architecture (:architecture platform)
             :docker.platform/os (:os platform)}
            (when-let [variant (:variant platform)]
              {:docker.platform/variant variant}))]))

(defn ->image-layers-entities
  "Generate entities for an image and manifest/label details retrieved from an api"
  [domain repository manifest & [tag]]
  (let [labels (:labels manifest)
        tags (if tag [tag] [])
        ports (:ports manifest)
        env (:env manifest)
        env (->> env
                 (map #(str/split % #"="))
                  ;; make sure we only make tuples here mcr is doing something weird
                 (filter #(= 2 (count %))))
        labels-tx (->>
                   (seq labels)
                   (map (fn [[k v]]
                          (let [ref (str "$label-" (name k))]
                            [ref
                             {:schema/entity-type :docker.image/label
                              :schema/entity ref
                              :docker.image.label/name (name k)
                              :docker.image.label/value (str v)}]))))
        sha (or (:org.opencontainers.image.revision labels) (:org.label-schema.vcs-ref labels))]
    (log/infof "label-map:  %s" labels-tx)
    (log/infof "label-map vcs-ref:  %s" sha)
    (log/infof "env-map: %s" env)
    (let [layers-and-blobs (concat
                            (map-indexed
                             (fn [index {:keys [size digest]}]
                               {:schema/entity-type :docker.image/layer
                                :schema/entity (str "$layer-" digest "-" index)
                                :docker.image.layer/ordinal index
                                :docker.image.layer/blob (str "$blob-" digest)})
                             (:layers manifest))
                            (->> (:layers manifest)
                                 (map (fn [{:keys [size digest]}] [size digest]))
                                 (into #{})
                                 (map (fn [[size digest]]
                                        {:schema/entity-type :docker.image/blob
                                         :schema/entity (str "$blob-" digest)
                                         :docker.image.blob/size size
                                         :docker.image.blob/digest digest}))))]

      (concat
       [(merge
         {:schema/entity-type :docker/image
          :schema/entity "$docker-image"
          :docker.image/digest (:digest manifest)
        ;; TODO - these are mutable, so we should be able to add/remove them?
          :docker.image/tags tags
          :docker.image/labels (->> labels-tx
                                    (map first)
                                    (into []))
          :docker.image/repository "$repository"
          :docker.image/layers {:set (->> layers-and-blobs
                                          (filter #(= :docker.image/layer (:schema/entity-type %)))
                                          (map :schema/entity))}}
         (when (not-empty ports)
           {:docker.image/ports (map
                                 (fn [[k _]]
                                   [(name k) (namespace k)])
                                 ports)})
         (when (not-empty env)
           {:docker.image/environment-variables {:set (map #(str "$" (first %)) env)}
            ;; deprecated because datomic tuple strings are limited to 256 chars
            :docker.image/env (vec (map
                                    (fn [pair] (vec (map #(subs % 0 256) pair)))
                                    env))})
         (when sha
           {:docker.image/sha sha}))
        {:schema/entity-type :docker/repository
         :schema/entity "$repository"
         :docker.repository/host domain
         :docker.repository/repository repository}]
       layers-and-blobs
       (when labels-tx (->> (seq labels-tx)
                            (map second)
                            (into [])))
       (when (not-empty env)
         (map
          (fn [[k v]] {:schema/entity-type :docker.image.environment/variable
                       :schema/entity (str "$" k)
                       :docker.image.environment.variable/name k
                       :docker.image.environment.variable/value v})
          env))

       (->platform (:platform manifest) "$docker-image")
       (when-let [manifest-list (-> manifest meta :manifest-list)]
         [{:schema/entity-type :docker/manifest-list
           :schema/entity "$manifest-list"
           :docker.manifest-list/digest (:digest manifest-list)
           :docker.manifest-list/repository "$repository"
           :docker.manifest-list/tags tags
           :docker.manifest-list/images {:add ["$docker-image"]}}])))))

(defn ->image-layers-entities-tagged
  "Generate entities for an image and manifest/label details retrieved from an api
   As above, but adds docker/tag"
  [domain repository manifest & [tag]]
  (let [labels (:labels manifest)
        tags (if tag [tag] [])
        ports (:ports manifest)
        env (:env manifest)
        env (->> env
                 (map #(str/split % #"="))
                  ;; make sure we only make tuples here mcr is doing something weird
                 (filter #(= 2 (count %))))
        labels-tx (->>
                   (seq labels)
                   (map (fn [[k v]]
                          (let [ref (str "$label-" (name k))]
                            [ref
                             {:schema/entity-type :docker.image/label
                              :schema/entity ref
                              :docker.image.label/name (name k)
                              :docker.image.label/value (str v)}]))))
        sha (or (:org.opencontainers.image.revision labels) (:org.label-schema.vcs-ref labels))]
    (log/infof "label-map:  %s" labels-tx)
    (log/infof "label-map vcs-ref:  %s" sha)
    (log/infof "env-map: %s" env)
    (let [layers-and-blobs (concat
                            (map-indexed
                             (fn [index {:keys [size digest]}]
                               {:schema/entity-type :docker.image/layer
                                :schema/entity (str "$layer-" digest "-" index)
                                :docker.image.layer/ordinal index
                                :docker.image.layer/blob (str "$blob-" digest)})
                             (:layers manifest))
                            (->> (:layers manifest)
                                 (map (fn [{:keys [size digest]}] [size digest]))
                                 (into #{})
                                 (map (fn [[size digest]]
                                        {:schema/entity-type :docker.image/blob
                                         :schema/entity (str "$blob-" digest)
                                         :docker.image.blob/size size
                                         :docker.image.blob/digest digest}))))]

      (concat
       [(merge
         {:schema/entity-type :docker/image
          :schema/entity "$docker-image"
          :docker.image/digest (:digest manifest)
        ;; TODO - these are mutable, so we should be able to add/remove them?
          :docker.image/tags tags
          :docker.image/labels (->> labels-tx
                                    (map first)
                                    (into []))
          :docker.image/repository "$repository"
          :docker.image/layers {:set (->> layers-and-blobs
                                          (filter #(= :docker.image/layer (:schema/entity-type %)))
                                          (map :schema/entity))}}
         (when (not-empty ports)
           {:docker.image/ports (map
                                 (fn [[k _]]
                                   [(name k) (namespace k)])
                                 ports)})
         (when (not-empty env)
           {:docker.image/environment-variables {:set (map #(str "$" (first %)) env)}
            ;; deprecated because datomic tuple strings are limited to 256 chars
            :docker.image/env (vec (map
                                    (fn [pair] (vec (map #(subs % 0 256) pair)))
                                    env))})
         (when sha
           {:docker.image/sha sha}))
        {:schema/entity-type :docker/repository
         :schema/entity "$repository"
         :docker.repository/host domain
         :docker.repository/repository repository}]
       layers-and-blobs
       (when labels-tx (->> (seq labels-tx)
                            (map second)
                            (into [])))
       (when (not-empty env)
         (map
          (fn [[k v]] {:schema/entity-type :docker.image.environment/variable
                       :schema/entity (str "$" k)
                       :docker.image.environment.variable/name k
                       :docker.image.environment.variable/value v})
          env))

       (map
        (fn [tag]
          (merge
           {:schema/entity-type :docker/tag
            :docker.tag/name tag
            :docker.tag/repository "$repository"
            :docker.tag/digest (or (-> manifest meta :manifest-list :digest)
                                   (:digest manifest))}
           (if (-> manifest meta :manifest-list)
             {:docker.tag/manifest-list "$manifest-list"}
             {:docker.tag/image "$docker-image"})))
        tags)
       (->platform (:platform manifest) "$docker-image")
       (when-let [manifest-list (-> manifest meta :manifest-list)]
         [{:schema/entity-type :docker/manifest-list
           :schema/entity "$manifest-list"
           :docker.manifest-list/digest (:digest manifest-list)
           :docker.manifest-list/repository "$repository"
           :docker.manifest-list/tags tags
           :docker.manifest-list/images {:add ["$docker-image"]}}])))))
(defn ->nice-image-name
  [docker-image]
  (str (-> docker-image :docker.image/repository :docker.repository/repository) "@" (:docker.image/digest docker-image)))

(defn layers-match?
  "Do layers in manifest match that of parent image?"
  [parent-image manifest]
  (let [base-layer-digests (->> manifest :layers (map :digest))
        current-image-digests (->> parent-image
                                   :docker.image/layers
                                   (sort-by :docker.image.layer/ordinal)
                                   (take (count base-layer-digests))
                                   (map :docker.image.layer/blob)
                                   (map :docker.image.blob/digest))]
    (if (= base-layer-digests current-image-digests)
      true
      (do
        (log/warnf "%s - FROM image layers don't match: %s"
                   (->nice-image-name parent-image)
                   (data/diff current-image-digests base-layer-digests))
        false))))

(defn matching-image
  "Return matching manifest (if any)"
  [parent-image manifests]
  (some->> manifests
           (filter (partial layers-match? parent-image))
           first))

(defn parse-image-name
  "Returns keys #{:tag :digest :domain :repository}"
  [image-url]
  (when (string? image-url)
    (when-let [parsed (js->clj (parse-docker-image-name image-url) :keywordize-keys true)]
      (cond-> (set/rename-keys parsed {:path :repository})
        (and
         (not (:tag parsed))
         (not (:digest parsed)))
        (assoc :tag "latest")

        (or
         (not (:domain parsed))
         (= "docker.io" (:domain parsed)))
        (assoc :domain "hub.docker.com")))))
