(ns atomist.container
  (:require [atomist.cljs-log :as log]
            [atomist.async :refer-macros [go-safe <?]]
            [atomist.promise :as promise]
            [atomist.api :as api]
            [cljs-node-io.core :as io]
            [atomist.json :as json]
            [cljs.core.async :refer [<! >! chan]]
            [atomist.meta :as meta]
            [goog.object :as g]
            [atomist.topics])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn read-atomist-payload
  "merge atomist payload into request"
  [handler]
  (letfn [(payload->owner [{:keys [data]}]
            (or (-> data :Push first :repo :owner)
                (-> data :Tag first :commit :repo :owner)))
          (payload->repo [{:keys [data]}]
            (or (-> data :Push first :repo :name)
                (-> data :Tag first :commit :repo :name)))]
    (fn [request]
      (go-safe
       (let [payload (-> (io/file (:payload request))
                         (io/slurp)
                         (json/->obj)
                         (api/event->request))]
         (<? (handler
              (cond-> request
                (contains? (:data payload) :Push) (assoc :owner (payload->owner payload)
                                                         :repo (payload->repo payload))
                :always (merge payload
                               {:api-key (->> payload
                                              :secrets
                                              (filter #(= "atomist://api-key" (:uri %)))
                                              first
                                              :value)})))))))))

(defn create-logger
  "wrap atomist logger"
  [handler]
  (fn [request]
    (go-safe
     (try
       (when (and
              (.. js/process -env -TOPIC)
              (not (= "true" (.. js/process -env -LOCAL_SKILL_RUNNER))))
         (log/create-logger (:correlation-id request) "unknown" (-> request :team-id) (-> request :skill :id)))
       (log/infof "----> starting %s %s %s (%s)" meta/module-name meta/version meta/generated-at meta/tag)
       (<! (handler request))
       (<! (log/close-logger))
       (catch :default ex
         (log/error ex)
         (<! (log/close-logger))
         request)))))

(defn check-environment
  "pull ENV vars out of Process and add
     :team-id
     :team {:id \"\"}
     :graphql-endpoint
     :payload
     :bucket-name
     :topic
     :correlation-id
     :sendreponse function

     fail if full ENV not found"
  [handler]
  (letfn [(env [& envs]
            (reduce (fn [agg x]
                      (or agg (g/get (. js/process -env) x))) false envs))]
    (fn [request]
      (go-safe
       (let [request
             (cond-> request
               (env "ATOMIST_WORKSPACE_ID" "WORKSPACE_ID") (assoc :team-id (env "ATOMIST_WORKSPACE_ID" "WORKSPACE_ID"))
               (env "ATOMIST_GRAPHQL_ENDPOINT" "GRAPHQL_ENDPOINT") (assoc :graphql-endpoint (env "ATOMIST_GRAPHQL_ENDPOINT" "GRAPHQL_ENDPOINT"))
               (env "ATOMIST_PAYLOAD" "PAYLOAD") (assoc :payload (env "ATOMIST_PAYLOAD"))
               (env "ATOMIST_STORAGE" "STORAGE") (assoc :bucket-name (nth (re-find #"gs://(.*)" (env "ATOMIST_STORAGE" "STORAGE")) 1))
               (env "ATOMIST_TOPIC" "TOPIC") (assoc :topic (env "ATOMIST_TOPIC" "TOPIC"))
               (env "ATOMIST_CORRELATION_ID") (assoc :correlation-id (env "ATOMIST_CORRELATION_ID")))]
         (if (and
              (:team-id request)
              (:topic request)
              (:graphql-endpoint request)
              (:payload request)
              (:bucket-name request)
              (:correlation-id request))
           (<? (handler (assoc request
                               :sendreponse (if (not (= "true" (.. js/process -env -LOCAL_SKILL_RUNNER)))
                                              (partial atomist.topics/sendreponse
                                                       (:topic request)
                                                       (:correlation-id request))
                                              (fn [obj]
                                                (js/Promise.
                                                 (fn [accept _]
                                                   (log/infof "-> %-10s%-20s%s"
                                                              (:topic request)
                                                              (:correlation-id request)
                                                              (js->clj obj :keywordize-keys true))
                                                   (accept true)))))
                               :team {:id (:team-id request)})))
             (assoc request 
                    :atomist/status 
                    {:code 1 
                     :reason "environment is missing some of WORKSPACE_ID GRAPHQL_ENDPOINT ATOMIST_PAYLOAD TOPIC or ATOMIST_STORAGE"})))))))

(defn- exit [exit-code]
  (if (not (= "true" (.. js/process -env -IN_REPL)))
    (.exit js/process 0)))

(defn ^:api make-container-request
  "wrap handler in a go block and return a Promise that waits on this block"
  [handler]
  (fn [request]
    (let [done-channel (chan)]
      ;; handler chain exists when this go block completes
      (go
        (try
          (<! (handler request))
          (>! done-channel "done")
          (catch :default ex
            (<! ((api/status #(go %)) (assoc request :atomist/status {:code 1 :reason (str ex)})))
            (>! done-channel :failed))))
      (let [prom (promise/chan->promise done-channel)]
        (.info js/console prom)
        (.catch
         (.then
          prom
          (fn [result]
            (.info js/console result)
            (exit 0)))
         (fn [error]
           (.error js/console error)
           (exit 1)))))))

(def mw-make-container-request
  (api/compose-middleware
   [create-logger]
   [read-atomist-payload]
   [check-environment]
   [make-container-request]))
