(ns atomist.middleware.repo-iterator
  (:require [atomist.promise :as promise]
            [cljs.core.async :refer [<!] :refer-macros [go]]
            [atomist.api :as api]
            [atomist.container :as container]
            [atomist.json :as json]
            [atomist.cljs-log :as log]
            [clojure.string :as s]
            [atomist.github :as github]))

(defn- ->clj [obj]
  (js->clj obj :keywordize-keys true))

(defn compile-patch-repo [request]
  (fn [repo-config]
    (promise/chan->promise
     (go
       (let [response (<! (github/patch-repo request (json/->obj repo-config)))]
         (log/infof "%s/%s -> %s"
                    (-> request :ref :owner)
                    (-> request :ref :repo)
                    (:status response))
         (:status response))))))

(defn maybe-creds [handler opts]
  (fn [request]
    (go
      (if (not (:token request))
        (<! ((api/extract-github-token handler) request))
        (<! (handler request))))))

(defn compile-branch-protection-rule [request]
  (fn [branch obj]
    (promise/chan->promise
     (go
       (let [rule (js->clj obj :keywordize-keys true)]
         (<! ((-> (fn [req] (go
                              (log/info "put branch protection rule status " (-> req :response :status))
                              (select-keys (:response req) [:status :body])))
                  ((fn [handler]
                     (fn [req]
                       (go
                         (let [response (<! (github/branch-protection-rule
                                             req
                                             (-> request :ref :owner)
                                             (-> request :ref :repo)
                                             branch
                                             rule))]
                           (<! (handler (assoc req :response response))))))))
                  (maybe-creds {}))
              request)))))))

(defn compile-branch-requires-signed-commits [request]
  (fn [branch b]
    (promise/chan->promise
     (go
       (<! ((-> (fn [req] (go
                            (log/info "put branch protection rule status " (-> req :response :status))
                            (select-keys (:response req) [:status :body])))
                ((fn [handler]
                   (fn [req]
                     (go
                       (let [response (<! (github/branch-protection-rule
                                           req
                                           (-> request :ref :owner)
                                           (-> request :ref :repo)
                                           branch
                                           b))]
                         (<! (handler (assoc req :response response))))))))
                (maybe-creds {}))
            request))))))

(defn with-branches
  [handler js-fun]
  (fn [request]
    (go
      (<! (handler (assoc request :return-value
                          (<! (promise/from-promise
                               (js-fun (-> request :branches (clj->js)))
                               ->clj
                               ->clj))))))))

(defn compile-with-branches [request]
  (fn [js-fun]
    (promise/chan->promise
     (go
       (<! ((-> (fn [req] (go (:return-value req)))
                (with-branches js-fun)
                ((fn [handler]
                   (fn [req]
                     (go (let [branches (<! (github/branches
                                             req
                                             (-> request :ref :owner)
                                             (-> request :ref :repo)))]
                           (<! (handler (assoc req :branches branches))))))))
                (maybe-creds {}))
            request))))))

(defn create-repo [request]
  (go
    (let [repo (<! (github/repo request))]
      (-> request
          (select-keys [:token])
          (merge (select-keys repo [:private :topics :archived :disabled :default_branch :open_issue_count :fork]))
          (assoc :patchRepo (compile-patch-repo request))
          (assoc :branchProtectionRule (compile-branch-protection-rule request))
          (assoc :withBranches (compile-with-branches request))
          (assoc :branchRequiresSignedCommits (compile-branch-requires-signed-commits request))
          (merge (select-keys (:ref request) [:owner :repo]))
          (merge (if (:project request) (assoc (:project request) :basedir (-> request :project :path))))
          (clj->js)))))

(defn with-github-repo
  "  params
       handler - async js request handler
       fun - async ({ref config}, async patch-repo() => status) => Any
           - (whatever the function returns will be stored as status on the request)"
  [handler js-fun]
  (fn [request]
    (go
      (<! (handler (assoc request
                          :return-value
                          (<! (promise/from-promise
                               (js-fun (<! (create-repo request)))
                               ->clj
                               ->clj))))))))

(defn maybe-clone [handler opts]
  (fn [request]
    (go
      (if (:clone opts)
        (<! ((api/clone-ref handler) request))
        (<! (handler request))))))

(defn maybe-edit [handler opts]
  (fn [request]
    (go
      (cond
        (contains? opts :with_pr)
        (<! ((api/edit-inside-PR
              handler
              (:with_pr opts)) request))
        (contains? opts :with_commit)
        (<! ((api/edit-inside-PR
              handler
              (assoc
               (:with_commit opts)
               :type :commit-then-push)) request))
        :else
        (<! (handler request))))))

(defn compile-repo-iterator
  [request]
  (fn [js-fun & opts]
    (promise/chan->promise
     (go
       (<! ((api/repo-iterator
             (fn [{:keys [plan]}]
               ;; just log the second parameter
               (go
                 (if (second opts)
                   (let [result ((second opts) (clj->js plan))]
                     (or result true))
                   (do
                     (log/info plan)
                     plan))))
             (-> (fn [request] (go (:return-value request)))
                 (with-github-repo js-fun)
                 (maybe-edit (if (first opts) (js->clj (first opts) :keywordize-keys true) {}))
                 (maybe-clone (if (first opts) (js->clj (first opts) :keywordize-keys true) {}))))
            request))))))

(defn compile-with-repo [request]
  (fn [js-fun & opts]
    (promise/chan->promise
     (go
       (<! ((-> (fn [request] (go request))
                (with-github-repo js-fun)
                (maybe-edit (if (first opts) (js->clj (first opts) :keywordize-keys true) {}))
                (maybe-clone (if (first opts) (js->clj (first opts) :keywordize-keys true) {}))
                (maybe-creds (if (first opts) (js->clj (first opts) :keywordize-keys true) {})))
            request))))))


