;; Copyright © 2021 Atomist, Inc.
;;
;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at
;;
;;     http://www.apache.org/licenses/LICENSE-2.0
;;
;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns atomist.docker
  (:require [atomist.async :refer-macros [<? go-safe] :refer [map-reduce]]
            [atomist.cljs-log :as log]
            [atomist.exports :as exports]
            [atomist.json :as json]
            [atomist.sha :as sha]
            [clojure.data :as data]
            [clojure.string :as str]
            [cljs.core.async :as async]
            ["parse-docker-image-name" :as parse-docker-image-name]
            [goog.string :as gstring]
            [http.client :as client]
            [clojure.set :as set]
            [atomist.time :as atm-time]
            [goog.crypt.base64 :as base64]))

(def timeout 5000)

(defn paging-next? [response]
  (println (:headers response))
  (if-let [link-header (-> response :headers :link)]
    (re-find #"; rel=\"next\"" link-header)))

(defn paging-last [response]
  (second (re-find #"<(.*)>; rel=\"next\"" (-> response :headers :link))))

(defn- bearer-or-basic
  "most of the docker v2 apis use 'Bearer' in the Authorization header but ECR uses 'Basic'"
  [domain]
  (cond
    (str/includes? domain "amazonaws.com") "Basic"
    ;; horrible hack for artifactory
    (str/includes? domain "http") "Basic"
    :else "Bearer"))

(defn get-with-paging
  [url v2-token page-size reduce-fn]
  (go-safe
   (loop [results nil url (gstring/format "%s?n=%s" url page-size)]
     (log/debugf "Getting %s" url)
     (let [response (<? (client/get url
                                    {:timeout timeout
                                     :throw-exceptions false
                                     :headers
                                     {"Authorization" (gstring/format "%s %s" (bearer-or-basic url) v2-token)}}))]
       (if (= 200 (:status response))
         (let [results (reduce-fn results (:body response))]
           (if-let [next (-> response :body :next)]
             (recur results next)
             results))
         (log/warnf "Error getting %s from gcr %s" url response))))))

(defn- default-https
  "Turns out we might want http :/"
  [domain-or-base-url]
  (if (str/starts-with? domain-or-base-url "http")
    domain-or-base-url
    (str "https://" domain-or-base-url)))

(defn fetch-catalog
  "verify that the v2 catalog is visible (during check-auth)"
  [domain v2-token]
  (go-safe
   (let [catalog (<? (get-with-paging (gstring/format "%s/v2/_catalog" (default-https domain))
                                      v2-token
                                      10
                                      (fn [repos body]
                                        ;; TODO aws response body is text/plain so would need parsing
                                        ;;      ecr integration currently using aws javascript SDK instead
                                        (concat (or repos [])
                                                (:repositories body)))))]
     (log/debugf "Downloaded catalog of %s repos" (count catalog))
     catalog)))

(defn- get-image-details-location
  "Weird re-direct thingy"
  [domain access-token repository image-digest]
  (go-safe
   (let [base-url (default-https domain)]
     (log/debugf "Fetching image-details location for %s/%s (image id %s)" base-url repository image-digest)
     (let [image-details (<? (client/get (gstring/format "%s/v2/%s/blobs/%s" base-url repository image-digest)
                                         {:timeout timeout
                                          :headers {"Authorization" (gstring/format "%s %s" (bearer-or-basic domain) access-token)}}))
           location (-> image-details :headers :location)]
       (cond
         (and location
              (#{307 302 200} (:status image-details)))
         location

         (and (= 200 (:status image-details))
              (not location))
         (json/->obj (str (:body image-details)))

         :else
         (log/errorf "Could not find image-details location for %s/%s (image id %s)" base-url repository image-digest))))))

(defn- get-image-details
  "Fetch labels, ports etc given an image-id"
  [domain access-token repository image-id]
  (go-safe
   (let [base-url (default-https domain)]
     (log/debugf "Fetching image details for %s/%s@%s" base-url repository image-id)
     (let [location-or-result (<? (get-image-details-location domain access-token repository image-id))]

       (cond
         (not location-or-result)
         (log/warnf "Could not find image details for %s/%s (image id %s)" base-url repository image-id)

         (string? location-or-result)
         (let [with-host (if (str/includes? base-url "docker.pkg.dev")
                           (gstring/format "%s%s" base-url location-or-result)
                           location-or-result)]
           (log/debugf "location:  %s" with-host)
           (let [metadata (<? (client/get with-host {:timeout timeout}))]
             (log/debugf "Location get response: %s" (pr-str metadata))
             (if (= 200 (:status metadata))
               (let [details (-> metadata :body (str) (json/->obj))]
                 (if (or (str/ends-with? base-url "gcr.io")
                     ;; some build tools use this date for repeatable builds
                         (and
                          (-> metadata :headers :last-modified)
                          (= "1970-01-01T00:00:00Z" (:created details))))
                   (assoc details :created (-> metadata :headers :last-modified))
                   details))
               (log/warnf "Error fetching details for %s/%s (image id %s) -> %s" base-url repository image-id (:status metadata)))))

         :else
         location-or-result)))))

(comment
  (go-safe (println (<? (client/get "https://production.cloudflare.docker.com/registry-v2/docker/registry/v2/blobs/sha256/9e/9e101926df47b9eab1fc03699461a2d263517b3fabb532b802a4fd117e4e363d/data?verify=1661772405-lBIlLpODliLOkiHertnq87%2Fj3pA%3D" {:timeout timeout})))))

(defn- response->token
  "Parse 401s and try to get an anonymous token"
  [response]
  (go-safe
   (let [auth (:www-authenticate (:headers response))]
     (if
      (and
       auth
       (= 401 (:status response)))
       ;;;;"Bearer realm=\"https://docker-auth.elastic.co/auth\",service=\"token-service\",scope=\"repository:elasticsearch/elasticsearch:pull\""
       (if-let [[_ realm service scope] (re-matches
                                         #"^.*realm=\"(.*?)\",service=\"(.*?)\",scope=\"(.*)\"$"
                                         auth)]

         (let [response (<? (client/get
                             (gstring/format "%s?service=%s&scope=%s" realm service scope)
                             {:timeout 5000}))]
           (if (= 200 (:status response))
             (-> response :body :token)
             (log/warnf "Unable to get anonymous token for %s: %s" auth response)))
         (log/infof "Could not parse challenge %s" auth))
       (log/infof "Was not a 401 with challenge: %s" response)))))

(defn- anonymous-access-token
  [domain repository tag-or-digest]
  (go-safe
   (try
     (let [base-url (default-https domain)]
       (log/debugf "Trying to get anonymous token %s %s:%s" base-url repository tag-or-digest)
       (let [url (gstring/format "%s/v2/%s/manifests/%s" base-url repository tag-or-digest)
             request-opts {:timeout timeout
                           :headers (merge
                                     {"Accept" "application/vnd.docker.distribution.manifest.v2+json,application/vnd.docker.distribution.manifest.list.v2+json,application/vnd.oci.image.manifest.v1+json,application/vnd.oci.image.index.v1+json"})}

             response (<? (client/get url request-opts))
             token (<? (response->token response))]
         (if token
           (do
             (log/infof "Got an anonymous token for %s" domain)
             token)
           (log/warnf "Failed to get a token for %s" domain))))
     (catch :default e
       (log/warnf e "Error fetching anonymous token")))))


(def index-content-types #{"application/vnd.docker.distribution.manifest.list.v2+json" "application/vnd.oci.image.index.v1+json"})
(def manifest-content-types #{"application/vnd.docker.distribution.manifest.v2+json", "application/vnd.oci.image.manifest.v1+json"})

(def accept-header (str/join "," (set/union index-content-types manifest-content-types)))


(defn- get-manifest
  "Fetch a manifest by digest or tag. Can return a manifest or a manifest-list"
  [domain access-token repository tag-or-digest]
  (go-safe
   (let [base-url (default-https domain)]
     (log/debugf "Fetching manifest %s %s:%s" base-url repository tag-or-digest)
     (let [url (gstring/format "%s/v2/%s/manifests/%s" base-url repository tag-or-digest)
           request-opts {:timeout timeout
                         :headers {"Authorization" (gstring/format "%s %s" (bearer-or-basic domain) access-token)
                                   "Accept" accept-header}}
         ;; for ECR, must also do a HEAD request - GET response does not have the docker-content-digest
           ecr-head-response (when (or
                                    (str/includes? base-url "amazonaws.com")
                                    (str/includes? base-url "public.ecr.aws"))
                               (<? (client/head url request-opts)))
           response (<? (client/get url request-opts))]
       (if (= 200 (:status response))
         (do
           (log/debugf "Raw manifest response: %s" (pr-str response))
           (if-let [errors (-> response :body :errors)]
             (log/warnf "Errors getting manifest: %s" errors)
             (let [manifest (-> response :body (str) (json/->obj))]
                 ;; sometimes we get a list of manifests if the image is multi-platform
               (if (->> response :headers :content-type (index-content-types))
                 (log/debugf "Found a manifest list for %s/%s:%s" base-url repository tag-or-digest)
                 (log/debugf "Fetched image manifest %s/%s:%s" base-url repository tag-or-digest))
               (if (->> response :headers :content-type ((set/union index-content-types manifest-content-types)))
                 (assoc manifest :digest (or
                                          (-> response :headers :docker-content-digest)
                                          (-> ecr-head-response :headers :docker-content-digest)))
                 (log/warnf "Unrecognized content type: %s" (->> response :headers :content-type))))))
         (log/errorf "error fetching manifest for %s/%s:%s status %s" base-url repository tag-or-digest (:status response)))))))


(defn ghcr-anonymous-auth
  [repository]
  (go-safe
   (log/debugf "Attempting anonymous auth for %s/%s" "ghcr.io" repository)
   (let [response (<? (client/get (gstring/format "https://ghcr.io/token?service=ghcr.io&scope=repository:%s:pull" repository)
                                  {:timeout 5000}))]
     (if (= 200 (:status response))
       {:access-token (-> response :body :token)
        :repository repository}
       (throw (ex-info (gstring/format "unable to auth %s" repository) response))))))

(defn- with-ghcr-token-if-required
  [domain repository request]
  (go-safe
   (if (= "ghcr.io" domain)
     (if-let [token (:access-token (<? (ghcr-anonymous-auth repository)))]
       (assoc-in request [:headers "Authorization"] (str "Bearer " token))
       request)
     request)))
(defn private-repo?
  "Is repository a private repo?"
  [domain repository tag-or-digest]
  (go-safe
   (boolean
    (let [url (gstring/format "https://%s/v2/%s/manifests/%s" domain repository tag-or-digest)
          _ (log/infof "Checking if %s is private using HEAD request to: %s" repository url)
          request-opts {:timeout timeout :headers {"Accept" "application/vnd.docker.distribution.manifest.v2+json,application/vnd.docker.distribution.manifest.list.v2+json,application/vnd.oci.image.manifest.v1+json,application/vnd.oci.image.index.v1+json"}
                        :throw-exceptions false}
          response (<? (client/head url (<? (with-ghcr-token-if-required domain repository request-opts))))]
      (if (or
           (some-> response :headers :www-authenticate (str/includes? "insufficient_scope"))
           (and (or
                 (some-> response :headers :www-authenticate (str/includes? "Bearer"))
                 (some-> response :headers :www-authenticate (str/includes? "Basic")))
                (or
                 (= 401 (:status response))
               ;; gcr responds with 200s when call from within google cloud for some reason
                 (and (= 200 (:status response))
                      (str/ends-with? domain "gcr.io"))
                 (log/warnf "is-private-repo? %s -> false. Response: %s" (pr-str response)))))
        (do
          (log/debugf "Repository %s is private: %s" repository response)
          true)
        (do
          (log/debugf "Repository %s probably public: %s" repository response)
          false))))))

(defn is-public?
  "Is repository a publically accessible?"
  [domain repository tag-or-digest]
  (go-safe
   (boolean
    (let [url (gstring/format "https://%s/v2/%s/manifests/%s?" domain repository tag-or-digest)
          _ (log/infof "Checking if %s is public using HEAD request to: %s" repository url)
          request-opts {:timeout timeout :headers {"Accept" "application/vnd.docker.distribution.manifest.v2+json,application/vnd.docker.distribution.manifest.list.v2+json,application/vnd.oci.image.manifest.v1+json,application/vnd.oci.image.index.v1+json"}
                        :throw-exceptions false}
          response (<? (client/head url (<? (with-ghcr-token-if-required domain repository request-opts))))]
      ;; note:  public.ecr.aws repos will return a :www-authenticate header of
      ;;        Bearer realm=https://public.ecr.aws/token/,service=public.ecr.aws,scope=aws
      ;;   so is-public? should return true for public.ecr.aws images
      (if (and
           (= 200 (:status response))
           (not (some-> response :headers :www-authenticate (str/includes? "insufficient_scope"))))
        (do
          (log/debugf "Repository %s is public: %s" repository response)
          true)
        (do
          (log/debugf "Repository %s is probably private: %s " repository response)
          false))))))

(defn- resolve-platform-manifest
  "Grab a manifest and associate it with its platform"
  [domain access-token repository platform-manifest]
  (go-safe
   (try
     (when-let [manifest (<? (get-manifest domain access-token repository (:digest platform-manifest)))]
       (assoc manifest :platform (:platform platform-manifest)))
     (catch :default _))))

(defn- get-manifests-for-list
  "Resolve all manifests in a given list"
  [domain access-token repository manifest-list]
  (go-safe
   (log/debugf "Resolving all manifests for list %s/%s" domain repository)
   (<? (->>
        manifest-list
        (filter #(#{"application/vnd.oci.image.index.v1+json" "application/vnd.docker.distribution.manifest.v2+json"} (:mediaType %)))
        (map (partial resolve-platform-manifest domain access-token repository))
        (async/merge)
        (async/reduce (fn [acc i]
                        (if (or (not i) (instance? js/Error i))
                          (do
                            (log/errorf i "Failed to resolve a manifest from manifest-list")
                            acc)
                          (conj acc i))) [])))))

(defn get-manifests
  "Get all manifests for a tag or digest"
  [domain access-token repository tag-or-digest]
  (go-safe
   (if-let [manifest (<? (get-manifest domain access-token repository tag-or-digest))]
     (if-let [manifest-list (:manifests manifest)]
       (map
        #(with-meta % {:manifest-list manifest})
        (<? (get-manifests-for-list domain access-token repository manifest-list)))
       [manifest])
     (log/warnf "Could not find manifests for %s/%s:%s" domain repository tag-or-digest))))

(defn merge-manifest-and-details
  "Merge image metadata/details with the v2 manifest"
  [manifest metadata]
  ;; TODO - jim - do you have an example of a weird date? This is returning nil for perfectly
  ;; good dates, and js/Date. seems to do the right thing for every example I've seen in the wild.
  (let [created (or (-> metadata :created atm-time/parse-weird-docker-datetime-string atm-time/->js-date) (js/Date. (:created metadata)))
        labels (-> metadata :config :Labels not-empty)
        env (-> metadata :config :Env not-empty)
        architecture (:architecture metadata)
        roof-fs (-> metadata :rootfs)
        os (:os metadata)
        ports (-> metadata :config :ExposedPorts not-empty)
        histories (remove
                   :empty_layer
                   (:history metadata))]

    (cond-> manifest
      labels
      (assoc :labels labels)

      env
      (assoc :env env)

      (and
       (empty? (:config metadata))
       (= "" architecture)
       (= "" os)
       (not (:docker_version metadata))
       (not (:container metadata))
       (some
        #(= "kaniko" (:author %))
        histories))
      (assoc :kaniko-cache true)

      ports
      (assoc :ports ports)

      created
      (assoc :created created)

      (not-empty os)
      (assoc :os os)

      (not-empty architecture)
      (assoc :architecture architecture)

      :always
      (update :layers #(map-indexed (fn [i layer]
                                      (let [history (nth histories i)]
                                        (assoc layer
                                               :diff-id (nth (:diff_ids roof-fs) i nil)
                                               :created-by (:created_by history)
                                               :created-at (:created history)))) %)))))
(defn get-labelled-manifests
  "Add labels, ports, env etc for each image to its manifest"
  [domain access-token repository tag-or-digest]
  (go-safe
   (log/debugf "Fetching labelled manifests for %s/%s:%s" domain repository tag-or-digest)
   (let [access-token (if access-token
                        access-token
                        (<? (anonymous-access-token domain repository tag-or-digest)))]
     (if-let [manifests (not-empty (remove
                                  ;; remove cosign images
                                    (fn [manifest]
                                      (some
                                       (fn [layer]
                                         (= "application/vnd.dev.cosign.simplesigning.v1+json"
                                            (:mediaType layer)))
                                       (:layers manifest)))
                                    (<? (get-manifests domain access-token repository tag-or-digest))))]
       (<? (map-reduce (fn [manifest]
                         (go-safe
                          (let [digest (-> manifest :config :digest)]
                            (if-let [metadata (<? (get-image-details domain access-token repository digest))]
                              (do
                                (log/debugf "Found image metadata... %s" metadata)
                                (merge-manifest-and-details manifest metadata))
                              (do
                                (log/warnf "Could not find metadata for %s/%s:%s" domain repository digest)
                                manifest)))))
                       manifests))
       (log/warnf "Could not find manifests for %s/%s:%s" domain repository tag-or-digest)))))

(defn chain-id
  "As per: https://github.com/opencontainers/image-spec/blob/54a822e528b91c8db63b873ad56daf200a2e5e61/config.md#layer-chainid"
  [previous next]
  (if previous
    (str "sha256:" (sha/sha-256 (str previous " " next)))
    next))



(defn ->platform
  [{:keys [os architecture platform] :as image} parent-image]
  (if platform
    [(merge {:schema/entity-type :docker/platform
             :docker.platform/image parent-image
             :docker.platform/architecture (:architecture platform)
             :docker.platform/os (:os platform)}
            (when-let [variant (:variant platform)]
              {:docker.platform/variant variant}))]
    (when (and os architecture)
      [{:schema/entity-type :docker/platform
        :docker.platform/image parent-image
        :docker.platform/architecture architecture
        :docker.platform/os os}])))

(defn ->image-layers-entities-tagged
  "Generate entities for an image and manifest/label details retrieved from an api
   As above, but adds docker/tag

   params
     manifest ()"
  [domain repository manifest & [tag-or-tags updated-at]]
  (let [labels (:labels manifest)
        annotations (merge
                     {}
                     ;; oci manifest annotations
                     (-> manifest meta :manifest-list :annotations)
                     (:annotations manifest))
        tags (if (sequential? tag-or-tags)
               tag-or-tags
               (if tag-or-tags
                 [tag-or-tags] []))
        ports (:ports manifest)
        env (:env manifest)
        env (->> env
                 (map #(str/split % #"="))
                  ;; make sure we only make tuples here mcr is doing something weird
                 (filter #(= 2 (count %))))
        manifest-list (-> manifest meta :manifest-list)
        labels-tx (->>
                   (seq labels)
                   (map (fn [[k v]]
                          (let [ref (str "$label-" (name k))]
                            [ref
                             {:schema/entity-type :docker.image/label
                              :schema/entity ref
                              :docker.image.label/name (name k)
                              :docker.image.label/value (str v)}]))))
        annotations-tx (->>
                        (seq annotations)
                        (map (fn [[k v]]
                               (let [ref (str "$annotation-" (name k))]
                                 [ref
                                  {:schema/entity-type :oci/annotation
                                   :schema/entity ref
                                   :oci.annotation/name (name k)
                                   :oci.annotation/value (str v)}]))))
        sha (or (:org.opencontainers.image.revision annotations)
                (:org.opencontainers.image.revision labels)
                (:org.label-schema.vcs-ref labels))
        updated-at (or updated-at (:created manifest-list) (:created manifest) (js/Date.))
        layers-with-cumulative-digest (reduce
                                       (fn [acc {:keys [digest diff-id] :as layer}]
                                         (let [previous-layer (last acc)]
                                           (conj acc (cond-> layer

                                                       digest
                                                       (assoc :blob-digest
                                                              (chain-id (:blob-digest previous-layer) digest))
                                                       diff-id
                                                       (assoc :chain-id
                                                              (chain-id (:chain-id previous-layer) diff-id))))))
                                       []
                                       (:layers manifest))
        layers-and-blobs (concat
                          (map-indexed
                           (fn [index {:keys [size digest blob-digest created-by created-at chain-id]}]
                             (merge {:schema/entity-type :docker.image/layer
                                     :schema/entity (str "$image-" (:digest manifest) "-layer-" digest "-" index)
                                     :docker.image.layer/blob-digest blob-digest
                                     :docker.image.layer/ordinal index
                                     ;; used in identity
                                     :docker.image.layer/image-digest (:digest manifest)
                                     :docker.image.layer/blob (str "$blob-" digest)}
                                    (when chain-id
                                      {:docker.image.layer/chain-id chain-id})
                                    (when created-by
                                      {:docker.image.layer/created-by created-by})
                                    (when created-at
                                      {:docker.image.layer/created-at (js/Date. created-at)})))
                           layers-with-cumulative-digest)
                          (->> (:layers manifest)
                               (group-by :digest)
                                 ;; there are dupes in here sometimes, but we only need one target ref here
                               (map (comp first second))
                               (map (fn [{:keys [digest size diff-id]}]
                                      (merge
                                       {:schema/entity-type :docker.image/blob
                                        :schema/entity (str "$blob-" digest)
                                        :docker.image.blob/size size
                                        :docker.image.blob/digest digest}
                                       (when diff-id
                                         {:docker.image.blob/diff-id diff-id}))))))]
    ;; (log/debugf "label-map:  %s" labels-tx)
    ;; (log/debugf "")
    ;; (log/debugf "label-map vcs-ref:  %s" sha)
    ;; (log/debugf "env-map: %s" env)
    (concat
     [(merge
       {:schema/entity-type :docker/image
        :schema/entity "$docker-image"
        :docker.image/digest (:digest manifest)
        ;; TODO - these are mutable, so we should be able to add/remove them?
        :docker.image/tags {:add tags}
        :docker.image/labels (->> labels-tx
                                  (map first)
                                  (into []))
        :docker.image/repository "$repository"
        :docker.image/repositories {:add ["$repository"]}}
       (when (not-empty layers-and-blobs)
         {:docker.image/layers {:set (->> layers-and-blobs
                                          (filter #(= :docker.image/layer (:schema/entity-type %)))
                                          (map :schema/entity))}})
       (when (not-empty annotations-tx)
         {:oci/annotations {:set (->> annotations-tx
                                      (map first)
                                      (into []))}})
       (let [last-layer (->>
                         layers-and-blobs
                         (sort-by :docker.image.layer/ordinal)
                         last)]
         (merge {}
                (when-let [blob-digest (:docker.image.layer/blob-digest last-layer)]
                  {:docker.image/blob-digest blob-digest})
                (when-let [chain-id (:docker.image.layer/chain-id last-layer)]
                  {:docker.image/diff-chain-id chain-id})))

       (when-let [created (:created manifest)]
         {:docker.image/created-at created})
       (when (not-empty ports)
         {:docker.image/ports (map
                               (fn [[k _]]
                                 [(name k) (namespace k)])
                               ports)})
       (when (not-empty env)
         {:docker.image/environment-variables {:set (map #(str "$" (first %)) env)}
            ;; deprecated because datomic tuple strings are limited to 256 chars
          :docker.image/env (vec (map
                                  (fn [pair] (vec (map #(subs % 0 256) pair)))
                                  env))})
       (when sha
         {:docker.image/sha sha}))
      (merge
       {:schema/entity-type :docker/repository
        :schema/entity "$repository"
        :docker.repository/host domain
        :docker.repository/repository repository}
       (when (and (:os manifest) (:architecture manifest))
         {:docker.repository/platforms {:add [(str (:os manifest) "/" (:architecture manifest))]}}))]
     layers-and-blobs
     (when (not-empty labels-tx)
       (->> (seq labels-tx)
            (map second)
            (into [])))
     (when (not-empty annotations-tx)
       (->> (seq annotations-tx)
            (map second)
            (into [])))
     (when (not-empty env)
       (map
        (fn [[k v]] {:schema/entity-type :docker.image.environment/variable
                     :schema/entity (str "$" k)
                     :docker.image.environment.variable/name k
                     :docker.image.environment.variable/value v})
        env))

     (map
      (fn [tag]
        (merge
         {:schema/entity-type :docker/tag
          :docker.tag/name tag
          :atomist/tx-iff {:where '[(or-join [?entity ?new-updated-at]
                                             [(missing? $ ?entity :docker.tag/updated-at)]
                                             (and
                                              [?entity :docker.tag/updated-at ?updated-at]
                                              [(< ?updated-at ?new-updated-at)]))]
                           :args {:new-updated-at updated-at}}
          :docker.tag/repository "$repository"
          :docker.tag/updated-at updated-at
          :docker.tag/digest (or (-> manifest meta :manifest-list :digest)
                                 (:digest manifest))}
         (if manifest-list
           {:docker.tag/manifest-list "$manifest-list"}
           {:docker.tag/image "$docker-image"})))
      tags)
     (->platform manifest "$docker-image")
     (when manifest-list
       [(merge {:schema/entity-type :docker/manifest-list
                :schema/entity "$manifest-list"

                :docker.manifest-list/digest (:digest manifest-list)
                :docker.manifest-list/repository "$repository"
                :docker.manifest-list/repositories {:add ["$repository"]}
                :docker.manifest-list/tags tags
                :docker.manifest-list/images {:add ["$docker-image"]}}
               (when-let [created (:created manifest-list)]
                 {:docker.manifest-list/created-at created}))]))))

(comment
  (go-safe
   ;;http://hal9000:8082/artifactory/api/docker/atomist-docker-local/kipz-test:latest
   #_(println (<? (get-labelled-manifests "http://hal9000:8082/artifactory/api/docker/atomist-docker-local"
                                          (base64/encodeString (str "admin" ":" "Welcome123"))
                                          "kipz-test"  "latest")))
   (println (pr-str (->image-layers-entities-tagged
                     "http://hal9000:8082"
                     "kipz-test"
                     (first (<? (get-labelled-manifests "http://hal9000:8082/artifactory/api/docker/atomist-docker-local"
                                                        (base64/encodeString (str "admin" ":" "xx"))
                                                        "kipz-test"  "latest")))
                     "latest")))
   #_(try
       (let [access-token (clojure.string/trim (io/slurp "./access-token"))
             manifests (<? (get-labelled-manifests "gcr.io" access-token "personalsdm-216019/kipz-docker-test"  ""))])
       (catch :default e
         (log/errorf e "Error fetching")))))

(defn ->nice-image-name
  [docker-image]
  (str (-> docker-image :docker.image/repository :docker.repository/repository) "@" (:docker.image/digest docker-image)))

(defn layers-match?
  "Do layers in manifest match that of parent image?"
  [parent-image manifest]
  (let [base-layer-digests (->> manifest :layers (map :digest))
        current-image-digests (->> parent-image
                                   :docker.image/layers
                                   (sort-by :docker.image.layer/ordinal)
                                   (take (count base-layer-digests))
                                   (map :docker.image.layer/blob)
                                   (map :docker.image.blob/digest))]
    (if (= base-layer-digests current-image-digests)
      true
      (do
        (log/warnf "%s - FROM image layers don't match: %s"
                   (->nice-image-name parent-image)
                   (data/diff current-image-digests base-layer-digests))
        false))))

(defn matching-image
  "Return matching manifest (if any)"
  [parent-image manifests]
  (some->> manifests
           (filter (partial layers-match? parent-image))
           first))


(defn parse-image-name
  "Returns keys #{:tag :digest :domain :repository}"
  [image-url]
  (when (string? image-url)
    (when-let [parsed (js->clj (parse-docker-image-name image-url) :keywordize-keys true)]
      (cond-> (set/rename-keys parsed {:path :repository})
        (and
         (not (:tag parsed))
         (not (:digest parsed)))
        (assoc :tag "latest")

        (or
         (not (:domain parsed))
         (= "docker.io" (:domain parsed)))
        (assoc :domain "hub.docker.com")))))

(comment
  (go-safe (println (<? (get-labelled-manifests "gcr.io" nil "distroless/python2.7" "sha256:baa2c3932cba84e88396e688b15908d9be8a4c18911b9f94818332205fd0ea09")))))
