(ns atomist.local-runner
  "convenience functions when REPLing
     - relies on environment variables like API_KEY_PROD, API_KEY_STAGING, API_KEY_PROD_GITHUB_AUTH, GITHUB_TOKEN"
  (:require [atomist.cljs-log :as log]
            [clojure.edn :as edn]
            [cljs.pprint :refer [pprint]]
            [cljs-node-io.core :as io]
            [atomist.json :as json]
            [atomist.api :as api]
            [cljs.core.async :refer-macros [go] :refer [<!]]))

(def environment (atom nil))
(defn set-env
  [x]
  {:pre [(#{:custom
            :prod :staging :prod-github-auth} x)]}
  (swap! environment (constantly x)))
(defn ^:export setEnv [s]
  (set-env (keyword s)))
(defn token []
  (if (not @environment)
    (throw (ex-info "select environment before running" {}))
    (case @environment
      :prod (.. js/process -env -API_KEY_PROD)
      :staging (.. js/process -env -API_KEY_STAGING)
      :prod-github-auth (.. js/process -env -API_KEY_PROD_GITHUB_AUTH)
      :custom (.. js/process -env -ATOMIST_API_KEY))))

(defn call-event-handler [event-data h]
  (let [{fake-handler :fake-handler :or {fake-handler (fn [& args] (log/info args))}} event-data]
    (.catch
     (.then
      (h (clj->js event-data :keyword-fn (fn [k]
                                           (if-let [n (namespace k)]
                                             (str n "/" (name k))
                                             (name k))))
         fake-handler)
      (fn [v] (if-not (= v :done)
                (log/infof "----> val %s" v))))
     (fn [error] (log/error "ERROR " error)))))
(defn ^:export callEventHandler [event-data h]
  (call-event-handler event-data h))

(defn fake-schedule [team-id]
  {:data {:OnSchedule {}}
   :secrets [{:uri "atomist://api-key" :value (token)}]
   :extensions {:team_id team-id
                :correlation_id "corrid"}})

(defn fake-push [team-id org {:keys [name id defaultBranch] :or {defaultBranch "master"}} branch]
  {:data {:Push [{:branch branch
                  :repo {:name name
                         :owner org
                         :id id
                         :org {:owner org}
                         :defaultBranch defaultBranch}
                  :after {:message ""}}]}
   :secrets [{:uri "atomist://api-key" :value (token)}]
   :extensions {:team_id team-id
                :operationName "OnAnyPush"
                :correlation_id "corrid"}})

(defn ^:export fakePushEvent [team-id org repo branch]
  (fake-push team-id org (js->clj repo :keywordize-keys true) branch))

(defn fake-pull-request [team-id org {:keys [name id]} head base]
  {:data {:PullRequest [{:branchName head
                         :baseBranchName base
                         :head {:sha ""}
                         :repo {:name name
                                :owner org
                                :id id
                                :org {:owner org}
                                :defaultBranch "master"}}]}
   :secrets [{:uri "atomist://api-key" :value (token)}]
   :extensions {:team_id team-id
                :operationName "OnPullRequest"
                :correlation_id "corrid"}})

(defn ^:export fakePullRequest [team-id org repo head base]
  (fake-pull-request team-id org (js->clj repo :keywordize-keys true) head base))

(defn fake-comment-on-issue [team-id org repo number comment]
  {:data {:Comment [{:body comment
                     :by {:name "Jim Clark"
                          :login "slimslenderslacks"
                          :person {:forename nil, :surname nil}}
                     :commentId "632960693"
                     :issue {:number number
                             :repo {:id "thing" :name repo, :owner org
                                    :org {:owner org}}}
                     :pullRequest nil}]}
   :secrets [{:uri "atomist://api-key" :value (token)}]
   :extensions {:team_id team-id
                :correlation_id "corrid"}})

(defn add-configuration [event-data configuration]
  (assoc-in event-data [:skill :configuration] configuration))
(defn ^:export addConfiguration [event-data configuration]
  (add-configuration event-data (js->clj configuration)))

(defn add-configuration-instance [event-data configuration]
  (update-in event-data [:skill :configuration :instances] (fnil conj []) configuration))

(defn fake-command-handler [team-id command raw-message channel-id user-id]
  {:command command
   :source {:slack {:channel {:id channel-id}
                    :user {:id user-id}
                    :team {:id "fake" :name "fake"}}}
   :api_version "1"
   :correlation_id "corrid"
   :team {:id team-id}
   :raw_message raw-message
   :secrets [{:uri "atomist://api-key" :value (token)}]})
(defn ^:export fakeCommandHandler [team-id command raw-message channel-id user-id]
  (fake-command-handler team-id command raw-message channel-id user-id))
(defn ^:export printEvent [event-data]
  (cljs.pprint/pprint event-data))

(defn set-atomist-home [s]
  (set! (.. js/process -env -ATOMIST_HOME) s))

(defn container-contract [event-data h]
  (set! (.. js/process -env -ATOMIST_WORKSPACE_ID) "T29E48P34")
  (set! (.. js/process -env -ATOMIST_GRAPHQL_ENDPOINT) "https://automation.atomist.com/graphql")
  (set! (.. js/process -env -ATOMIST_PAYLOAD) "./payload.json")
  (set! (.. js/process -env -ATOMIST_CORRELATION_ID) "corrid")
  (set! (.. js/process -env -ATOMIST_STORAGE) "gs://storage")
  (set! (.. js/process -env -ATOMIST_TOPIC) "topic")
  (go
    (let [{fake-handler :fake-handler :or {fake-handler (fn [& args] (log/info args))}} event-data]
      (io/spit "./payload.json" (json/->str event-data))
      (let [{:keys [project] :as request} (<! ((-> (fn [request] (go (assoc request
                                                                            :atomist.api/do-not-delete-cloned-repo true)))
                                                   (api/clone-ref)
                                                   (api/extract-github-token)
                                                   (api/create-ref-from-event)) event-data))]
        (println "set ATOMIST_HOME to " (:path project))
        (println "delete policy " (:atomist.api/do-not-delete-cloned-repo request))
        (atomist.local-runner/set-atomist-home (:path project)))

      (.catch
       (.then
        (h (clj->js event-data) fake-handler)
        (fn [v] (if-not (= v :done)
                  (log/infof "----> val %s" v))))
       (fn [error] (log/error "ERROR " error))))))

(defn set-container-env [team-id f]
  (set! (.. js/process -env -ATOMIST_WORKSPACE_ID) team-id)
  (set! (.. js/process -env -ATOMIST_GRAPHQL_ENDPOINT)  "https://automation.atomist.com/graphql")
  (set! (.. js/process -env -ATOMIST_PAYLOAD) f)
  (set! (.. js/process -env -ATOMIST_STORAGE)  "gs://dev/null")
  (set! (.. js/process -env -ATOMIST_TOPIC) "mock-topic")
  (set! (.. js/process -env -ATOMIST_CORRELATION_ID) "mock-corrid")
  (set! (.. js/process -env -LOCAL_SKILL_RUNNER) "true")
  (set! (.. js/process -env -HADOLINT) "hadolint"))

(defn set-function-env []
  (set! (.. js/process -env -GRAPHQL_ENDPOINT)  "https://automation.atomist.com/graphql")
  (set! (.. js/process -env -LOCAL_SKILL_RUNNER) "true"))

(defn run-container-handler
  [team-id owner h result-builder]
  (go
    (set-container-env team-id "payload.json")
    (set-env :prod)
    (io/spit
     (io/file "payload.json")
     (json/->str
      {:skill {:id "mock-skill"}
       :correlation_id "mock-corrid"
       :subscription (<! (result-builder {:team-id team-id :owner owner}))
       :team {:id team-id}
       :secrets [{:uri "atomist://api-key" :value (token)}]}
      :keyword-fn (fn [k] (if (and (keyword? k) (namespace k))
                            (str (namespace k) "/" (name k))
                            (name k)))))
    (.catch
     (.then
      (h)
      (fn [result] (println "end result:  " result)))
     (fn [error] (println "end error:  " error)))))

(defn run-function-handler
  [team-id owner h result-builder]
  (go
    (set-function-env)
    (set-env :prod)
    (println "kick it off")
    (.catch
     (.then
      (h (clj->js
          {:skill {:id "mock-skill"}
           :eventId "mock-event-id"
           :correlation_id "mock-corrid"
           :subscription (<! (result-builder {:team-id team-id :owner owner}))
           :team {:id team-id}
           :secrets [{:uri "atomist://api-key" :value (token)}]}
          :keyword-fn (fn [k] (if (and (keyword? k) (namespace k))
                                (str (namespace k) "/" (name k))
                                (name k))))
         (fn sendreponse [corrid obj]
           (js/Promise. (fn [accept reject]
                          (println "---> corrid " corrid)
                          (pprint (clj->js obj :keywordize-keys true))
                          (println "<---- corrid " corrid)
                          (accept #js {:mock true})))))
      (fn [result] (println "end result:  " result)))
     (fn [error] (println "end error:  " error)))))

(defn get-installation-token
  "requires environment for getting a valid api-key - queries team graphql api for installation token by org name"
  [team-id org-name]
  (go (:token (<! ((-> #(go %)
                       (api/extract-github-token))
                   {:team {:id team-id}
                    :ref {:owner org-name}
                    :secrets [{:uri "atomist://api-key" :value (token)}]})))))

(defn mw-add-token-to-subscription [handler]
  (fn [request]
    (go (<! (handler
             (assoc-in
              request [:subscription :result 0 0 :git.commit/repo :git.repo/org :github.org/installation-token]
              (:token request)))))))

(def add-token-to-subscription
  (api/compose-middleware
   [mw-add-token-to-subscription]
   [api/extract-github-token]))

(defn js-obj->entities-pprint [& args]
  (go (-> args
          first
          (js->clj :keywordize-keys true)
          :entities
          (edn/read-string)
          pprint)))
