(ns atomist.cli
  (:require [atomist.local-runner :as lr]
            [atomist.middleware :as mw]
            [clojure.tools.cli :as tc]
            [atomist.cljs-log :as log]
            [cljs.nodejs :as node]
            [goog.string :as gstring]
            [goog.string.format]
            [cljs-node-io.core :as io :refer [slurp spit]]
            [cljs.reader]
            [clojure.string :as s]
            [atomist.github :as github]
            [atomist.api :as api]
            [cljs.pprint]
            [cljs.core.async :refer [<!] :refer-macros [go]]))

(enable-console-print!)

(defn merge-atomist-prs [request]
  (go
    (doseq [{{:keys [number title id] {:keys [nameWithOwner]} :repository} :node}
            (-> request :atomist/prs :data :search :edges)]
      (println "merge " number " id " id)
      (println
       (<! (github/github-v4
            (:token request)
            (io/slurp "graphql/mutation/merge-pull-request.graphql")
            {:id id}))))))

(comment
  (lr/set-env :prod-github-auth)
  (lr/token)
  (go
    (<! ((-> blast
             ((fn [handler]
                (fn [request]
                  (go
                    (let [response (<! (github/github-v4
                                        (:token request)
                                        (io/slurp "graphql/query/search_open_prs.graphql")
                                        {}))]
                      (<! (handler (assoc request :atomist/prs response))))))))
             (api/extract-github-token))
         {:team {:id "T29E48P34"}
          :ref {:owner "atomist-skills"}
          :secrets [{:uri "atomist://api-key" :value (lr/token)}]}))))

(defn edit
  [{:keys [owner repo]} path]
  (if-let [matches (let [f (io/file (io/file path) "README.md")]
                     (and (.exists f) (re-seq #"`#support`" (io/slurp f))))]
    (let [f (io/file (io/file path) "README.md")]
      (io/spit f (s/replace (io/slurp f) #"`#support`" "`#help`"))
      {:matches matches :slug (gstring/format "%s/%s" owner repo)})
    {}))

(defn pr-campaign [edit {:keys [token team-id]}]
  (go
    (<! ((-> #(go
                (cljs.pprint/pprint (->> (:plan %) (filter :matches)))
                (log/infof "processed %d" (->> (:plan %) (filter :matches) (count)))
                (log/infof "unprocessed %d" (->> (:plan %) (filter (complement :matches)) (count))))
             (api/repo-iterator
              (fn [{:keys [ref]} _] (go (#{"atomist" "atomist-skills"} (:owner ref))))
              (-> (fn [{{:keys [path]} :project ref :ref}]
                    (go
                      (edit ref path)))
                  (api/edit-inside-PR (fn [request]
                                        {:branch "supportToHelp"
                                         :target-branch (-> request :ref :branch)
                                         :title "#support to #help"
                                         :body "#support to #help"
                                         :labels ["campaign-1"]}))
                  (api/clone-ref))))
         {:team {:id team-id}
          :secrets [{:uri "atomist://api-key" :value token}]}))))

(defn repo-promise [f]
  (fn [repo]
    (js/Promise.
     (fn [accept reject]
       (accept (f repo))))))

(defn ^:export visitRepo
  "Set an environment and reate a fake command handler
    returns Promise that will fulfill when the work is done"
  [{:keys [env team mapper reducer]}]
  (lr/set-env (keyword env))
  (-> (lr/fake-command-handler team "sync" "raw-message" "channel-id" "user-id")
      (lr/add-configuration {:name "default"
                             :parameters []})
      (assoc :fake-handler (constantly true))
      (lr/call-event-handler
       (mw/handler #js {:sync (fn [request]
                                (.withRepoIterator request
                                                   (repo-promise mapper)
                                                   #js {:clone true}
                                                   reducer))}))))

(defn ^:export visitRepoNoClone
  "Set an environment and reate a fake command handler
    returns Promise that will fulfill when the work is done"
  [{:keys [env team mapper reducer]}]
  (lr/set-env (keyword env))
  (-> (lr/fake-command-handler team "sync" "raw-message" "channel-id" "user-id")
      (lr/add-configuration {:name "default"
                             :parameters []})
      (assoc :fake-handler (constantly true))
      (lr/call-event-handler
       (mw/handler #js {:sync (fn [request]
                                (.withRepoIterator request
                                                   (repo-promise mapper)
                                                   #js {:clone false}
                                                   reducer))}))))

(def cli-options [["-e" "--env ENV" "Environment Name"
                   :default "custom"
                   :validate [#{"custom" "prod" "staging" "prod-github-auth"}
                              "only these environments are supported"]]
                  ["-t" "--team TEAM" "workspace id"]
                  ["-m" "--module MODULE" "javascript mapper module file"]
                  ["-h" "--help" nil]])

(defn load-module [s]
  (let [js (node/require s)]
    [(. js -mapper) (. js -reducer)]))

(defn ^:export main []
  (let [{:keys [options summary errors]}
        (tc/parse-opts (. js/process -argv) cli-options)]
    (cond
      (contains? options :help)
      (do
        (log/info summary)
        (.exit js/process 0))
      (and errors (not (empty? errors)))
      (do
        (log/error errors)
        (log/info summary)
        (.exit js/process 1))
      :else
      (.then
       (visitRepo (merge options
                         (let [[m r] (load-module (:module options))]
                           {:mapper m
                            :reducer r})))
       (fn [_] (.exit js/process 0))))))

(defn- has-common-clj [f]
  (let [m (cljs.reader/read-string (slurp f))]
    (->> (drop 3 m)
         (partition 2)
         (map #(into [] %))
         (into {})
         :dependencies
         (some #(if (= 'atomist/common-clj (first %)) %))
         (second))))

(comment
  (visitRepo
   {:team "T095SFFBK"
    :env "prod"
    :mapper (fn [repo]
              (let [f (io/file (. repo -basedir) "project.clj")]
                (merge
                 {:repo (gstring/format "%s/%s" (. repo -owner) (. repo -repo))}
                 {:exists (and (.exists f) (has-common-clj f))})))
    :reducer (fn [coll] (cljs.pprint/pprint (->> (js->clj coll :keywordize-keys true)
                                                 (filter :exists)
                                                 (into []))))})
  (visitRepoNoClone
   {:team "T095SFFBK"
    :env "prod"
    :mapper (fn [repo]
              (if (s/starts-with? (. repo -repo) "iac")
                (do
                  (go
                    (println (<! (github/put-topics {:token (. repo -token)
                                                     :ref {:owner (. repo -owner)
                                                           :repo (. repo -repo)}}
                                                    (conj (into #{} (js->clj (. repo -topics))) "infrastructure-as-code")))))
                  {:repo (gstring/format "%s/%s" (. repo -owner) (. repo -repo))
                   :private (. repo -private)
                   :topics (. repo -topics)})
                {}))
    :reducer (fn [coll] (cljs.pprint/pprint (->> (js->clj coll :keywordize-keys true)
                                                 (into []))))})
  (visitRepoNoClone
   {:team "AEIB5886C"
    :env "prod-github-auth"
    :mapper (fn [repo]
              {:repo (gstring/format "%s/%s" (. repo -owner) (. repo -repo))
               :private (. repo -private)
               :default_branch (. repo -default_branch)
               :topics (. repo -topics)})
    :reducer (fn [coll] (cljs.pprint/pprint (->> (js->clj coll :keywordize-keys true)
                                                 (filter #(not (contains? x :errors)))
                                                 (into []))))}))
