(ns atomist.github
  (:require [http.client :as client]
            [goog.string :as gstring]
            [goog.string.format]
            [clojure.string :as s]
            [atomist.cljs-log :as log]
            [atomist.json :as json]
            [goog.crypt.base64 :as b64]
            [cljs.spec.alpha :as spec]
            [cljs.core.async :refer [<!] :as async :refer-macros [go]]))

(defn github-v4 [token query variables]
  (go
    (let [response (<! (http.client/post
                        "https://api.github.com/graphql"
                        {:headers {"Authorization" (gstring/format "bearer %s" token)
                                   "Accept" "application/vnd.github.bane-preview+json"
                                   "User-Agent" "atomist"}
                         :body (json/->str {:query query
                                            :variables variables})}))]
      (if (= 200 (:status response))
        (:body response)
        (log/error "status was not 200")))))

(defn beautify [s]
  (try
    (let [pattern #"(\[[-\w]+:[a-z]+\])"
          ;; p (set! (.-flags pattern) "gm")
          tags (->> (re-seq pattern s)
                    (map second)
                    (sort))]
      (str
       (-> s
           (s/replace pattern "")
           (s/replace #"\n\s*\n\s" "\n\n"))
       "\n\n---\n<details><summary>Tags</summary><br/>\n"
       (->> tags
            (map #(str "<code>" % "</code>"))
            (interpose "\n")
            (apply str))
       "\n</details>"))
    (catch :default ex
      (log/warnf "failed to run pr beautification %s" ex)
      s)))

(defn get-first-pr-for-head
  "GET /repos/:owner/:repo/pulls?state=open&head="
  [{:keys [token owner repo]} head-ref-name]
  (go
    (-> (<! (client/get (gstring/format "https://api.github.com/repos/%s/%s/pulls" owner repo)
                        {:headers {"Authorization" (gstring/format "bearer %s" token)
                                   "User-Agent" "atomist"}
                         :query-params {:state "open"
                                        :head (gstring/format "%s:%s" owner head-ref-name)}}))

        :body
        first)))

(defn get-pr-by-number
  "GET /repos/:owner/:repo/pulls/:number"
  [{:keys [token owner repo]} number]
  (go
    (-> (<! (client/get (gstring/format "https://api.github.com/repos/%s/%s/pulls/%s" owner repo number)
                        {:headers {"Authorization" (gstring/format "bearer %s" token)
                                   "User-Agent" "atomist"}}))

        :body)))

(defn- patch-pr-title
  "PATCH /repos/:owner/:repo/pulls/:number"
  [{:keys [token owner repo]} number title]
  (client/patch (gstring/format "https://api.github.com/repos/%s/%s/pulls/%s" owner repo number)
                {:headers {"Authorization" (gstring/format "bearer %s" token)
                           "User-Agent" "atomist"}
                 :body (json/->str {:title title})}))

(defn- patch-pr-state
  "PATCH /repos/:owner/:repo/pulls/:number"
  [{:keys [token owner repo]} number state]
  (if (#{"open" "closed"} state)
    (client/patch (gstring/format "https://api.github.com/repos/%s/%s/pulls/%s" owner repo number)
                  {:headers {"Authorization" (gstring/format "bearer %s" token)
                             "User-Agent" "atomist"}
                   :body (json/->str {:state state})})
    (log/warn "status must be #{\"open\" \"closed\"}")))

(defn post-pr-comment
  "POST /repos/:owner/:repo/pulls/:number/comments"
  [{:keys [token owner repo]} number body]
  (client/post (gstring/format "https://api.github.com/repos/%s/%s/issues/%s/comments" owner repo number)
               {:headers {"Authorization" (gstring/format "bearer %s" token)
                          "User-Agent" "atomist"}
                :body (json/->str {:body body})}))

(def post-issue-comment post-pr-comment)

(defn post-pr
  "POST /repos/:owner/:repo/pulls
     body is (s/keys :req-un {title body base head draft})"
  [{:keys [token owner repo]} body labels]
  (go
    (let [response (<! (client/post (gstring/format "https://api.github.com/repos/%s/%s/pulls" owner repo)
                                    {:headers {"Authorization" (gstring/format "bearer %s" token)
                                               "User-Agent" "atomist"}
                                     :body (json/->str body)}))]
      (if (and (not (empty? labels)) (= 201 (:status response)))
        (<! (client/patch (gstring/format "https://api.github.com/repos/%s/%s/issues/%s" owner repo (-> response :body :number))
                          {:headers {"Authorization" (gstring/format "bearer %s" token)
                                     "User-Agent" "atomist"}
                           :body (json/->str {:labels labels})})))
      response)))

(defn log-github-http-status [expected s response]
  (if (not (= expected (:status response)))
    (log/warnf "%s: http status %s - %s" s (:status response) (:body response))))

(defn raise-pr
  "Create a PR or edit an existing one
    params
      p - should have keys :branch :owner :repo :token
    returns p with pull request number"
  [{:keys [branch] :as p} title body base-branch-ref labels]
  (go
    (if-let [pr (<! (get-first-pr-for-head p branch))]
      (do
        (log-github-http-status 200 "patch PR title" (<! (patch-pr-title p (:number pr) title)))
        (log-github-http-status 201 "post PR comment" (<! (post-pr-comment p (:number pr) (beautify body))))
        (assoc p :pull-request-number (:number pr)))
      (let [response (<! (post-pr p {:title title :body (beautify body) :head branch :base base-branch-ref} labels))]
        (if (= 201 (:status response))
          (assoc p :pull-request-number (-> response :body :number))
          (let [message (gstring/format "PR creation failed %s <- %s: %s" base-branch-ref branch (:status response))]
            (log/error message)
            (assoc p :error message)))))))

(defn close-pr
  [p head-branch-ref]
  (go
    (if-let [pr (<! (get-first-pr-for-head p head-branch-ref))]
      (let [response (<! (patch-pr-state p (:number pr) "closed"))]
        (log/info "closing " (:number pr))
        (if (not (= 200 (:status response)))
          (log/warn "unable to close pr %s" (:url pr)))
        response)
      (let [error (gstring/format "did not find pr for head ref %s" head-branch-ref)]
        (log/warn error)
        {:errors [error]}))))

(defn mark-pr-ready
  "  params
       id - global node id - not the same as the id in v3 (node_id for the v3 entitiy)"
  [{:keys [token]} id]
  (go
    (let [response
          (<!
           (github-v4
            token
            "mutation markPrReady( $id: ID!) { markPullRequestReadyForReview (input: { pullRequestId: $id, clientMutationId: \"atomist\"}) {clientMutationId pullRequest {isDraft}}}"
            {:id id}))]
      (log/info "response is " response)
      response)))

(defn pr-is-ready-by-number
  [p number]
  (go
    (if-let [pr (<! (get-pr-by-number p number))]
      (let [response (<! (mark-pr-ready p (:node_id pr)))]
        (log/info "id is " (:node_id pr))
        (log/info "changed state " (:data response) " for " (:number pr))
        {:status response})
      (let [message (gstring/format "did not find pr for number %s" number)]
        (log/warn message)
        {:errors [message]}))))

(defn pr-is-ready-by-branch
  [p head-branch-ref]
  (go
    (if-let [pr (<! (get-first-pr-for-head p head-branch-ref))]
      (let [response (<! (mark-pr-ready p (:node_id pr)))]
        (log/info "id is " (:node_id pr))
        (log/info "changed state " (:data response) " for " (:number pr))
        {:status response})
      (let [message (gstring/format "did not find pr for head ref %s" head-branch-ref)]
        (log/warn message)
        {:errors [message]}))))

(defn pr-channel [request branch-name]
  (get-first-pr-for-head (assoc (:ref request) :token (:token request)) branch-name))

(defn post-commit-comment [{:keys [owner repo token]} sha body]
  (client/post (gstring/format "https://api.github.com/repos/%s/%s/commits/%s/comments" owner repo sha)
               {:headers {"Authorization" (gstring/format "bearer %s" token)
                          "User-Agent" "atomist"}
                :body (json/->str {:body body})}))

;; TODO handle paging
(defn all-labels-channel [request]
  (go
    (try
      (let [response (<! (client/get
                          (gstring/format "https://api.github.com/repos/%s/%s/labels" (-> request :ref :owner) (-> request :ref :repo))
                          {:headers {"User-Agent" "atomist"
                                     "Authorization" (gstring/format "Bearer %s" (:token request))}}))]
        (if (= 200 (:status response))
          (-> response :body)
          (log/warn "no GitHub labels found")))
      (catch :default ex
        (log/error "raised exception " ex)))))

(defn get-label [request name]
  (go
    (let [response (<!
                    (client/get
                     (gstring/format "https://api.github.com/repos/%s/%s/labels/%s"
                                     (-> request :ref :owner)
                                     (-> request :ref :repo)
                                     name)
                     {:headers {"User-Agent" "atomist"
                                "Authorization" (gstring/format "Bearer %s" (:token request))}}))]
      (if (= 200 (:status response))
        (:body response)))))

(defn add-label [request {:keys [name description color]}]
  (go

    (let [response (<! (client/post
                        (gstring/format "https://api.github.com/repos/%s/%s/labels" (-> request :ref :owner) (-> request :ref :repo))
                        {:body (json/->str {:name name
                                            :description description
                                            :color color})
                         :headers {"User-Agent" "atomist"
                                   "Authorization" (gstring/format "Bearer %s" (:token request))}}))]
      (if (not (= 200 (:status response)))
        (log/warnf "status %s - %s" (:status response) (-> response :body)))
      response)))

(defn patch-label [request {:keys [name description color]}]
  (go
    (try
      (let [response (<! (client/patch
                          (gstring/format "https://api.github.com/repos/%s/%s/labels/%s"
                                          (-> request :ref :owner) (-> request :ref :repo)
                                          name)
                          {:body (json/->str {:description description
                                              :color color})
                           :headers {"User-Agent" "atomist"
                                     "Authorization" (gstring/format "Bearer %s" (:token request))}}))]
        (if (= 200 (:status response))
          (-> response :body)
          (log/warnf "status %s - %s" (:status response) (-> response :body))))
      (catch :default ex
        (log/error "raised exception " ex)))))

(defn delete-label [request name]
  (go
    (try
      (let [response (<! (client/delete
                          (gstring/format "https://api.github.com/repos/%s/%s/labels/%s"
                                          (-> request :ref :owner) (-> request :ref :repo)
                                          name)
                          {:headers {"User-Agent" "atomist"
                                     "Authorization" (gstring/format "Bearer %s" (:token request))}}))]
        (if (= 204 (:status response))
          (-> response :body)
          (log/warn "no GitHub labels found")))
      (catch :default ex
        (log/error "raised exception " ex)))))

(defn put-label
  "POST /repos/:owner/:repo/pulls
    labels is a vector of strings"
  [{:keys [labels number] :as request}]
  (client/post (gstring/format "https://api.github.com/repos/%s/%s/issues/%s/labels"
                               (-> request :ref :owner) (-> request :ref :repo) number)
               {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                          "User-Agent" "atomist"}
                :body (json/->str {:labels labels})}))

(defn get-labels
  ""
  [{:keys [number] :as request}]
  (go
    (->
     (<! (client/get (gstring/format "https://api.github.com/repos/%s/%s/issues/%s/labels"
                                     (-> request :ref :owner) (-> request :ref :repo) number)
                     {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                                "User-Agent" "atomist"}}))
     :body)))

(defn rm-label
  ""
  [{:keys [number] :as request} label]
  (client/delete (gstring/format "https://api.github.com/repos/%s/%s/issues/%s/labels/%s"
                                 (-> request :ref :owner) (-> request :ref :repo) number label)
                 {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                            "User-Agent" "atomist"}}))

(defn patch-repo
  [request config]
  (client/patch (gstring/format "https://api.github.com/repos/%s/%s"
                                (-> request :ref :owner) (-> request :ref :repo))
                {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                           "User-Agent" "atomist"}
                 :body (json/->str config)}))

(defn repo-topics
  [request]
  (go
    (let [response
          (<! (client/get (gstring/format "https://api.github.com/repos/%s/%s/topics"
                                          (-> request :ref :owner) (-> request :ref :repo))
                          {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                                     "User-Agent" "atomist"
                                     "Accept" "application/vnd.github.mercy-preview+json"}}))]
      (if (= 200 (:status response))
        (-> response :body :names)
        (do
          (log/warnf "topics for %s/%s %s %s" (-> request :ref :owner) (-> request :ref :repo) (-> response :status) (-> response :body))
          [])))))

(defn put-topics
  [request topic-names]
  (go
    (let [response
          (<! (client/put (gstring/format "https://api.github.com/repos/%s/%s/topics"
                                          (-> request :ref :owner) (-> request :ref :repo))
                          {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                                     "User-Agent" "atomist"
                                     "Accept" "application/vnd.github.mercy-preview+json"}
                           :body (json/->str {:names (into [] topic-names)})}))]
      (:status response))))

(defn repo
  [request]
  (go
    (let [response
          (<! (client/get (gstring/format "https://api.github.com/repos/%s/%s"
                                          (-> request :ref :owner) (-> request :ref :repo))
                          {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                                     "User-Agent" "atomist"
                                     "Accept" "application/vnd.github.mercy-preview+json"}}))]
      (when (= 200 (:status response))
        (-> response :body)))))

(defn content
  [request owner repo path]
  (go
    (let [response (<! (client/get (gstring/format "https://api.github.com/repos/%s/%s/contents/%s" owner repo path)
                                   (merge
                                    {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                                               "User-Agent" "atomist"}}
                                    (when-let [branch (-> request :ref :branch)]
                                      {:query-params {:ref branch}}))))]
      (if (and (= 200 (:status response))
               (= (-> response :body :type) "file"))
        (cond
          (= "base64" (-> response :body :encoding))
          {:content (-> response
                        :body
                        :content
                        (b64/decodeString))}
          :else
          {:error "file encoding only supports base64"})
        {:error "unsupported content"
         :status (:status response)
         :type (-> response :body :type)}))))

(comment
  (go (println (<! (content
                    {:token (.. js/process -env -GITHUB_TOKEN)}
                    "atomist-skills" "git-content-sync-skill" "src/atomist/main.cljs"))))
  (go (println (<! (create-issue
                    {:token (.. js/process -env -GITHUB_TOKEN)}
                    "atomist-skills" "git-content-sync-skill" {:title "test title" :body "body" :labels ["atomist:content-sync"]})))))

(defn create-issue
  "https://developer.github.com/v3/issues/#create-an-issue"
  [request owner repo body]
  (go
    (println owner repo)
    (let [response (<! (client/post (gstring/format "https://api.github.com/repos/%s/%s/issues" owner repo)
                                    {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                                               "User-Agent" "atomist"}
                                     :body (json/->str body)}))]
      response)))

(defn patch-issue
  [request owner repo number body]
  (go
    (let [response (<! (client/post (gstring/format "https://api.github.com/repos/%s/%s/issues/%s" owner repo number)
                                    {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                                               "User-Agent" "atomist"}
                                     :body (json/->str body)}))]
      response)))

(defn lock-issue
  [request owner repo number reason]
  {:pre [(#{"unlock" "off topic" "too heated" "resolved" "spam"} reason)]}
  (go
    (let [response (<! ((if (= reason "unlock")
                          client/delete
                          client/put)
                        (gstring/format "https://api.github.com/repos/%s/%s/issues/%s/lock" owner repo number)
                        (merge
                         {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                                    "User-Agent" "atomist"
                                    "Accept" "application/vnd.github.mercy-preview+json"}}
                         (if (not (#{:unlock} reason))
                           {:query-params {:lock_reason reason}}))))]
      response)))

(spec/def :checkrun.action/label string?)
(spec/def :checkrun.action/description string?)
(spec/def :checkrun.action/identifier string?)
(spec/def :checkrun-output/title string?)

(spec/def :checkrun-output/summary string?)
(spec/def :checkrun-output/text string?)
(spec/def :checkrun-output/annotation any?)
(spec/def :checkrun-output/annotations (spec/coll-of :checkrun-output/annotation))
(spec/def :checkrun/output (spec/keys :req-un [:checkrun-output/title :checkrun-output/summary]
                                      :opt-un [:checkrun-output/text :checkrun-output/annotations]))
(spec/def :checkrun/conclusion #{"success" "failure" "neutral" "cancelled" "skipped" "timed_out" "action_required"})
(spec/def :checkrun/status #{"queued" "in_progress" "completed"})
(spec/def :checkrun/head_sha string?)
(spec/def :checkrun/name string?)
(spec/def :checkrun/action (spec/keys :req-un [:checkrun.action/label :checkrun.action/description :checkrun.action/identifier]))
(spec/def :checkrun/actions (spec/coll-of :checkrun/action))
(spec/def :checkrun/check-run (spec/keys :req-un [:checkrun/name :checkrun/head_sha]
                                         :opt-un [:checkrun/status :checkrun/conclusion :checkrun/output :checkrun/actions]))
(defn create-check
  [request owner repo parameters]
  (go
    (log/info "parameters valid? " (spec/valid? :checkrun/check-run parameters))
    (let [response (<! (client/post (gstring/format "https://api.github.com/repos/%s/%s/check-runs" owner repo)
                                    {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                                               "User-Agent" "atomist"
                                               "Accept" "application/vnd.github.antiope-preview+json"}
                                     :body (json/->str parameters)}))]
      response)))

(defn update-check
  [request owner repo check-run-id parameters]
  (go
    (log/info "CheckRun parameters valid? " (spec/valid? :checkrun/check-run parameters))
    (log/info "parameters " parameters)
    (let [response (<! (client/patch (gstring/format "https://api.github.com/repos/%s/%s/check-runs/%s" owner repo check-run-id)
                                     {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                                                "User-Agent" "atomist"
                                                "Accept" "application/vnd.github.antiope-preview+json"}
                                      :body (json/->str parameters)}))]
      (when-not (= 200 (:status response))
        (log/warn (:body response)))
      response)))

(defn gist-content [request id]
  (go
    (let [response (<! (client/get (gstring/format "https://api.github.com/gists/%s" id)
                                   {:headers {"User-Agent" "atomist"}}))]
      (when (= 200 (:status response))
        (-> response
            :body
            :files
            seq
            first
            second
            :content)))))

(defn branch-protection-rule [request owner repo branch rule]
  (go
    (let [response (<! (client/put (gstring/format "https://api.github.com/repos/%s/%s/branches/%s/protection"
                                                   owner
                                                   repo
                                                   branch)
                                   {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                                              "User-Agent" "atomist"
                                              "Accept" "application/vnd.github.luke-cage-preview+json"}
                                    :body (json/->str rule)}))]
      response)))

(defn branch-requires-signed-commits [request owner repo branch b]
  (go
    (let [response (<! ((if b client/post client/delete) (gstring/format "https://api.github.com/repos/%s/%s/branches/%s/protection/required_signatures"
                                                                         owner
                                                                         repo
                                                                         branch)
                                                         {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                                                                    "User-Agent" "atomist"
                                                                    "Accept" "application/vnd.github.zzzax-preview+json"}}))]
      response)))

(defn wrap-github-paging [client]
  (fn [req]
    (-> #(assoc % :links {})
        (async/map [(client req)]))))
(defn paged-get [url & [req]]
  (let [paged-request (-> client/request
                          (wrap-github-paging))]
    (paged-request (merge req {:method :get :url url}))))

(defn branches [request owner repo]
  (let [request-params {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                                  "User-Agent" "atomist"}
                        :query-params {:per_page 100}}]
    (letfn [(concat-branches [response branches] (->> response
                                                      :body
                                                      (map :name)
                                                      (concat branches)))]
      (go
        (loop [response (<! (paged-get (gstring/format "https://api.github.com/repos/%s/%s/branches"
                                                       owner
                                                       repo)
                                       request-params))
               branches []]
          (cond

            (not (= 200 (:status response)))
            branches

            (and
             (= 200 (:status response))
             (-> response :links :next not))
            (concat-branches response branches)

            :else
            (recur
             (<! (paged-get (-> response :links :next) request-params))
             (concat-branches response branches))))))))
