(ns atomist.container
  (:require [atomist.cljs-log :as log]
            [atomist.async :refer-macros [go-safe <?]]
            [atomist.promise :as promise]
            [atomist.api :as api]
            [atomist.time :as time]
            [cljs-node-io.core :as io]
            [atomist.json :as json]
            [cljs.core.async :refer [<! >! chan]]
            [atomist.meta :as meta]
            [goog.object :as g]
            [clojure.string :as string]
            [goog.string :as gstring]
            [goog.string.format]
            ["express" :as express]
            ["body-parser" :as body-parser]
            ["@atomist/skill/lib/payload_resolve" :as payload_resolve])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(set! *warn-on-infer* false)

(def resolvePayload (. payload_resolve -resolvePayload))

(defn read-atomist-payload
  "merge atomist payload into request"
  [handler]
  (letfn [(payload->owner [{:keys [data]}]
            (or (-> data :Push first :repo :owner)
                (-> data :Tag first :commit :repo :owner)))
          (payload->repo [{:keys [data]}]
            (or (-> data :Push first :repo :name)
                (-> data :Tag first :commit :repo :name)))]
    (fn [request]
      (go-safe

       (let [payload (-> (io/file (:payload request))
                         (io/slurp)
                         (json/->obj)
                         (api/event->request))]
         (<? (handler
              (cond-> request
                (contains? (:data payload) :Push) (assoc :owner (payload->owner payload)
                                                         :repo (payload->repo payload))
                :always (merge payload
                               {:api-key (->> payload
                                              :secrets
                                              (filter #(= "atomist://api-key" (:uri %)))
                                              first
                                              :value)})))))))))

(defn create-logger
  "wrap atomist logger"
  [handler]
  (fn [request]
    (go-safe
     (try
       (when (not (= "true" (.. js/process -env -LOCAL_SKILL_RUNNER)))
         (log/create-logger
          (merge
           {:skillId (-> request :skill :id)
            :workspaceId (-> request :team-id)
            :correlationId (:correlation-id request)
            :eventId (or (:event-id request) "unknown")
            :name (or (:operation request) (:command request) "default")
            :skill (gstring/format "%s/%s@%s"
                                   (-> request :skill :namespace)
                                   (-> request :skill :name)
                                   (-> request :skill :version))}
           (if (:trace-id request) {:traceId (:trace-id request)}))))
       (let [start (time/now)]
         (when (:atomist/cloudrun? request)
           (log/debug "Cloud Run execution started"))
         (log/infof "----> starting %s/%s@%s %s %s %s (%s)"
                    (or (-> request :skill :namespace) "missing")
                    (or (-> request :skill :name) "missing")
                    (or (-> request :skill :version) "missing")
                    meta/module-name meta/version meta/generated-at meta/tag)
         (<! (handler request))
         (when (:atomist/cloudrun? request)
           (log/debugf "Cloud Run execution took %d ms, finished with status: 'ok'" (- (time/now) start))))
       (when (and
              (.. js/process -env -TOPIC)
              (not (= "true" (.. js/process -env -LOCAL_SKILL_RUNNER))))
         (<! (log/close-logger)))
       (catch :default ex
         (log/error ex)
         (<! (log/close-logger))
         request)))))

(defn compile-send-reponse-function [request]
  (if (not (= "true" (.. js/process -env -LOCAL_SKILL_RUNNER)))
    (partial (:sendreponse request)
             (or
              (:correlation-id request)
              (:correlation_id request)))
    (fn [obj]
      (js/Promise.
       (fn [accept _]
         (log/infof "-> %-10s%-20s%s"
                    (:topic request)
                    (:correlation-id request)
                    (js->clj obj :keywordize-keys true))
         (accept true))))))

(defn check-environment
  "pull ENV vars out of Process and add
     :team-id
     :team {:id \"\"}
     :graphql-endpoint
     :payload
     :bucket-name
     :topic
     :correlation-id
     :sendreponse function

     fail if full ENV not found"
  [handler]
  (letfn [(env [& envs]
            (reduce (fn [agg x]
                      (or agg (g/get (. js/process -env) x))) false envs))]
    (fn [request]
      (go-safe
       (let [request
             (cond-> request
               (env "ATOMIST_WORKSPACE_ID" "WORKSPACE_ID") (assoc :team-id (env "ATOMIST_WORKSPACE_ID" "WORKSPACE_ID"))
               (env "ATOMIST_GRAPHQL_ENDPOINT" "GRAPHQL_ENDPOINT") (assoc :graphql-endpoint (env "ATOMIST_GRAPHQL_ENDPOINT" "GRAPHQL_ENDPOINT"))
               (env "ATOMIST_PAYLOAD" "PAYLOAD") (assoc :payload (env "ATOMIST_PAYLOAD"))
               (env "ATOMIST_STORAGE" "STORAGE") (assoc :bucket-name (nth (re-find #"gs://(.*)" (env "ATOMIST_STORAGE" "STORAGE")) 1))
               (env "ATOMIST_TOPIC" "TOPIC") (assoc :topic (env "ATOMIST_TOPIC" "TOPIC"))
               (env "ATOMIST_CORRELATION_ID") (assoc :correlation-id (env "ATOMIST_CORRELATION_ID")))]
         (if (and
              (:team-id request)
              (:topic request)
              (:graphql-endpoint request)
              (:payload request)
              (:bucket-name request)
              (:correlation-id request))
           (<? (handler (assoc request
                               :sendreponse (compile-send-reponse-function request)
                               :team {:id (:team-id request)})))
           (assoc request
                  :atomist/status
                  {:code 1
                   :reason "environment is missing some of WORKSPACE_ID GRAPHQL_ENDPOINT ATOMIST_CORRELATION_ID ATOMIST_PAYLOAD TOPIC or ATOMIST_STORAGE"})))))))

(defn- exit [exit-code]
  (when (and
         (.. js/process -env -ATOMIST_PAYLOAD)
         (not (= "true" (.. js/process -env -IN_REPL))))
    (.exit js/process 0)))

(defn ^:api make-container-request
  "wrap handler in a go block and return a Promise that waits on this block"
  [handler]
  (fn [request]
    (let [done-channel (chan)]
      ;; handler chain exists when this go block completes
      (go-safe
       (try
         (>! done-channel (<? (handler request)))
         (catch :default ex
           (<! ((api/status #(go %)) (assoc request :atomist/status {:code 1 :reason (str ex)})))
           (>! done-channel :failed))))
      (let [prom (promise/chan->promise done-channel)]
        (.info js/console prom)
        (.catch
         (.then
          prom
          (fn [result]
            (.info js/console (clj->js result))
            (exit 0)))
         (fn [error]
           (.error js/console error)
           (exit 1)))))))

(defn ^:api cloud-run
  [handler]
  (fn [request]
      ;; put the chain behind an expression listener for CloudRun
    (let [app (new express)]
      (log/info "listen for requests")
      (.use app (.json body-parser #js {:limit "1024mb"}))
      (.post app "/" (fn [req resp]
                       (.catch
                        (.then
                         (.then
                          (resolvePayload (.. req -body -message))
                          (fn [message]
                            (.info js/console (.stringify js/JSON message))
                            (handler (assoc
                                       (js->clj message :keywordize-keys true)
                                       :trace-id (-> (.get req "x-cloud-trace-context")
                                                     (string/split #"/")
                                                     first)
                                       :event-id (.. req -body -message -messageId)))))
                         (fn [_]
                           (.sendStatus ^js resp 201)))
                        (fn [_]
                          (.sendStatus ^js resp 201)))))
      (.listen app (or (.. js/process -env -PORT) 8080)))))

(defn ^:api add-metadata
  [handler]
  (fn [request]
    (go-safe
     (<! (handler (-> request
                      (api/event->request)
                      ((fn [request]
                         (-> request
                             (assoc :correlation-id (:correlation_id request))
                             (assoc :atomist/cloudrun? true)
                             (assoc :team {:id (or (:team_id request) (-> request :team :id))})
                             (assoc :team-id (or (:team_id request) (-> request :team :id)))
                             (assoc :sendreponse (compile-send-reponse-function request)))))))))))

(def mw-cloudrunner-request
  (api/compose-middleware
   [create-logger]
   [add-metadata]
   [make-container-request]
   [cloud-run]))

(def mw-skill-runner-request
  (api/compose-middleware
   [create-logger]
   [read-atomist-payload]
   [check-environment]
   [make-container-request]))

(defn mw-make-container-request
  [handler]
  (fn [request]
    (if (.. js/process -env -ATOMIST_PAYLOAD)
      ((mw-skill-runner-request handler) request)
      ((mw-cloudrunner-request handler) request))))
