;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at
;;
;;     http://www.apache.org/licenses/LICENSE-2.0
;;
;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.
;; Copyright © 2021 Atomist, Inc.
;;
;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at
;;
;;     http://www.apache.org/licenses/LICENSE-2.0
;;
;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns atomist.ecr
  (:require
   [goog.string :as gstring]
   [atomist.time]
   [atomist.gcp :as gcp]
   [atomist.cljs-log :as log]
   [atomist.async :refer-macros [go-safe <?]]
   [atomist.promise :as promise]
   [cljs.core.async :as async :refer [close! go chan <! >!]]
   [atomist.docker :as docker]
   [http.client :as http]
   [cljs-node-io.core :as io]
   [cljs.pprint :refer [pprint]]
   [goog.object :as o]
   ["@aws-sdk/client-ecr" :as ecr-service]
   ["@aws-sdk/client-ecr-public" :as ecr-service-public]
   ["@aws-sdk/client-sts" :as sts-service]
   ["aws-sdk" :as aws-sdk]
   [atomist.json :as json]))

(set! *warn-on-infer* false)

(def private-ecr-service-constructor (.-ECR ecr-service))

(enable-console-print!) 

(defn account-host
  [account-id region]
  (gstring/format "%s.dkr.ecr.%s.amazonaws.com" account-id region))

(defn wrap-error-in-exception [message err]
  (ex-info message {:err err}))

(defn list-repositories
 "assumes that the AWS sdk is initialized (assumeRole may have already switched roles to third party ECR)
    get the first 1000 repositories using DescribeRepositoriesCommand" 
  [third-party-account-id ecr-client]
  (promise/from-promise 
    (.send ecr-client (new (.-DescribeRepositoriesCommand ecr-service) #js {:registryId third-party-account-id
                                                                            :maxResults 1000}))
    (fn [data]
      (-> (.-repositories data) 
          (js->clj :keywordize-keys true)))
    (partial wrap-error-in-exception "failed to run DescribeRepositoriesCommand")))

(defn describe-images
  "assumes that the AWS sdk is intialized
    get the first 1000 Images in a repo using DescribeImagesCommand"
  [third-party-account-id repository-name ecr-client]
  (promise/from-promise
   (.send ecr-client (new (.-DescribeImagesCommand ecr-service) #js {:registryId third-party-account-id
                                                                     :repositoryName repository-name
                                                                     :maxResults 1000}))
   (fn [data]
     (-> (.-imageDetails data) 
         (js->clj :keywordize-keys true)))
   (partial wrap-error-in-exception "failed to run DescribeImagesCommand")))

(defn get-authorization-token-command
  [ecr-client]
  (promise/from-promise
   (.send ecr-client (new (.-GetAuthorizationTokenCommand ecr-service) #js {}))
   (fn [data]
     (-> data
         (. -authorizationData)
         (aget 0)
         (. -authorizationToken)))
   (partial wrap-error-in-exception "failed to get authorization token command")))

(defn get-public-authorization-token-command
  [ecr-client]
  (promise/from-promise
   (.send ecr-client (new (.-GetAuthorizationTokenCommand ecr-service-public) #js {}))
   (fn [data]
     (-> data
         (. -authorizationData)
         (. -authorizationToken)))
   (partial wrap-error-in-exception "failed to get authorization token command")))

(defn ingest-latest-tags
  [account-id region callback ecr-client]
  (go-safe
   (let [access-token (:access-token (<? (get-authorization-token-command ecr-client)))
         repositories (<? (list-repositories account-id ecr-client))]
     (log/infof "Ingesting from %s repositories" (count repositories))
     (doseq [repo repositories
             {:keys [imageDigest
                     imagePushedAt
                     imageTags]} (<? (describe-images
                                       account-id
                                       (:repositoryName repo)
                                       ecr-client))
             :let [repository (:repositoryName repo)
                   tag-or-digest (or imageDigest (first imageTags)) ]]
       (when-let [manifests (not-empty (<? (docker/get-labelled-manifests
                                            (account-host account-id region)
                                            access-token
                                            repository
                                            tag-or-digest)))]
         (log/infof "Found %s manifests for %s:%s" (count manifests) repository tag-or-digest)
         (<? (callback repository manifests tag-or-digest)))))))

(defn call-aws-sdk-service
  "call aws-sdk v3 operations 
     - may use STS to assume role if there is an arn present.  Otherwise, default to use creds without STS"
  [{:keys [role-arn external-id access-key-id secret-access-key region]}
   service-constructor
   operation]
  (let [client (new (.-STS sts-service) #js {:region region
                                             :credentials #js {:accessKeyId access-key-id
                                                               :secretAccessKey secret-access-key}})]
    (if (and role-arn external-id)
      (promise/from-promise
       (.send client
              (new (.-AssumeRoleCommand sts-service)
                   #js {:ExternalId external-id
                        :RoleArn role-arn
                        :RoleSessionName "atomist"
                        :credentials #js {:accessKeyId access-key-id
                                          :secretAccessKey secret-access-key}}))
       (with-meta
         (fn [data]
           (operation
            (new service-constructor
                 #js {:region region
                      :credentials #js {:accessKeyId (.. data -Credentials -AccessKeyId)
                                        :secretAccessKey (.. data -Credentials -SecretAccessKey)
                                        :sessionToken (.. data -Credentials -SessionToken)}})))
         {:async true})
       (partial wrap-error-in-exception "failed to create token"))
      (operation (new service-constructor
                      #js {:region region
                           :credentials #js {:accessKeyId access-key-id
                                             :secretAccessKey secret-access-key}})))))

(defn ecr-auth
  "get an ecr authorization token"
  [{:keys [region] :as params}]
  (go-safe
   (log/infof "Authenticating ECR in region %s" region)
   {:access-token
    (<? (call-aws-sdk-service
         params
         (.-ECR ecr-service)
         get-authorization-token-command))}))

(defn ecr-public-auth
  "get an ecr authorization token"
  [{:keys [region] :as params}]
  (go-safe
   (log/infof "Authenticating ECR in region %s" region)
   {:access-token
    (<? (call-aws-sdk-service
         params
         (.-ECRPUBLIC ecr-service-public)
         get-public-authorization-token-command))}))

(defn get-labelled-manifests-from-private-repository
  "get manifests (with labels) for private registries
    - use region and account-id to figure out host"
  [{:keys [account-id region access-key-id] :as params} repository tag-or-digest]
  (log/infof "get-image-info:  %s@%s/%s" region access-key-id tag-or-digest)
  (go-safe
   (let [auth-context (<? (ecr-auth params))]
     (<? (docker/get-labelled-manifests
          (account-host account-id region)
          (:access-token auth-context) repository tag-or-digest)))))

(defn get-labelled-manifests-from-public-repository
  "get manifests (with labels) for public registries
     host will probably always just be public.ecr.aws"
  [{:keys [repository-name host digest tag] :as params}]
  (log/infof "get-image-info:  %s/%s@%s" host repository-name (or digest tag))
  (go-safe
    ;; do not use role-arn or external-id for public registry access
   (let [auth-context (<? (ecr-public-auth (dissoc params :external-id :role-arn)))]
     (<? (docker/get-labelled-manifests
          host
          (:access-token auth-context) repository-name (or digest tag))))))


