;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at
;;
;;     http://www.apache.org/licenses/LICENSE-2.0
;;
;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.
;; Copyright © 2021 Atomist, Inc.
;;
;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at
;;
;;     http://www.apache.org/licenses/LICENSE-2.0
;;
;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns atomist.ecr
  (:require
   [goog.string :as gstring]
   [atomist.time]
   [atomist.gcp :as gcp]
   [atomist.cljs-log :as log]
   [atomist.async :refer-macros [go-safe <?]]
   [atomist.promise :as promise]
   [cljs.core.async :refer [close! go chan <! >!]]
   [atomist.docker :as docker]
   [cljs-node-io.core :as io]
   [cljs.pprint :refer [pprint]]
   [goog.object :as o]
   ["@aws-sdk/client-ecr" :as ecr-service]
   ["@aws-sdk/client-ecr-public" :as ecr-service-public]
   ["@aws-sdk/client-sts" :as sts-service]
   ["aws-sdk" :as aws-sdk]
   [atomist.json :as json]))

(set! *warn-on-infer* false)

(comment
  (def third-party-account-id (.. js/process -env -ECR_ACCOUNT_ID))
  (def third-party-arn "arn:aws:iam::111664719423:role/atomist-ecr-integration")
  (def third-party-external-id "atomist")
  (def third-party-secret-key (.. js/process -env -ECR_SECRET_ACCESS_KEY))
  (def third-party-access-key-id (.. js/process -env -ECR_ACCESS_KEY_ID))
  (def atomist-account-id (.. js/process -env -ASSUME_ROLE_ACCOUNT_ID))
  (def atomist-access-key-id (.. js/process -env -ASSUME_ROLE_ACCESS_KEY_ID))
  (def atomist-secret-key (.. js/process -env -ASSUME_ROLE_SECRET_ACCESS_KEY))
  (def params-with-arn {:region "us-east-1"
                        :account-id third-party-account-id
                        :access-key-id atomist-access-key-id
                        :secret-access-key atomist-secret-key
                        :role-arn third-party-arn
                        :external-id third-party-external-id})
  (def params-without-arn
    {:region "us-east-1"
     :access-key-id third-party-access-key-id
     :secret-access-key third-party-secret-key})
  (enable-console-print!))

(defn wrap-error-in-exception [message err]
  (ex-info message {:err err}))

(defn list-repositories
 "assumes that the AWS sdk is initialized (assumeRole may have already switched roles to third party ECR)" 
  [third-party-account-id ecr-client]
  (promise/from-promise 
    (.send ecr-client (new (.-DescribeRepositoriesCommand ecr-service) #js {:registryId third-party-account-id}))
    (fn [data]
      (-> (.-repositories data) (js->clj :keywordize-keys true)))
    (partial wrap-error-in-exception "failed to run DescribeRepositoriesCommand")))

(defn get-authorization-token-command
  [ecr-client]
  (promise/from-promise
   (.send ecr-client (new (.-GetAuthorizationTokenCommand ecr-service) #js {}))
   (fn [data]
     (-> data
         (. -authorizationData)
         (aget 0)
         (. -authorizationToken)))
   (partial wrap-error-in-exception "failed to get authorization token command")))

(defn get-public-authorization-token-command
  [ecr-client]
  (promise/from-promise
   (.send ecr-client (new (.-GetAuthorizationTokenCommand ecr-service-public) #js {}))
   (fn [data]
     (-> data
         (. -authorizationData)
         (. -authorizationToken)))
   (partial wrap-error-in-exception "failed to get authorization token command")))

(defn call-aws-sdk-service
  "call aws-sdk v3 operations 
     - may use STS to assume role if there is an arn present.  Otherwise, default to use creds without STS"
  [{:keys [role-arn external-id access-key-id secret-access-key region]}
   service-constructor
   operation]
  (let [client (new (.-STS sts-service) #js {:region region
                                             :credentials #js {:accessKeyId access-key-id
                                                               :secretAccessKey secret-access-key}})]
    (if (and role-arn external-id)
      (promise/from-promise
       (.send client
              (new (.-AssumeRoleCommand sts-service)
                   #js {:ExternalId external-id
                        :RoleArn role-arn
                        :RoleSessionName "atomist"
                        :credentials #js {:accessKeyId access-key-id
                                          :secretAccessKey secret-access-key}}))
       (with-meta
         (fn [data]
           (operation
            (new service-constructor
                 #js {:region region
                      :credentials #js {:accessKeyId (.. data -Credentials -AccessKeyId)
                                        :secretAccessKey (.. data -Credentials -SecretAccessKey)
                                        :sessionToken (.. data -Credentials -SessionToken)}})))
         {:async true})
       (partial wrap-error-in-exception "failed to create token"))
      (operation (new service-constructor
                      #js {:region region
                           :credentials #js {:accessKeyId access-key-id
                                             :secretAccessKey secret-access-key}})))))

(comment
  ;; we are querying across accounts here (using role arn and sts)
  (go (pprint (<! (call-aws-sdk-service
                   params-with-arn 
                   (.-ECR ecr-service)
                   (partial list-repositories third-party-account-id)))))
  ;; use creds to get an auth token
  (go (pprint (<! (call-aws-sdk-service
                   params-without-arn
                   (.-ECR ecr-service)
                   get-authorization-token-command))))
  ;; use role arn and sts to get an auth token
  (go (pprint (<! (call-aws-sdk-service
                   params-with-arn
                   (.-ECR ecr-service)
                   get-authorization-token-command)))))

(defn account-host
  [account-id region]
  (gstring/format "%s.dkr.ecr.%s.amazonaws.com" account-id region))

(defn ecr-auth
  "get an ecr authorization token"
  [{:keys [region] :as params}]
  (go-safe
   (log/infof "Authenticating ECR in region %s" region)
   {:access-token
    (<? (call-aws-sdk-service
         params
         (.-ECR ecr-service)
         get-authorization-token-command))}))

(defn ecr-public-auth
  "get an ecr authorization token"
  [{:keys [region] :as params}]
  (go-safe
   (log/infof "Authenticating ECR in region %s" region)
   {:access-token
    (<? (call-aws-sdk-service
         params
         (.-ECRPUBLIC ecr-service-public)
         get-public-authorization-token-command))}))

(defn get-labelled-manifests-from-private-repository
  "get manifests (with labels) for private registries
    - use region and account-id to figure out host"
  [{:keys [account-id region access-key-id] :as params} repository tag-or-digest]
  (log/infof "get-image-info:  %s@%s/%s" region access-key-id tag-or-digest)
  (go-safe
   (let [auth-context (<? (ecr-auth params))]
     (<? (docker/get-labelled-manifests
          (account-host account-id region)
          (:access-token auth-context) repository tag-or-digest)))))

(defn get-labelled-manifests-from-public-repository
  "get manifests (with labels) for public registries
     host will probably always just be public.ecr.aws"
  [{:keys [repository-name host digest tag] :as params}]
  (log/infof "get-image-info:  %s/%s@%s" host repository-name (or digest tag))
  (go-safe
    ;; do not use role-arn or external-id for public registry access
   (let [auth-context (<? (ecr-public-auth (dissoc params :external-id :role-arn)))]
     (<? (docker/get-labelled-manifests
          host
          (:access-token auth-context) repository-name (or digest tag))))))

(comment

  (go (def token (:access-token (<! (ecr-public-auth {:secret-access-key atomist-secret-key
                                                      :access-key-id atomist-access-key-id
                                                      :region "us-east-1"
                                                      :host "public.ecr.aws"
                                                      :repository-name "amazonlinux/amazonlinux"
                                                      :tag "latest"})))))

  ;; note public repositories use a whole different set of apis for authentication
  ;; this tests with atomist creds (the ones we pull from secrets with function identity)
  (go
    (pprint
     (<!
      (get-labelled-manifests-from-public-repository
       {:secret-access-key atomist-secret-key
        :access-key-id atomist-access-key-id
        :region "us-east-1"
        :host "public.ecr.aws"
        :repository-name "amazonlinux/amazonlinux"
        :tag "latest"}))))

  ;; first test with the third party creds
  (go
    (pprint
     (<!
      (get-labelled-manifests-from-private-repository
       params-without-arn
       "pin-test"
       "sha256:5a703f57de904d1b189df40206458a6e3d9e9526b7bfd94dab8519e1c51b7a0c"))))

  ;; next, test with the atomist creds, and a third party trusted arn
  (go
    (pprint
     (<!
      (get-labelled-manifests-from-private-repository
       params-with-arn
       "pin-test"
       "sha256:5a703f57de904d1b189df40206458a6e3d9e9526b7bfd94dab8519e1c51b7a0c")))))
