(ns atomist.cli
  (:require [atomist.local-runner :as lr]
            [atomist.middleware :as mw]
            [clojure.tools.cli :as tc]
            [atomist.cljs-log :as log]
            [cljs.nodejs :as node]
            [goog.string :as gstring]
            [goog.string.format]
            [cljs-node-io.core :as io :refer [slurp spit]]
            [cljs.reader]))

(defn repo-promise [f]
  (fn [repo]
    (js/Promise.
     (fn [accept reject]
       (accept (f repo))))))

(defn ^:export visitRepo
  "Set an environment and reate a fake command handler
    returns Promise that will fulfill when the work is done"
  [{:keys [env team mapper reducer]}]
  (lr/set-env (keyword env))
  (-> (lr/fake-command-handler team "sync" "raw-message" "channel-id" "user-id")
      (lr/add-configuration {:name "default"
                             :parameters []})
      (assoc :fake-handler (constantly true))
      (lr/call-event-handler
       (mw/handler #js {:sync (fn [request]
                                (.withRepoIterator request
                                                   (repo-promise mapper)
                                                   #js {:clone true}
                                                   reducer))}))))

(def cli-options [["-e" "--env ENV" "Environment Name"
                   :default "custom"
                   :validate [#{"custom" "prod" "staging" "prod-github-auth"}
                              "only these environments are supported"]]
                  ["-t" "--team TEAM" "workspace id"]
                  ["-m" "--module MODULE" "javascript mapper module file"]
                  ["-h" "--help" nil]])

(defn load-module [s]
  (let [js (node/require s)]
    [(. js -mapper) (. js -reducer)]))

(defn ^:export main []
  (let [{:keys [options summary errors]}
        (tc/parse-opts (. js/process -argv) cli-options)]
    (cond
      (contains? options :help)
      (do
        (log/info summary)
        (.exit js/process 0))
      (and errors (not (empty? errors)))
      (do
        (log/error errors)
        (log/info summary)
        (.exit js/process 1))
      :else
      (.then
       (visitRepo (merge options
                         (let [[m r] (load-module (:module options))]
                           {:mapper m
                            :reducer r})))
       (fn [_] (.exit js/process 0))))))

(defn- has-common-clj [f]
  (let [m (cljs.reader/read-string (slurp f))]
    (->> (drop 3 m)
         (partition 2)
         (map #(into [] %))
         (into {})
         :dependencies
         (some #(if (= 'atomist/common-clj (first %)) %))
         (second))))

(comment
  (visitRepo
   {:team "T095SFFBK"
    :env "prod"
    :mapper (fn [repo]
              (let [f (io/file (. repo -basedir) "project.clj")]
                (merge
                 {:repo (gstring/format "%s/%s" (. repo -owner) (. repo -repo))}
                 {:exists (and (.exists f) (has-common-clj f))})))
    :reducer (fn [coll] (cljs.pprint/pprint (->> (js->clj coll :keywordize-keys true)
                                                 (filter :exists)
                                                 (into []))))}))