(ns atomist.gitflows
  (:require [cljs.core.async :refer [<! >! close! timeout chan] :as async]
            [atomist.promise :as promise]
            [atomist.git :as git]
            [atomist.cljs-log :as log]
            [goog.string :as gstring]
            [goog.string.format]
            [cljs-node-io.proc :as proc]
            [cljs-node-io.core :refer [slurp]]
            [clojure.string :as s]
            ["os" :as os]
            ["tmp" :as tmp]
            ["fast-glob" :as fast-glob]
            ["semver" :as semver]
            [cljs-node-io.core :as io]
            [atomist.github :as github])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn- empty-tmp-dir [context]
  (let [c (chan)]
    (.dir tmp
          (clj->js {:keep false :prefix (str "atm-" (. js/process -pid))})
          (fn [err path]
            (go
              (if err
                (>! c (merge context {:error err}))
                (>! c (merge context {:path path})))
              (close! c))))
    c))

;; Note:  this is important when running gcf functions because this can leak memory
;;        for in mem tmp filesystem
(defn- recursive-delete
  "spawn rm to clean up Project.baseDir"
  [p]
  (go
    (let [[error stdout stderr] (<! (proc/aexec (gstring/format "rm -fr %s" (:path p))))]
      (if error
        {:error error :stderr stderr}))))

(defn- filter-token [s]
  (try
    (-> s
        (s/replace #"//.*:x-oauth-basic@" "//token@x-oauth-basic@")
        (s/replace #"//atomist:.*@" "//atomist:token@"))
    (catch :default _
      (log/warn "error filtering " s)
      "")))

(defn- log-error [{:keys [error stderr stdout exit-code] :as x}]
  (if error
    (log/warnf "%s (%d) stderr: %s\nstdout: %s"
               (filter-token (str error))
               exit-code
               (if stderr (filter-token stderr) "")
               (if stdout (filter-token stdout) "")))
  x)

(defn- error? [{:keys [error]}]
  (not (nil? error)))

(defn- no-errors
  [c ops]
  (if (first ops)
    (let [[type op & args] (first ops)]
      (recur
       (go
         (let [context (<! c)]
           (if (not (:error context))
             (case type
               :sync (log-error (apply op context args))
               :async (log-error (<! (apply op context args)))
               :async-git (log-error (<! (git/from-git context (apply op context args)))))
             context)))
       (rest ops)))
    c))

(defn do-with-shallow-cloned-project
  "project middleware that does a shallow clone of the ref in a tmp directory
    returns chan that will emit the value returned by the async project-callback or close if there's a failure"
  [project-callback token ref]
  (go
    (let [{:keys [error path sha]} (log-error (<! (no-errors
                                                   (go (merge ref {:token token}))
                                                   [[:async empty-tmp-dir]
                                                    [:async-git git/clone-ref]
                                                    [:async-git git/config "user.name" (or (-> ref :author :name) "atomist-bot")]
                                                    [:async-git git/config "user.email" (or (-> ref :author :email) "bot@atomist.com")]
                                                    [:async-git git/config "core.symlinks" "false"]
                                                    [:async-git git/sha]
                                                    [:sync (fn [c] (assoc c :sha (s/trim (:stdout c))))]])))]
      (when (not error)
        (let [result (<! (project-callback (merge ref {:path path :sha sha :token token})))]
          (when-not (:atomist.api/do-not-delete-cloned-repo result)
            (<! (recursive-delete {:path path})))
          result)))))

(defn- remote?->chan
  "asynchronously look for a remote branch ref on origin

    channel emits nil or the short name of the remote branch ref"
  [p branch]
  (go
    (let [{:keys [stdout]} (log-error (<! (git/from-git p (git/ls-remote p))))]
      (->> (s/split-lines (if (string? stdout) stdout (slurp stdout)))
           (map #(second (re-find #".*refs/heads/(.*)$" %)))
           (filter #(= branch %))
           (empty?)
           (not)))))

(defn- hasBranch?
  "the sdm hasBranch can only check for local branches.  This function also checks for remote refs.
    returns channel that emits boolean (true if there is a local or remote origin branch with this name)"
  [p branch-name]
  (go
    (let [{:keys [stdout]} (<! (git/from-git p (git/has-branch p branch-name)))]
      (if (s/includes? stdout branch-name)
        true
        (<! (remote?->chan p branch-name))))))

(defn commit-then-push
  "project middleware for wrapping an project editor in a commit then push flow
     project-callback returns chan that will emit either :done or :failure"
  [project-callback commit-message]
  (fn [p]
    (go
      (<! (project-callback p))
      (<! (no-errors
           (go p)
           [[:async-git git/add-all]
            [:async-git git/commit commit-message]
            [:async-git git/push (:branch p)]])))))

(defn with-commit-on-branch
  [project-callback {:keys [message]}]
  (fn [p]
    (go
      (<! (project-callback p))
      (let [context (<! (no-errors
                         (go p)
                         [[:async-git git/add-all]
                          [:async-git git/commit message]
                          [:async-git git/push (:branch p)]]))]
        (cond
          (and (error? context)
               (= 1 (:exit-code context))
               (s/includes? (:stdout context) "nothing to commit")) :skipped
          (error? context) :failure
          :else :success)))))

(defn working-dir-clean? [p]
  (go
    (let [{:keys [stdout stderr exit-code]} (<! (git/from-git p (git/porcelain-status p)))]
      (and (or (= 0 exit-code) (nil? exit-code))
           (= "" stdout)))))

(defn- complete [git-ops]
  (cond
    (and (error? git-ops)
         (= 1 (:exit-code git-ops))
         (s/includes? (:stdout git-ops) "nothing to commit")) :skipped
    (error? git-ops) :failure
    (:pull-request-number git-ops) {:raised (:pull-request-number git-ops)}
    :else :unknown))

(defn with-pr
  [project-callback {:keys [branch target-branch body title labels]}]
  (fn [p]
    ;; case 1:  new push to base, no existing branch/PR -> create new edit-branch and make change
    ;; case 2:  new push to base, branch exists -> rebase edit-branch and edit again?
    ;; case 3:  new push to edit-branch, if it's the rebase skill, we should run again.
    ;;          It's probably safe to run again in all cases because it'll be an empty commit so nothing would get Pushed.
    (go
      (try
        (log/debugf "cloned-branch: %s, target-branch: %s, edit-branch: %s"
                    (or (:branch p) "missing") (or target-branch "missing") (or branch "missing"))
        (when (= (:branch p) branch)
          (log/warnf "cloned branch (%s) should be the same as branch %s" (:branch p) branch)
          (throw (ex-info "bad parameters for with-pr" {:cloned-branch (:branch p) :branch branch})))

        ;; STEP 0 - apply and check whether this branch is already in sync
        ;;        - close any open PRs with this head if the base is now in sync
        (let [results (<! (project-callback p))]

          (if (<! (working-dir-clean? p))

            (do
              (<! (github/close-pr p branch))
              :skipped)

            (do
              ;; NO remote branch ref
              (if (not (<! (hasBranch? p branch)))
                (-> (<! (no-errors
                         (go p)
                         [[:async-git git/create-branch branch]
                          [:async-git git/checkout-branch branch]
                          [:sync (fn [p] (assoc p :branch branch))]
                          [:async-git git/add-all]
                          [:async-git git/commit body]
                          [:async-git git/push]
                          [:async github/raise-pr title body target-branch labels]]))
                    (log-error)
                    (complete))
                ;; fetch remote, move on to branch and re-apply
                (-> (<! (no-errors
                         (go p)
                         [[:async-git git/fetch-branch branch]
                          [:async-git git/checkout-force branch]
                          [:sync (fn [p] (assoc p :branch branch))]
                          [:async (fn [p] (go (<! (project-callback p)) p))]
                          [:async-git git/add-all]
                          [:async-git git/commit body]
                          [:async-git git/push]
                          [:async github/raise-pr title body target-branch labels]]))
                    (log-error)
                    (complete))))))
        (catch :default ex
          (log/error ex (str (ex-data ex)))
          :failure)))))

(defn do-with-files
  "middleware for project handling - wrap project-callback in file iterator
     project-callback returns chan which emits vector of callback returns"
  [file-callback & patterns]
  (fn [p]
    (go
      (let [entries (<! (promise/from-promise (fast-glob
                                               (clj->js (into [] patterns))
                                               (clj->js {:dot true :cwd (:path p)}))))
            file-callbacks (<! (async/reduce
                                conj []
                                (async/merge
                                 (for [entry entries]
                                   (file-callback (io/file (io/file (:path p)) entry))))))]
        file-callbacks))))

(defn next-version [last-tag]
  (let [new-tag ((.-inc semver) last-tag "patch")]
    (log/infof "%s -> %s" last-tag new-tag)
    new-tag))

(defn with-tag [handler k]
  (fn [request]
    ;; report Check failures
    (go
      (log/info "with-tag")
      (let [context (merge
                     (:ref request)
                     {:token (:token request)}
                     {:path (.getPath (io/file (.. js/process -env -ATOMIST_HOME)))})]
        (let [{:keys [response]}
              (<! (atomist.gitflows/no-errors
                   (go context)
                   [[:async-git git/fetch-tags]
                    [:async-git git/git-rev-list]
                    [:async-git git/git-describe-tags]
                    [:sync #(assoc % :tag (next-version (s/trim (:stdout %))))]
                    [:async (fn [{:keys [tag] :as context}]
                              (go
                                (let [response (<! (handler (assoc request k tag)))]
                                  (merge
                                   context
                                   {:response response}
                                   (if (= "failure" (:checkrun/conclusion response))
                                     {:error true})))))]
                    [:async-git git/tag]
                    [:async-git git/push-tag]]))]
          response)))))

(comment
  (set! (.. js/process -env -ATOMIST_HOME) "/Users/slim/tmp/common-clj")
  (go (println (<! ((with-tag #(go %) :tag) {:ref {:owner "atomisthq" :repo "common-clj" :branch "master"}
                                             :token ""})))))

(comment
 ;; test a do-with-files over a file callback
  (go (println (<! ((do-with-files (fn [f] (go (.getPath f))) "**/*.cljs") {:path "/Users/slim/atmhq/api-cljs"}))))
 ;; test a with-commit-on-master
  (go (println (<! (-> (fn [p] (go (io/spit (io/file (:path p) "touch.txt") "stuff5")))
                       (with-commit-on-master)
                       (do-with-shallow-cloned-project (.. js/process -env -GITHUB_TOKEN)
                                                       {:owner "slimslender"
                                                        :repo "spring-boot-220"
                                                        :branch "master"})))))
 ;; new branch
  (go (println (<! (-> (fn [p] (go (io/spit (io/file (:path p) "touch.txt") "stuff6")))
                       (with-pr {:branch "branch1"
                                 :target-branch "master"
                                 :title "great new title"
                                 :body "something interesting"})
                       (do-with-shallow-cloned-project (.. js/process -env -GITHUB_TOKEN)
                                                       {:owner "slimslender"
                                                        :repo "spring-boot-220"
                                                        :branch "master"})))))
 ;; new change, but use the same branch
  (go (println (<! (-> (fn [p] (go (io/spit (io/file (:path p) "touch.txt") "stuff8")))
                       (with-pr {:branch "branch1"
                                 :target-branch "master"
                                 :title "great new title"
                                 :body "something interesting"})
                       (do-with-shallow-cloned-project (.. js/process -env -GITHUB_TOKEN)
                                                       {:owner "slimslender"
                                                        :repo "spring-boot-220"
                                                        :branch "master"})))))
 ;; existing branch where the change is no longer required
  )