(ns atomist.middleware.repo-iterator
  (:require [atomist.promise :as promise]
            [cljs.core.async :refer [<!] :refer-macros [go]]
            [atomist.api :as api]
            [atomist.container :as container]
            [atomist.json :as json]
            [atomist.cljs-log :as log]
            [clojure.string :as s]
            [atomist.github :as github]))

(defn- ->clj [obj]
  (js->clj obj :keywordize-keys true))

(defn compile-patch-repo [request]
  (fn [repo-config]
    (promise/chan->promise
     (go
       (let [response (<! (github/patch-repo request (json/->obj repo-config)))]
         (log/infof "%s/%s -> %s"
                    (-> request :ref :owner)
                    (-> request :ref :repo)
                    (:status response))
         (:status response))))))

(defn with-github-repo
  "  params
       handler - async js request handler
       fun - async ({ref config}, async patch-repo() => status) => Any
           - (whatever the function returns will be stored as status on the request)"
  [handler js-fun]
  (fn [request]
    (go
      (<! (handler (assoc request
                          :return-value
                          (<! (promise/from-promise
                               (js-fun
                                (-> request
                                    (select-keys [:token])
                                    (assoc :topics (<! (github/repo-topics request)))
                                    (assoc :patchRepo (compile-patch-repo request))
                                    (merge (select-keys (:ref request) [:owner :repo]))
                                    (merge (if (:project request) (assoc (:project request) :basedir (-> request :project :path))))
                                    (clj->js)))
                               ->clj
                               ->clj))))))))

(defn maybe-clone [handler opts]
  (fn [request]
    (go
      (if (:clone opts)
        (<! ((api/clone-ref handler) request))
        (<! (handler request))))))

(defn maybe-creds [handler opts]
  (fn [request]
    (go
     (if (and (:clone opts) (not (:token request)))
       (<! ((api/extract-github-token handler) request))
       (<! (handler request))))))

(defn maybe-edit [handler opts]
  (fn [request]
    (go
      (cond
        (contains? opts :with_pr)
        (<! ((api/edit-inside-PR
              handler
              (:with_pr opts)) request))
        (contains? opts :with_commit)
        (<! ((api/edit-inside-PR
              handler
              (assoc
               (:with_commit opts)
               :type :commit-then-push)) request))
        :else
        (<! (handler request))))))

(defn compile-repo-iterator [request]
  (fn [js-fun & opts]
    (promise/chan->promise
     (go
       (<! ((api/repo-iterator
             (fn [_] (go true))
             (-> (fn [request] (go request))
                 (with-github-repo js-fun)
                 (maybe-edit (if (first opts) (js->clj (first opts) :keywordize-keys true) {}))
                 (maybe-clone (if (first opts) (js->clj (first opts) :keywordize-keys true) {}))))
            request))))))

(defn compile-with-repo [request]
  (fn [js-fun & opts]
    (promise/chan->promise
     (go
       (<! ((-> (fn [request] (go request))
                (with-github-repo js-fun)
                (maybe-edit (if (first opts) (js->clj (first opts) :keywordize-keys true) {}))
                (maybe-clone (if (first opts) (js->clj (first opts) :keywordize-keys true) {}))
                (maybe-creds (if (first opts) (js->clj (first opts) :keywordize-keys true) {})))
            request))))))
