(ns atomist.middleware
  (:require [atomist.promise :as promise]
            [cljs.core.async :refer [<!] :refer-macros [go]]
            [atomist.api :as api]
            [atomist.container :as container]
            [atomist.json :as json]
            [atomist.cljs-log :as log]
            [clojure.string :as s]
            [atomist.github :as github]
            [atomist.slack.block-messages :as blockmessage]
            [atomist.middleware.repo-iterator :as repo-iterator]
            [atomist.local-runner :as lr]))

(defn destination [request s]
  (cond
    (s/starts-with? s "#") (api/channel request (apply str (rest s)))
    (s/starts-with? s "@") (api/user request (apply str (rest s)))
    :else (throw (ex-info "destinations must start with # or @" request))))

(defn- ->js [request]
  (-> request (clj->js)))

(defn- ->promise [d]
  (js/Promise. (fn [accept _] (accept d))))

(defn- ->clj [obj]
  (js->clj obj :keywordize-keys true))

(defn ^:export dispatch
  "generate a dispatch function for different event handler (and command handlers) but make them look the same
   the dispatch is a map of names to async request handler functions

   each request handler function can be wrapped in middleware but should ultimately return the request to go down
   into the status handler"
  [obj]
  (fn [data callback]
    (api/make-request
     data callback
     (-> (fn [request]
           (go
             (let [dispatch-map (js->clj obj)]
               (log/infof "dispatch %s %s" dispatch-map (or (:command request) (:operation request)))
               (cond
                 (contains? dispatch-map (:command request))
                 (<! (promise/from-promise
                      ((get dispatch-map (:command request)) (->js request))
                      ->clj ->clj))
                 (contains? dispatch-map (-> request :operation))
                 (<! (promise/from-promise
                      ((get dispatch-map (:operation request)) (->js request))
                      ->clj ->clj))
                 :else
                 (<! (api/finish request "failed to dispatch any keys %s" (keys dispatch-map)))))))

         (api/set-message-id)
         (api/status)))))

(defn with-check-run-producing-handler
  "used in clj-kondo"
  [& {:keys [cmd ->args on-success on-failure ext middleware]}]
  (-> (api/finished)
      (api/run-command :cmd cmd
                       :->args ->args
                       :on-success on-success
                       :on-failure on-failure)
      (api/with-github-check-run :name :check-name :conclusion (fn [request]
                                                                 (if (:neutral-only? request)
                                                                   "neutral"
                                                                   (:checkrun/conclusion request))))
      (middleware)
      (api/extract-github-token)
      (api/create-ref-from-event)
      (api/add-skill-config)
      (api/stop-if-no-files-detected :substr ext)
      (api/status)
      (container/mw-make-container-request)))

(defn from-js [f]
  (fn [& args]
    (go (js->clj (<! (promise/from-promise (apply f (map clj->js args)))) :keywordize-keys true))))

(defn to-js [f]
  (fn [& args]
    (promise/chan->promise (apply f (map (fn [arg] (js->clj arg :keywordize-keys true)) args)))))

(defn ^:export withCheckRunSendingLinter
  [obj]
  ((with-check-run-producing-handler :cmd (. obj -cmd)
     :ext (. obj -ext)
     :->args (from-js (. obj -constructArgs))
     :on-success (from-js (. obj -onSuccess))
     :on-failure (from-js (. obj -onError))) {}))

(defn ^:export handler
  "create a top-level handler function and dispatch using a map of command/eventOperationNames to request handlers
   written in JavaScript.
   Set up the middleware as if the request will want to use Slack, but not GitHub.
   Inject a blockMessage function into the request Object passed to each handler function.

   Used in greet-bot-skill,git-repo-config-skill"
  [dispatch-map]
  (letfn [(compile-block-message-fn [request]
            (fn [message & args]
              (promise/chan->promise
               (api/block-message (if (first args)
                                    (-> request
                                        (update-in [:source :slack] dissoc :thread_ts)
                                        (destination (first args)))
                                    (-> request
                                        (update-in [:source :slack] dissoc :thread_ts)))
                                  (-> message
                                      (js->clj :keywordize-keys true)
                                      (blockmessage/fix-js-naming-for-actions)
                                      (->> (into [])))))))
          (wrapper [f] (-> (api/finished)
                           ((fn [h]
                              (fn [request]
                                (go
                                  (api/trace "javascript")
                                  (let [java-script-result (<!
                                                            (promise/from-promise
                                                             (f (-> request
                                                                    (assoc :blockMessage (compile-block-message-fn request))
                                                                    (assoc :withRepoIterator (repo-iterator/compile-repo-iterator request))
                                                                    (assoc :withRepo (repo-iterator/compile-with-repo request))
                                                                    (clj->js)))))]
                                    (log/info "javascript result:  " java-script-result)
                                    (<! (h request)))))))
                           (api/add-skill-config)
                           (api/set-message-id)
                           (api/add-slack-source-to-event)
                           (api/create-ref-from-event :stop-if-missing false)
                           (api/log-event)
                           (api/extract-parameters)
                           (api/status)))]
    (fn [data sendreponse]
      (api/make-request
       data
       sendreponse
       (api/dispatch (reduce-kv #(assoc %1 %2 (wrapper %3)) {} (js->clj dispatch-map :keywordize-keys true)))))))

(comment
  (do
    (require '[atomist.local-runner :as lr])
    (lr/set-env :prod)
    (let [h (handler (clj->js {:sync (fn [request]
                                       (log/infof "target config %s" (. request -config))
                                       (.withRepoIterator request
                                                          (fn [repo]
                                                            (when (.includes (. repo -topics) (. request -topic))
                                                              (log/infof "%s/%s (%s)"
                                                                         (or (. repo -owner) "missing")
                                                                         (or (. repo -repo) "missing")
                                                                         (or (. repo -topics) "missing")))
                                                            (js/Promise. (fn [accept reject] (accept true))))))}))]
      (-> (lr/fake-command-handler "T095SFFBK" "sync" "sync it" "D0HMP77EZ" "U09MZ63EW")
          (assoc :configurations [{:name "default" :parameters [{:name "config" :value "{}"}
                                                                {:name "topic" :value "atomist-skill"}]}])
          (lr/call-event-handler h)))))