(ns atomist.middleware
  (:require [atomist.promise :as promise]
            [cljs.core.async :refer [<!] :refer-macros [go]]
            [atomist.api :as api]
            [atomist.container :as container]
            [atomist.json :as json]
            [atomist.cljs-log :as log]
            [clojure.string :as s]
            [atomist.github :as github]
            [atomist.slack.block-messages :as blockmessage]))

(defn destination [request s]
  (cond
    (s/starts-with? s "#") (api/channel request (apply str (rest s)))
    (s/starts-with? s "@") (api/user request (apply str (rest s)))
    :else (throw (ex-info "destinations must start with # or @" request))))

(defn- ->js [request]
  (-> request (clj->js)))

(defn- ->promise [d]
  (js/Promise. (fn [accept _] (accept d))))

(defn- ->clj [obj]
  (js->clj obj :keywordize-keys true))

(defn ^:export withSlack
  "Make a handler ready to send block messages

    param
      handler - async function (data, send-blocks) => Void

    returns Promise"
  [handler]
  (fn [data callback]
    (api/make-request
     data callback
     (-> (fn [request]
           (go
             (let [handler-response (<! (promise/from-promise
                                         (handler
                                          (->js request)
                                          (fn [channel-or-user json-blocks]
                                            (try
                                              (let [blocks (-> json-blocks (json/->obj) :blocks)]
                                                (-> request
                                                    (destination channel-or-user)
                                                    (api/block-message blocks)))
                                              (catch :default ex
                                                (log/error ex "invalid application/json blocks")
                                                {:failed ex}))))))]
               (if (:failed handler-response)
                 (<! (api/finish request :failure (str "handler failure " (str (:failed handler-response)))))
                 (assoc request :handler-response handler-response)))))
         (api/add-skill-config :config :topic)
         (api/add-slack-source-to-event)
         (api/create-ref-from-event)
         (api/status)))))

(defn ^:export dispatch
  "generate a dispatch function for different event handler (and command handlers) but make them look the same
   the dispatch is a map of names to async request handler functions

   each request handler function can be wrapped in middleware but should ultimately return the request to go down
   into the status handler"
  [obj]
  (fn [data callback]
    (api/make-request
     data callback
     (-> (fn [request]
           (go
             (let [dispatch-map (js->clj obj)]
               (log/infof "dispatch %s %s" dispatch-map (or (:command request) (:operation request)))
               (cond
                 (contains? dispatch-map (:command request))
                 (<! (promise/from-promise
                      ((get dispatch-map (:command request)) (->js request))
                      ->clj ->clj))
                 (contains? dispatch-map (-> request :operation))
                 (<! (promise/from-promise
                      ((get dispatch-map (:operation request)) (->js request))
                      ->clj ->clj))
                 :else
                 (<! (api/finish request "failed to dispatch any keys %s" (keys dispatch-map)))))))

         (api/set-message-id)
         (api/status)))))

(defn withPolicy
  "  params
       handler - async js request handler
       fun - async ({ref config}, async patch-repo() => status) => Any
           - (whatever the function returns will be stored as status on the request)"
  [handler fun]
  (fn [request]
    (go
      (<! (handler (assoc request
                          :status
                          (<! (promise/from-promise
                               (fun
                                (clj->js (select-keys request [:ref :config]))
                                (fn [] (promise/chan->promise
                                        (go
                                          (let [response (<! (github/patch-repo request (:repo-config request)))]
                                            (log/infof "%s/%s -> %s"
                                                       (-> request :ref :owner)
                                                       (-> request :ref :repo)
                                                       (:status response))
                                            (:status response))))))
                               ->clj
                               ->clj))))))))

(defn ^:export withRepoIteratorAndFilter
  ([filter-fun apply-fun]
   (withRepoIteratorAndFilter ->promise filter-fun apply-fun))
  ([handler filter-fun apply-fun]
   (fn [request]
     (promise/chan->promise
      (go
        (<! ((-> (fn [r] (go (<! (promise/from-promise (handler (->js r)) ->clj ->clj))))
                 (api/repo-iterator
                  (fn [r _] (go (<! (promise/from-promise
                                     (filter-fun (-> r
                                                     (assoc :topics (<! (github/repo-topics r)))
                                                     (->js)))
                                     ->clj
                                     ->clj))))
                  (-> (fn [request]
                        (go
                          (log/infof "finished with %s - %s" (:ref request) (:status request))
                          request))
                      (withPolicy apply-fun)))) (js->clj request :keywordize-keys true))))))))

(defn ^:export withSlackMessage
  ([handler slack-callback]
   (fn [request]
     (promise/chan->promise
      (go
        (<! ((-> (fn [r] (go (<! (promise/from-promise (handler (->js r)) ->clj ->clj))))
                 (api/from-channel (fn [r] (go (<! (promise/from-promise
                                                    (slack-callback
                                                     (->js r)
                                                     #js {:simpleMessage
                                                          (fn [dest, s]
                                                            (log/info "send message to " r)
                                                            (promise/chan->promise
                                                             (go (<! (-> r
                                                                         (destination dest)
                                                                         (api/simple-message s))))))}))))))
                 (api/set-message-id)
                ;; TODO need to not break when called from a command handler
                 #_(api/add-slack-source-to-event)) (js->clj request :keywordize-keys true)))))))
  ([slack-callback]
   (withSlackMessage ->promise slack-callback)))

(defn with-check-run-producing-handler [& {:keys [cmd ->args on-success on-failure ext middleware]}]
  (-> (api/finished)
      (api/run-command :cmd cmd
                       :->args ->args
                       :on-success on-success
                       :on-failure on-failure)
      (api/with-github-check-run :name :check-name :conclusion (fn [request]
                                                                 (if (:neutral-only? request)
                                                                   "neutral"
                                                                   (:checkrun/conclusion request))))
      (middleware)
      (api/extract-github-token)
      (api/create-ref-from-event)
      (api/add-skill-config)
      (api/stop-if-no-files-detected :substr ext)
      (api/status)
      (container/mw-make-container-request)))

(defn from-js [f]
  (fn [& args]
    (go (js->clj (<! (promise/from-promise (apply f (map clj->js args)))) :keywordize-keys true))))

(defn to-js [f]
  (fn [& args]
    (promise/chan->promise (apply f (map (fn [arg] (js->clj arg :keywordize-keys true)) args)))))

(defn ^:export withCheckRunSendingLinter
  [obj]
  ((with-check-run-producing-handler :cmd (. obj -cmd)
     :ext (. obj -ext)
     :->args (from-js (. obj -constructArgs))
     :on-success (from-js (. obj -onSuccess))
     :on-failure (from-js (. obj -onError))) {}))

(defn ^:export handler [dispatch-map]
  (letfn [(compile-block-message-fn [request]
            (fn [message & args]
              (promise/chan->promise
               (api/block-message (if (first args)
                                    (-> request
                                        (destination (first args)))
                                    request)
                                  (-> message
                                      (js->clj :keywordize-keys true)
                                      (blockmessage/fix-js-naming-for-actions)
                                      (->> (into [])))))))
          (wrapper [f] (-> (api/finished)
                           ((fn [h]
                              (fn [request]
                                (go
                                  (api/trace "dispatch")
                                  (let [java-script-result (<!
                                                            (promise/from-promise
                                                             (f (-> request
                                                                    (assoc :blockMessage (compile-block-message-fn request))
                                                                    (clj->js)))))]
                                    (log/info "javascript result:  " java-script-result)
                                    (<! (h request)))))))
                           (api/add-skill-config)
                           (api/set-message-id)
                           (api/add-slack-source-to-event)
                           (api/log-event)
                           (api/extract-parameters)
                           (api/status)))]
    (fn [data sendreponse]
      (api/make-request
       data
       sendreponse
       (api/dispatch (reduce-kv #(assoc %1 %2 (wrapper %3)) {} (js->clj dispatch-map :keywordize-keys true)))))))