(ns atomist.api
  (:require [atomist.json :as json]
            [atomist.graphql :as graphql]
            [atomist.graphql-channels :as channels]
            [atomist.json :as json]
            [atomist.meta :as meta]
            [atomist.schemata]
            [cljs.core.async :refer [>! <! timeout chan] :as async]
            [cljs.pprint :refer [pprint]]
            [atomist.promise :as promise]
            [cljs-node-io.core :as io :refer [slurp spit]]
            [atomist.cljs-log :as log]
            [com.rpl.specter :as specter]
            [atomist.time]
            [goog.string :as gstring]
            [goog.string.format]
            [atomist.shell :as shell]
            [atomist.gitflows :as gitflow]
            [clojure.string :as s]
            [atomist.time :as time]
            [atomist.repo-filter]
            [cljs.spec.alpha :as spec]
            [atomist.slack :as slack]
            [atomist.github :as github])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn repo->slug [{repo-name :name {org :owner} :org}]
  (gstring/format "%s/%s" org repo-name))

(defn- send-on-response-channel [o]
  (let [callback (chan)]
    (go
      (try
        ;; TODO stop sending invalid response messages
        (try
          (if (not (spec/valid? :response/message o))
            (log/debugf "invalid response spec:  %s" (with-out-str (spec/explain :response/message o))))
          (catch :default ex
            (log/error "unable to verify :response/message")))
        (log/debug "send-on-response-channel " o " with promise "
                   (<! (promise/from-promise ((:sendreponse o) (clj->js o)))))
        (>! callback {:success true})
        (catch :default ex
          (>! callback {:error "unable to send"
                        :ex ex}))))
    callback))

(defn- default-destination [o]
  (if (or (not (:destinations o)) (empty? (:destinations o)))
    (-> o
        (update :destinations (constantly [(merge
                                            (:source o)
                                            {:user_agent "slack"})]))
        (update-in [:destinations 0] (fn [destination]
                                       (if (some? (:slack destination))
                                         (update destination :slack dissoc :user)
                                         destination))))
    o))

(defn ^:api trace [s]
  (log/debug "----> " s))

(defn add-destination [o {:keys [id name] :as chat-team} {:keys [id name] :as channel}]
  (update o :destinations (fnil conj []) {:user_agent "slack"
                                          :slack {:team chat-team
                                                  :channel channel}}))

(defn ^:api channel
  "set message destination channel name
     params
       o      - command request or incoming event payload
       c      - string name of message channel"
  [o c]
  (-> o
      (default-destination)
      (update-in [:destinations 0 :slack] (fn [x] (-> x (assoc :channel {:name c}) (dissoc :user))))))

(defn ^:api user
  "set message destination to user DM
     params
       o      - command request or incoming event payload
       c      - string name of user to DM"
  [o u]
  (-> o
      (default-destination)
      (update-in [:destinations 0 :slack] (fn [x] (-> x (assoc :user {:name u}) (dissoc :channel))))))

(defn ^:api add-post-mode [request mode]
  (if (not (#{:ttl :update_only :always} mode))
    (do
      (log/warnf "%s is not a valid post_mode: using ttl" mode)
      request)
    (assoc request :post_mode mode)))

(defn ^:api add-actions
  "actions must be an [] of {:keys [id command parameters]} (parameter optional)
     - we are not adding the :automation map to the action (assumes all actions route to this skill)
     - TODO we need to warn if a request does not contain an automation field (which can
       happen when things come in from events)."
  [o actions]
  (letfn [(add-message-id-parameter [action]
            (update action :parameters (fn [parameters]
                                         (if (not (some #(= (:name %) "messageId") parameters))
                                           (conj parameters {:name "messageId" :value (:id o)})
                                           parameters))))]
    (-> o
        (update :actions (fnil concat []) actions)
        (update :actions (fn [actions]
                           (if (:id o)
                             (map add-message-id-parameter actions)
                             actions))))))

(defn ^:api ingest
  "ingest a new custom event
     params
       o         - incoming event or command request
       x         - custom event
       channel   - name of custom event channel"
  [o x channel]
  (-> o
      (select-keys [:sendreponse :api_version :correlation_id :team :automation])
      (assoc :content_type "application/json"
             :body (json/->str x)
             :destinations [{:user_agent "ingester"
                             :ingester {:root_type channel}}])
      (send-on-response-channel)))

(defn ^:api success-status
  "on command request, send status that the invocation was successful"
  ([o status-message visibility]
   (-> (select-keys o [:sendreponse :correlation_id :api_version :automation :team :command :source :destinations])
       (assoc :status (merge
                       {:code 0 :reason status-message}
                       (if visibility
                         {:visibility (name visibility)})))
       (assoc :content_type "application/x-atomist-status+json")
       (default-destination)
       (send-on-response-channel)))
  ([o]
   (success-status o "success" :info)))

(defn ^:api failed-status
  "on command request, send status that the invocation failed"
  ([o status-message]
   (-> (select-keys o [:sendreponse :correlation_id :api_version :automation :team :command :source :destinations])
       (assoc :status {:code 1 :reason status-message})
       (assoc :content_type "application/x-atomist-status+json")
       (default-destination)
       (send-on-response-channel)))
  ([o]
   (failed-status o "failure")))

(defn- send-status [request status]
  (ingest request {:category "SkillLog"
                   :correlation_context {:correlation_id (-> request :correlation_id)
                                         :automation {:name "skill"
                                                      :version "0.1.0"}}
                   :level "INFO"
                   :message (json/->str {:status (name status)
                                         :event_id (:eventId request)})
                   :team_id (-> request :team :id)
                   :timestamp (time/week-date-time-no-ms-now)} "AtomistLog"))

(defn ^:api finish [request & {:keys [failure success visibility] :as status}]
  (go
    (<! (if failure
          (failed-status request failure)
          (success-status request (or success "success") visibility)))
    (assoc request :api/status (or status {:finished true}))))

(defn ^:api snippet-message
  "send snippet as bot
    params
      o           - command request or event
      content-str - content as string
      filetype    - valid slack filetype
      title       - string title"
  [o content-str filetype title]
  (-> (select-keys o [:sendreponse :correlation_id :api_version :automation :team :source :command :destinations :id])
      (assoc :content_type "application/x-atomist-slack-file+json")
      (assoc :body (json/clj->json {:content content-str :filetype filetype :title title}))
      (default-destination)
      (send-on-response-channel)))

(defn ^:api simple-message
  "send simple message as bot
     params
       o - command request or event
       s - string message"
  [o s]
  (-> (select-keys o [:sendreponse :correlation_id :api_version :automation :team :source :command :destinations :id])
      (assoc :content_type "text/plain")
      (assoc :body s)
      (assoc :timestamp (atomist.time/now))
      (default-destination)
      (send-on-response-channel)))

(defn ^:api continue
  "continue this request with some additional parameter specs
     (all added parameters must be required)"
  [o params]
  (-> (select-keys o [:sendreponse :correlation_id :parameters :api_version :automation :team :source :command :destinations :parameter_specs :id])
      (assoc :content_type "application/x-atomist-continuation+json")
      (update :parameter_specs (fnil concat []) (->> params (map #(assoc % :required true)) (into [])))
      (default-destination)
      (send-on-response-channel)))

(defn ^:api get-secret-value
  [o secret-uri]
  (->> o :secrets (filter #(= secret-uri (:uri %))) first :value))

(defn ^:api get-parameter-value
  "search command request for parameter
     params
       o              - command request
       parameter-name - string name
     returns nil if there's no parameter"
  [o parameter-name]
  (some->> (get-in o [:parameters])
           (filter #(= parameter-name (:name %)))
           first
           :value))

(defn ^:api parameter-expression [ch-request expression]
  (reduce
   (fn [agg [token param-name]]
     (clojure.string/replace agg token (or (get ch-request (keyword param-name)) (get-parameter-value ch-request param-name))))
   expression
   (re-seq #"\$\{(.*?)\}" expression)))

(defn ^:api mapped-parameter-value
  "search command request for mapped parameter
    params
      o              - command request
      parameter-name - string name
    returns nil if there's no parameter"
  [o parameter-name]
  (some->> (get-in o [:mapped_parameters])
           (filter #(= parameter-name (:name %)))
           first
           :value))

(defn ^:api delete-message
  [o]
  (-> (select-keys o [:sendreponse :correlation_id :api_version :automation :team :source :command :destinations :id])
      (assoc :content_type "application/x-atomist-delete")
      (assoc :timestamp (.getTime (js/Date.)))
      (default-destination)
      (send-on-response-channel)))

(defn- event->request
  [request]
  (if (:extensions request)
    (log/debugf "handling event %s for %s/%s - %s" (->> request :data str (take 40) (apply str)) (-> request :extensions :team_id) (-> request :extensions :team_name) (-> request :extensions :correlation_id)))
  (cond-> request
    (:extensions request) (assoc :api_version "1"
                                 :operation (-> request :extensions :operationName)
                                 :correlation_id (-> request :extensions :correlation_id)
                                 :team {:id (-> request :extensions :team_id)
                                        :name (-> request :extensions :team_name)})))

(defn- add-ids-to-commands [slack message-id]
  (let [num (atom 0)]
    (specter/transform [:attachments specter/ALL :actions specter/ALL]
                       #(if (:atomist/command %)
                          (-> %
                              (assoc-in [:atomist/command :id]
                                        (str (get-in % [:atomist/command :command])
                                             "-"
                                             (swap! num inc)))
                              (update-in [:atomist/command :parameters] (fnil conj []) {:name "messageId" :value message-id})
                              (assoc-in [:atomist/command :automation]
                                        {:name "@atomisthq/stupendabot-command-handler-cljs"
                                         :version "0.1.9"}))
                          %)
                       slack)))

(defn- transform-to-slack-actions [slack]
  (specter/transform [:attachments specter/ALL :actions specter/ALL]
                     #(if (:atomist/command %)
                        (let [action-id (get-in % [:atomist/command :id])]
                          (case (:type %)
                            "button"
                            (-> %
                                (dissoc :atomist/command)
                                (assoc :name (str "automation-command::" action-id))
                                (assoc :value action-id))
                            "select"
                            (-> %
                                (dissoc :atomist/command)
                                (assoc :name (str "automation-command::" action-id)))
                            %))
                        %)
                     slack))

(defn ^:api actionable-message
  "  params
       o       - incoming command request or event payload
       slack   - slack Message data where all actions may refer to
                 other CommandHandlers"
  [o slack & [opts]]

  (let [commands-with-ids (add-ids-to-commands slack (or (:id o) (str (random-uuid))))]

    (-> (select-keys o [:sendreponse :correlation_id :api_version :automation :team :source :command :destinations :id :post_mode])

        (merge opts)
        (assoc :content_type "application/x-atomist-slack+json")
        (assoc :timestamp (atomist.time/now))
        (assoc :body (-> commands-with-ids
                         (transform-to-slack-actions)
                         (json/->str)))
        (assoc :actions (->> (:attachments commands-with-ids)
                             (mapcat :actions)
                             (filter :atomist/command)
                             (mapv :atomist/command)))
        (default-destination)
        (send-on-response-channel))))

(defn ^:api block-message
  "  params
       blocks - array of Slack blocks (not a map!)
   TODO - validate that all blocks with actions have valid action attached"
  [o blocks]
  (-> (select-keys o [:sendreponse :correlation_id :api_version :automation :actions :team :source :command :destinations :id :post_mode])
      (assoc :content_type "application/x-atomist-slack+json")
      (assoc :timestamp (atomist.time/now))
      (assoc :body (-> {:text "fallback"
                        :blocks (json/->str blocks)}
                       (json/->str)))
      (default-destination)
      (send-on-response-channel)))

(defn ^:api add-message-id [ch-request]
  (if-let [id (or (:id ch-request) (get-parameter-value ch-request "messageId"))]
    (assoc ch-request :id id)
    (assoc ch-request :id (str (random-uuid)))))

(defn ^:mw post-mode [handler mode]
  (fn [request]
    (go
      (trace (str "set-post-mode " mode))
      (<! (handler (add-post-mode request mode))))))

(defn ^:mw log-event [handler]
  (fn [request]
    (go
      (trace "log-event")
      (log/debug (dissoc request :secrets :token))
      (<! (handler request)))))

(defn ^:mw validate-repo-in-graph
  "request middleware
     ensure Repo referenced in a command is present in graph - if present, add it to the request with key :repo"
  [handler repo-name-fn]
  (fn [ch-request]
    (go
      (if-let [repo-name (repo-name-fn ch-request)]
        (let [repo (<! (channels/repo-query->channel ch-request repo-name))]
          (if repo
            (<! (handler (assoc ch-request :repo repo)))
            (let [message (gstring/format "we were unable to find the repo %s" (first repo))]
              (<! (simple-message ch-request message))
              (<! (finish ch-request :failure message)))))
        (let [message (gstring/format "this request did not contain a repo name")]
          (<! (simple-message ch-request message))
          (<! (finish ch-request :failure message)))))))

(defn ^:mw check-required-parameters
  "request middleware
     used during command handler processing to check that required parameters are present

   - if the request is a command handler request and has missing parameters then the bot will go back to the user
     to get more data
   - if the required parameters are all present then the pipeline will continue with the parameter values keywordized
     and merged into the request"
  [handler & parameters]
  (fn [ch-request]
    (go
      (let [missing (->> parameters (filter (complement #(or (get-parameter-value ch-request (:name %))
                                                             (contains? ch-request (keyword (:name %)))))))]
        (if (empty? missing)
          (<! (handler (merge
                        ch-request
                        (reduce
                         (fn [agg p] (if-let [v (get-parameter-value ch-request (:name p))]
                                       (assoc agg (keyword (:name p)) v)
                                       agg))
                         {}
                         parameters))))
          (do
            (if-let [p (and
                        (= 1 (count missing))
                        (#{"single" "multiple"} (-> missing first :type :kind))
                        (first missing))]

              (<! (actionable-message ch-request {:attachments
                                                  [{:callback_id "callbackid"
                                                    :text (gstring/format "Please Enter a value for parameter %s" (:name p))
                                                    :markdwn_in ["text"]
                                                    :actions [{:text (gstring/format "select %s" (:name p))
                                                               :type "select"
                                                               :name "rug"
                                                               :options (->> (-> p :type :options)
                                                                             (map #(assoc {} :text (:description %) :value (:value %)))
                                                                             (into []))
                                                               :atomist/command {:command (:command ch-request)
                                                                                 :parameter_name (:name p)
                                                                                 :parameters []}}]}]}))
              (<! (continue ch-request missing)))
            (<! (finish ch-request :success "asking bot for additional parameters"))))))))

(defn ^:mw extract-cli-parameters
  "request middleware
    used during command handler processing to parse the raw bot message into cli parameters.  The parameters
    are merged into the request keywordized.

    middleware-params
    options - must contain a clojure.tools.cli option parser definition"
  [handler options]
  (fn [ch-request]
    (go
      (trace "extract-cli-parameters")
      (try
        (let [{:keys [options]} (shell/raw-message->options ch-request options)]
          (if options
            (<! (handler (merge ch-request options)))
            (<! (handler ch-request))))
        (catch :default ex
          (log/error "extract-cli-parameters:  " ex)
          (<! (finish :failure "unable to parse raw_message")))))))

(defn ^:mw extract-github-user-token
  "request middleware
     used during command handler processing to extract a github user token from a request

     - if this middleware finds that the chat user is linked to an authorized ScmProvider then the token will be added
         to the request
     - if the current user is not authorized then the bot will ask the user authorize and then stop the pipeline with status success."
  [handler]
  (fn [ch-request]
    (go
      (let [linked-person (<! (channels/linked-person->channel ch-request (-> ch-request :source :slack :user :id)))]
        (if linked-person
          (<! (handler (assoc ch-request :token (-> linked-person :scmId :credential :secret)
                              :github-login (-> linked-person :scmId :login)
                              :person linked-person)))
          (let [github-provider (<! (channels/github-provider->channel ch-request))]
            (if github-provider
              (<! (block-message ch-request (slack/github-auth-blocks ch-request (:id github-provider))))
              (<! (simple-message ch-request "there is no scm credential linked for this repo - please type `@atomist authorize github`")))
            (<! (finish ch-request :success "bot should ask user to authorize github"))))))))

(defn ^:mw extract-linked-repos
  "request middleware
     used during command handler processing to extract the set of linked repos for this channel

     - processor will stop if there are no linked channels so this should be inserted in the pipeline only when >0
       linked channels should be found.  It will output a message if there are zero linked channels"
  [handler]
  (fn [ch-request]
    (go
      (let [linked-repos (<! (channels/linked-repos->channel ch-request (-> ch-request :source :slack :channel :id)))]
        (if (not (empty? linked-repos))
          (<! (handler (assoc ch-request :linked-repos linked-repos)))
          (do
            (<! (simple-message ch-request "there are no linked repos in this channel.  Please run this from a channel with linked repos or link a repo using `@atomist repos`"))
            (<! (finish ch-request))))))))

(defn ^:mw set-message-id
  "request middleware
     used during command handler processing to extract messageIds or create new random ones

     This is useful to insert into a pipeline that will keep updating a message over a set of events."
  [handler]
  (fn [ch-request]
    (go
      (<! (handler (add-message-id ch-request))))))

(defn ^:mw create-ref-from-first-linked-repo
  "request middleware
    use a requests's :linked-repos to construct a git ref with {:keys [repo owner branch]}
    - the repo name is a parameter expression which can use parameter values
    - currently always using master branch (should this be a middleware parameter?)"
  [handler]
  (fn [request]
    (go
      (let [repo (first (:linked-repos request))]
        (<! (handler (assoc request :ref {:repo (:name repo)
                                          :owner (-> repo :org :owner)
                                          :branch (:branch request)})))))))

(defn ^:mw user-should-choose-one-linked-repo
  "request middleware
     when the request :linked-repos has >1 linked repo, force the user to choose
     if, on the other hand, the request has zero repos, stop and tell the user that this command should be run from a channel
       with at least one linked repo"
  [handler]
  (fn [request]
    (go
      (let [slug (or (:slug request) (get-parameter-value request "slug"))
            linked-repos
            (->> (:linked-repos request)
                 (filter #(or (not slug) (= slug (repo->slug %)))))]
        (cond
          (= 1 (count linked-repos))
          (<! (handler (assoc request :linked-repos linked-repos)))

          (= 0 (count linked-repos))
          (do
            (<! (simple-message request "This command must run from a channel with at least one linked Repo"))
            (<! (finish request :success "no linked repos: user notified")))

          (> (count linked-repos) 1)
          (do
            (<! (actionable-message request {:attachments
                                             [{:callback_id "callbackid"
                                               :text "Please choose from one of the currently linked Repos"
                                               :markdwn_in ["text"]
                                               :actions [{:text "select one of the Repos"
                                                          :type "select"
                                                          :name "rug"
                                                          :options (->> linked-repos
                                                                        (map #(let [slug (repo->slug %)]
                                                                                (assoc {} :text slug :value slug)))
                                                                        (into []))
                                                          :atomist/command {:command (:command request)
                                                                            :parameter_name "slug"
                                                                            :parameters (or (:parameters request) [])}}]}]}))
            (<! (finish request :success "choose a Repo:  user choice sent"))))))))

(defn ^:mw show-results-in-slack
  "request middleware
    will send a slack snippet containing the results in the current request.  Mostly for debugging
    This request must already have been configured to send to a particular Slack channel or be a command handler request."
  [handler & {:keys [result-type] :or {result-type "results"}}]
  (fn [request]
    (go
      (trace "show-results-in-slack")
      (if-let [results (:results request)]
        (<! (snippet-message request (json/->str results) "application/json" result-type))
        (<! (simple-message request (gstring/format "no %s" result-type))))
      (<! (handler request)))))

(defn ^:mw extract-github-token
  "request middleware
     will extract a github token by trying a few strategies

     - first check whether any request middleware has already found a github provider with a credential secret
       This is used when we have a shared authorization for the ResourceProvider

     - second, check whether any request middleware has stored a ref with an owner org.  If so, check whether this
       org has a GitHub Installation and then fetch an Installation token for this Org.

     If we find a token store it in the request under the :token key.  If we don't find a token, continue anyway.
     It's possible a good idea to supply an option to stop processing if no token is found."
  [handler]
  (fn [request]
    (go
      (let [{:keys [ref provider]} request
            secret (-> provider :credential :secret)]
        (if secret
          (<! (handler (assoc request :token secret)))
          (if-let [token (-> (<! (channels/graphql->channel request graphql/githubAppInstallationByOwner {:name (:owner ref)}))
                             :body
                             :data
                             :GitHubAppInstallation
                             first
                             :token
                             :secret)]
            (<! (handler (assoc request :token token)))
            (do
              (log/warn "did not find any GitHub creds for this Push")
              (<! (finish request :failure "stopping because of missing GitHub App Installation")))))))))

(defn ^:mw create-ref-from-event
  [handler]
  (fn [request]
    (log/info "request " request)
    (go
      (cond
        (contains? (:data request) :Push)
        (if-let [{:keys [name] repo-id :id {:keys [owner scmProvider] owner-id :id} :org :as repo} (-> request :data :Push first :repo)]
          (<! (handler (assoc request :ref {:repo name
                                            :repo-id repo-id
                                            :owner (or owner (:owner repo))
                                            :owner-id owner-id
                                            :sha (-> request :data :Push first :after :sha)
                                            :branch (-> request :data :Push first :branch)}
                              :provider scmProvider)))
          (<! (finish request :failure "this event did not contain a valid Push")))
        (contains? (:data request) :Comment)
        (if-let [{:keys [issue pullRequest] :as comment} (-> request :data :Comment first)]
          (let [{{:keys [name] repo-id :id {:keys [owner scmProvider] owner-id :id} :org} :repo}
                (or issue pullRequest)]
            (<! (handler (assoc request :ref {:repo name
                                              :repo-id repo-id
                                              :owner owner
                                              :owner-id owner-id
                                              :branch (-> request :data :Push first :branch)}
                                :provider scmProvider))))
          (<! (finish request :failure "this event did not contain a valid Comment")))
        (contains? (:data request) :Repo)
        (if-let [{:keys [name] repo-id :id {:keys [owner scmProvider] owner-id :id} :org} (-> request :data :Repo first)]
          (<! (handler (assoc request :ref {:repo name
                                            :repo-id repo-id
                                            :owner owner
                                            :owner-id owner-id
                                            :branch (-> request :data :Push first :branch)}
                              :provider scmProvider)))
          (<! (finish request :failure "this event did not contain a valid Push")))
        :else
        (<! (finish request :failure (gstring/format "can not map ref from event type %s" (-> request :data keys))))))))

(defn ^:mw clone-ref
  [handler]
  (fn [{:keys [ref token] :as request}]
    (go
      (trace (gstring/format "clone-ref %s/%s" (-> ref :owner) (-> ref :repo)))
      ;; channel will close with no response if clone has failed
      (let [response (<! (gitflow/do-with-shallow-cloned-project
                          (fn [project] (go (<! (handler (assoc request :project project)))))
                          token
                          ref))]
        (or response (update request :errors (fnil conj []) (gstring/format "unable to clone %s" ref)))))))

(defn ^:mw edit-inside-PR
  "mostly run editor inside a PR but also allow for running with a commit on master"
  [handler configuration]
  (fn [{:keys [project] :as request}]
    (go
      (trace "edit-inside-PR")
      (try
        (if (:commit-on-master request)
          (let [commit-result (<! ((gitflow/with-commit-on-master
                                     (fn [project] (go (<! (handler (assoc request :project project)))))) project))]
            (cond
              (= :failure commit-result)
              (assoc request :edit-result :failure)
              :else
              (assoc request :edit-result :committed)))
          (let [pr-result (<! ((gitflow/with-pr
                                 (fn [project] (go (<! (handler (assoc request :project project)))))
                                 (if (map? configuration)
                                   configuration
                                   (configuration request))) project))]
            (cond
              (= :skipped pr-result)
              (assoc request :edit-result :skipped)
              (= :failure pr-result)
              (assoc request :edit-result :failure)
              :else
              (assoc request :edit-result :raised))))
        (catch :default _
          (log/error "error within edit-inside-PR"))))))

(defn ^:mw send-fingerprints
  "request middleware
     that will look for fingerprints in the request and send them to Atomist"
  [handler]
  (fn [request]
    (go
      (doseq [by-type (partition-by :type (:results request))]
        (if (not (empty? by-type))
          (let [o (assoc request :team {:id (-> request :extensions :team_id)})
                repo-branch-by-id (<! (channels/graphql->channel
                                       o
                                       graphql/repo-branch-ids
                                       {:owner (-> request :data :Push first :repo :org :owner)
                                        :repo (-> request :data :Push first :repo :name)
                                        :branch (-> request :data :Push first :branch)}))
                variables {:additions (->> by-type
                                           (map (fn [{:as fp}]
                                                  (-> fp
                                                      (assoc :data (json/->str (:data fp)))))))
                           :isDefaultBranch (=
                                             (-> request :data :Push first :repo :defaultBranch)
                                             (-> request :data :Push first :branch))
                           :type (-> by-type first :type)
                           :branchId (-> repo-branch-by-id :body :data :Repo first :branches first :id)
                           :sha (or
                                 (-> request :data :Push first :after :sha)
                                 (-> request :project :sha))
                           :repoId (-> request :data :Push first :repo :id)}
                response (<! (channels/graphql->channel
                              o
                              graphql/send-fingerprints
                              variables))]
            (log/info "send fingerprints:  " (:status response) (:body response)))))
      (<! (handler request)))))

(defn ^:mw from-channel
  "request middleware
     that pauses to run an async function ->channel and then continues with the functions result merged into request

   middleware params
     ->channel function (request) => channel where the channel result is needed by downstream processors
     key - the key to use when merging the async result into the request map"
  [handler ->channel & {:keys [key] :or {key :results}}]
  (fn [request]
    (go
      (try
        (trace "from-channel")
        (let [value (<! (->channel request))]
          (<! (handler (assoc request key value))))
        (catch :default ex
          (log/errorf "Error:  %s" ex))))))

(defn ^:mw merge-into-request
  [handler m]
  (fn [request]
    (go (<! (handler (merge request m))))))

(defn ^:mw from
  [handler f & {:keys [key]}]
  (fn [request]
    (go
      (try
        (<! (handler (assoc request key (f request))))
        (catch :default ex
          (<! (finish request :failure (gstring/format "from middleware failed to add key %s:  %s" key ex))))))))

(defn ^:mw add-skill-config
  "request middleware
     that looks for configuration added for a skill with dispatchStyle equal to multiple

   if the request has a configuration then named configuration values
     are merged into the request using a keywordized map"
  [handler & ks]
  (fn [request]
    (go
      (let [configuration (or (-> request :configuration) (-> request :configurations first))]
        (log/infof "found configuration %s - %s" (:name configuration) (:parameters configuration))
        (<! (handler (reduce
                      (fn [req k] (if-let [value (->> configuration :parameters (filter #(= (name k) (:name %))) first :value)]
                                    (assoc req k value)
                                    req))
                      request
                      ks)))))))

(defn ^:mw add-skill-config-by-configuration-parameter
  "request middleware
     that can be used by command handlers to lookup a particular skill configuration and reflect it into the request.

     params
       parameter-name is keyword for the required configuration name
       ks the names of config parameters that should be reflected into the skill"
  [handler parameter-name & ks]
  (fn [request]
    (go
      (let [config-name (get request parameter-name)
            configuration (->> (:configurations request)
                               (filter #(= (:name %) config-name))
                               first)]
        (if configuration
          (<! (handler (-> (reduce
                            (fn [req k] (if-let [value (->> configuration :parameters (filter #(= (name k) (:name %))) first :value)]
                                          (assoc req k value)
                                          (do
                                            (log/warnf "configuration %s did not contain the key %s" config-name k)
                                            req)))
                            request
                            ks)
                           (assoc :configuration configuration))))
          (do
            (<! (simple-message request (gstring/format "did not find a configuration named %s" parameter-name)))
            (<! (finish request :failure (gstring/format "stopped because we did not find a configuration named %s" parameter-name)))))))))

(defn ^:mw skip-push-if-atomist-edited
  "request middleware
    that checks whether a Push event contains a message with [atomist:edited]

    handler chain stops if the message contains [atomist:edited] and otherwise continues"
  [handler]
  (fn [request]
    (go
      (if (and (-> request :data :Push first :after :message) (s/includes? (-> request :data :Push first :after :message) "[atomist:edited]"))
        (do
          (log/info "skipping Push because after commit was made by Atomist")
          (<! (finish request
                      :success "skipping Push for Atomist edited Commits."
                      :visibility :hidden)))
        (<! (handler request))))))

(defn ^:mw add-slack-source-to-event
  "api middleware
     team-id is optional - if not provided, we pull it in from the graph
     channel is mandatory and serves as the default destination"
  [handler & {:keys [channel team-id]}]
  (fn [request]
    (go
      (let [slack-team (if team-id
                         {:id team-id}
                         (<! (channels/linked-slack-team->channel request)))]
        (<! (handler (-> request
                         (default-destination)
                         (update-in [:destinations 0 :slack] assoc :team slack-team :channel {:name channel}))))))))

(defn ^:mw add-gitub-installation-token [handler k]
  (fn [request]
    (go
      (<! (handler (assoc request :token (<! (channels/github-app-installation-token->channel request (request k)))))))))

(defn ^:mw repo-iterator
  "run a handler (forked-handler) over all of our current repos

    forked-handler should return a channel that will emit one value when the forked handler is finished

     - start with a channel that emits all the visible repo-refs in this team
     - map the forked-handler over these refs which means we will end up with a coll of channels
     - merge these channels and then reduce them into an array - this array
       will contain the results of all of the forked handlers

   when the final channel for the reduced results closes, we can run the join handler to continue
      the outer pipeline"
  ([join-handler forked-handler]
   (repo-iterator join-handler (constantly (go true)) forked-handler))
  ([join-handler ref-filter forked-handler]
   (fn [request]
     (go
       ;; map forked-handler against stream of repo-refs produces a stream of channels each containing the results
       ;; of the forked-handler
       ;; merge all channels and reduce to make sure everything closes
       (let [forks (<! (->> [(channels/repo-refs->channel request ref-filter)]
                            (async/map forked-handler)
                            (async/into [])))
             joined (<! (->> forks
                             (async/merge)
                             (async/reduce conj [])))]
         (log/debugf "joined %d forks" (count joined))
         (<! (join-handler (assoc request :plan (into [] joined)))))))))

(defn ^:mw status [handler & {:keys [visibility send-status send-visibility] :or {send-status (constantly "success")
                                                                                  send-visibility (constantly nil)}}]
  (fn [request]
    (go
      (let [{:keys [message success] :as response} (<! (handler request))]
        (log/infof "----> send status %s" (or message ""))
        (if (not (:api/status response))
          (<! (finish request :success (or success (send-status response)) :visibility (or visibility (send-visibility response)))))
        response))))

(defn ^:mw apply-repo-filter [handler & ks]
  (fn [request]
    (go
      (log/info "check repo filter scope " (:scope request))
      (if (or (not (:scope request))
              (atomist.repo-filter/ref-matches-repo-filter? request (:ref request) (:scope request)))
        (<! (handler request))
        (<! (finish request :success "should have been filtered" :visibility :hidden))))))

(defn ^:mw with-github-check-run [handler & {:keys [name]}]
  (fn [request]
    (go
      (cond
        (not name)
        (<! (finish request :failure "with-github-check-run must define a name.  Stopping."))
        (not (spec/valid? :atomist/ref-with-sha (:ref request)))
        (<! (finish request :failure (gstring/format
                                      "with-github-check-run: %s request requires a valid ref with owner, repo, sha.  Stopping\n  %s"
                                      (:ref request)
                                      (with-out-str (spec/explain :atomist/ref-with-sha (:ref request))))))
        :else
        (let [check-result (<! (github/create-check request
                                                    (-> request :ref :owner)
                                                    (-> request :ref :repo)
                                                    {:status "in_progress"
                                                     :name name
                                                     :head_sha (-> request :ref :sha)}))]
          (log/info "create-check status:  " (:status check-result))
          (let [response (<! (handler request))]
            (if (or (not (:checkrun/conclusion response))
                    (not (:checkrun/output response)))
              (log/warn "with-github-check-run: handlers should add :checkrun/conclusion and :checkrun/output to the request.  GitHub checkrun results will not be complete."))
            (log/info "update-check status:  "
                      (:status
                       (<! (github/update-check request
                                                (-> request :ref :owner)
                                                (-> request :ref :repo)
                                                (-> check-result :body :id)
                                                (merge
                                                 {:status "completed"
                                                  :name name
                                                  :head_sha (-> request :ref :sha)
                                                  :conclusion (or (:checkrun/conclusion response)
                                                                  "neutral")
                                                  :output (or (:checkrun/output response)
                                                              {:title "warning - skill did set output"
                                                               :summary "skill implementation should set a summary and an explicit conclusion"})})))))
            response))))))

(defn ^:api finished
  "middleware
     to create and send status Response messages and then to signal the function that processing is complete

   options
     :message - log a finished message into the User log
     :success or :send-status - set the `reason` string in a success status message
     :failure - set the `reason` string in a failure status
     :level or :send-level - set the `level` string in a success or failure messsage (use one of :info, :debug, :error, :warning)"
  [& {:keys [message success]}]
  (fn [request]
    (go
      (log/infof "----> finished %s" (or message ""))
      (assoc request :message message :success success))))

(defn ^:api dispatch [dispatch-map]
  (fn [request]
    (go
      (let [op (or (-> request :command keyword)
                   (-> request :extensions :operationName keyword))]
        (trace (gstring/format "dispatch %s" op))
        (cond

          (op dispatch-map)
          (<! ((op dispatch-map) request))

          (:default dispatch-map)
          (<! ((:default dispatch-map) request))

          :else
          (<! (finish request :failure (gstring/format "failed to dispatch any keys %s" (keys dispatch-map)))))))))

(defn ^:api make-request
  "This is probably the only function that should be called in a cljs skill handler.  It kicks off the request handler.

    - start with the incoming PubSubMessage.data, keywordize it, and use it as the request map
    - wrap the whole request processor in a Promise to satisfy the gcf Node.js runtime
    - create a done channel in the request so that request processing pipeline can finish early
    - wrap the whole pipeline in a try catch block
    - add a callback sendreponse function to the request map to handle writing events to the response topic
    - success/fail status are normally handled in the pipeline, but exception cases handled here will result in fail status"
  [data sendreponse handler]
  (promise/chan->promise
   (let [done-channel (chan)
         request (-> (js->clj data :keywordize-keys true)
                     (assoc :sendreponse sendreponse)
                     (event->request))]
     ;; handler chain exists when this go block completes
     (go
       (try
         ;;
         (if (.. js/process -env -TOPIC)
           (log/create-logger (:correlation_id request) (:eventId request) (-> request :team :id)))
         ;;
         (log/infof "----> starting %s %s %s %s %s %s (%s)"
                    (:correlation_id request) (:eventId request) (-> request :team :id)
                    meta/module-name meta/version meta/generated-at meta/tag)
         (<! (handler request))
         (>! done-channel :done)
         (catch :default ex
           (log/error ex)
           (<! (failed-status request))
           (>! done-channel :failed))))
     ;; wrap this channel in a promise
     done-channel)))

(defn compose-middleware [& args]
  (fn [handler]
    (fn [request]
      ((reduce (fn [h [mw & mw-args]] (apply mw h mw-args)) handler args) request))))
