(ns atomist.github
  (:require [http.client :as client]
            [goog.string :as gstring]
            [goog.string.format]
            [clojure.string :as s]
            [atomist.cljs-log :as log]
            [cljs.core.async :refer [<!]]
            [atomist.json :as json]
            [goog.crypt.base64 :as b64])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn github-v4 [token query variables]
  (go
    (let [response (<! (http.client/post
                        "https://api.github.com/graphql"
                        {:headers {"Authorization" (gstring/format "bearer %s" token)
                                   "Accept" "application/vnd.github.bane-preview+json"
                                   "User-Agent" "atomist"}
                         :body (json/->str {:query query
                                            :variables variables})}))]
      (log/debug "response " response)
      (if (= 200 (:status response))
        (:body response)
        (log/error "status was not 200")))))

(defn beautify [s]
  (try
    (let [pattern #"(\[[-\w]+:[a-z]+\])"
          ;; p (set! (.-flags pattern) "gm")
          tags (->> (re-seq pattern s)
                    (map second)
                    (sort))]
      (str
       (-> s
           (s/replace pattern "")
           (s/replace #"\n\s*\n\s" "\n\n"))
       "\n\n---\n<details><summary>Tags</summary><br/>\n"
       (->> tags
            (map #(str "<code>" % "</code>"))
            (interpose "\n")
            (apply str))
       "\n</details>"))
    (catch :default ex
      (log/warnf "failed to run pr beautification %s" ex)
      s)))

(defn- get-first-pr-for-head
  "GET /repos/:owner/:repo/pulls?state=open&head="
  [{:keys [token owner repo]} head-ref-name]
  (go
    (-> (<! (client/get (gstring/format "https://api.github.com/repos/%s/%s/pulls" owner repo)
                        {:headers {"Authorization" (gstring/format "bearer %s" token)
                                   "User-Agent" "atomist"}
                         :query-params {:state "open"
                                        :head (gstring/format "%s:%s" owner head-ref-name)}}))

        :body
        first)))

(defn- patch-pr-title
  "PATCH /repos/:owner/:repo/pulls/:number"
  [{:keys [token owner repo]} number title]
  (client/patch (gstring/format "https://api.github.com/repos/%s/%s/pulls/%s" owner repo number)
                {:headers {"Authorization" (gstring/format "bearer %s" token)
                           "User-Agent" "atomist"}
                 :body (json/->str {:title title})}))

(defn- patch-pr-state
  "PATCH /repos/:owner/:repo/pulls/:number"
  [{:keys [token owner repo]} number state]
  (if (#{"open" "closed"} state)
    (client/patch (gstring/format "https://api.github.com/repos/%s/%s/pulls/%s" owner repo number)
                  {:headers {"Authorization" (gstring/format "bearer %s" token)
                             "User-Agent" "atomist"}
                   :body (json/->str {:state state})})
    (log/warn "status must be #{\"open\" \"closed\"}")))

(defn- post-pr-comment
  "POST /repos/:owner/:repo/pulls/:number/comments"
  [{:keys [token owner repo]} number body]
  (client/post (gstring/format "https://api.github.com/repos/%s/%s/issues/%s/comments" owner repo number)
               {:headers {"Authorization" (gstring/format "bearer %s" token)
                          "User-Agent" "atomist"}
                :body (json/->str {:body body})}))

(defn- post-pr
  "POST /repos/:owner/:repo/pulls"
  [{:keys [token owner repo]} title body head base]
  (client/post (gstring/format "https://api.github.com/repos/%s/%s/pulls" owner repo)
               {:headers {"Authorization" (gstring/format "bearer %s" token)
                          "User-Agent" "atomist"}
                :body (json/->str {:title title
                                   :body body
                                   :base base
                                   :head head})}))

(defn log-github-http-status [expected s response]
  (if (not (= expected (:status response)))
    (log/warnf "%s: http status %s - %s" s (:status response) (:body response))))

(defn raise-pr
  "Create a PR or edit an existing one
    params
      p - should have keys :branch :owner :repo :token
    returns p with pull request number"
  [{:keys [branch] :as p} title body base-branch-ref]
  (go
    (if-let [pr (<! (get-first-pr-for-head p branch))]
      (do
        (log-github-http-status 200 "patch PR titld" (<! (patch-pr-title p (:number pr) title)))
        (log-github-http-status 201 "post PR comment" (<! (post-pr-comment p (:number pr) (beautify body))))
        (assoc p :pull-request-number (:number pr)))
      (let [response (<! (post-pr p title (beautify body) branch base-branch-ref))]
        (if (= 201 (:status response))
          (assoc p :pull-request-number (-> response :body :number))
          (let [message (gstring/format "PR creation failed %s <- %s: %s" base-branch-ref branch (:status response))]
            (log/error message)
            (assoc p :error message)))))))

(defn close-pr
  [p head-branch-ref]
  (go
    (if-let [pr (<! (get-first-pr-for-head p head-branch-ref))]
      (let [response (<! (patch-pr-state p (:number pr) "closed"))]
        (log/info "closing " (:number pr))
        (if (not (= 200 (:status response)))
          (log/warn "unable to close pr %s" (:url pr))))
      (log/warn "did not find pr for head ref %s" head-branch-ref))))

(defn pr-channel [request branch-name]
  (get-first-pr-for-head (assoc (:ref request) :token (:token request)) branch-name))

;; TODO handle paging
(defn all-labels-channel [request]
  (go
    (try
      (let [response (<! (client/get
                          (gstring/format "https://api.github.com/repos/%s/%s/labels" (-> request :ref :owner) (-> request :ref :repo))
                          {:headers {"User-Agent" "atomist"
                                     "Authorization" (gstring/format "Bearer %s" (:token request))}}))]
        (if (= 200 (:status response))
          (-> response :body)
          (log/warn "no GitHub labels found")))
      (catch :default ex
        (log/error "raised exception " ex)))))

(defn get-label [request name]
  (go
    (let [response (<!
                    (client/get
                     (gstring/format "https://api.github.com/repos/%s/%s/labels/%s"
                                     (-> request :ref :owner)
                                     (-> request :ref :repo)
                                     name)
                     {:headers {"User-Agent" "atomist"
                                "Authorization" (gstring/format "Bearer %s" (:token request))}}))]
      (if (= 200 (:status response))
        (:body response)))))

(defn add-label [request {:keys [name description color]}]
  (go

    (let [response (<! (client/post
                        (gstring/format "https://api.github.com/repos/%s/%s/labels" (-> request :ref :owner) (-> request :ref :repo))
                        {:body (json/->str {:name name
                                            :description description
                                            :color color})
                         :headers {"User-Agent" "atomist"
                                   "Authorization" (gstring/format "Bearer %s" (:token request))}}))]
      (if (not (= 200 (:status response)))
        (log/warnf "status %s - %s" (:status response) (-> response :body)))
      response)))

(defn patch-label [request {:keys [name description color]}]
  (go
    (try
      (let [response (<! (client/patch
                          (gstring/format "https://api.github.com/repos/%s/%s/labels/%s"
                                          (-> request :ref :owner) (-> request :ref :repo)
                                          name)
                          {:body (json/->str {:description description
                                              :color color})
                           :headers {"User-Agent" "atomist"
                                     "Authorization" (gstring/format "Bearer %s" (:token request))}}))]
        (if (= 200 (:status response))
          (-> response :body)
          (log/warnf "status %s - %s" (:status response) (-> response :body))))
      (catch :default ex
        (log/error "raised exception " ex)))))

(defn delete-label [request name]
  (go
    (try
      (let [response (<! (client/delete
                          (gstring/format "https://api.github.com/repos/%s/%s/labels/%s"
                                          (-> request :ref :owner) (-> request :ref :repo)
                                          name)
                          {:headers {"User-Agent" "atomist"
                                     "Authorization" (gstring/format "Bearer %s" (:token request))}}))]
        (if (= 204 (:status response))
          (-> response :body)
          (log/warn "no GitHub labels found")))
      (catch :default ex
        (log/error "raised exception " ex)))))

(defn put-label
  "POST /repos/:owner/:repo/pulls
    labels is a vector of strings"
  [{:keys [labels number] :as request}]
  (client/post (gstring/format "https://api.github.com/repos/%s/%s/issues/%s/labels"
                               (-> request :ref :owner) (-> request :ref :repo) number)
               {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                          "User-Agent" "atomist"}
                :body (json/->str {:labels labels})}))

(defn get-labels
  ""
  [{:keys [number] :as request}]
  (go
    (->
     (<! (client/get (gstring/format "https://api.github.com/repos/%s/%s/issues/%s/labels"
                                     (-> request :ref :owner) (-> request :ref :repo) number)
                     {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                                "User-Agent" "atomist"}}))
     :body)))

(defn rm-label
  ""
  [{:keys [number] :as request} label]
  (client/delete (gstring/format "https://api.github.com/repos/%s/%s/issues/%s/labels/%s"
                                 (-> request :ref :owner) (-> request :ref :repo) number label)
                 {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                            "User-Agent" "atomist"}}))

(defn patch-repo
  [request config]
  (client/patch (gstring/format "https://api.github.com/repos/%s/%s"
                                (-> request :ref :owner) (-> request :ref :repo))
                {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                           "User-Agent" "atomist"}
                 :body (json/->str config)}))

(defn repo-topics
  [request]
  (go
    (let [response
          (<! (client/get (gstring/format "https://api.github.com/repos/%s/%s/topics"
                                          (-> request :ref :owner) (-> request :ref :repo))
                          {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                                     "User-Agent" "atomist"
                                     "Accept" "application/vnd.github.mercy-preview+json"}}))]
      (if (= 200 (:status response))
        (-> response :body :names)
        (do
          (log/warnf "topics for %s/%s %s %s" (-> request :ref :owner) (-> request :ref :repo) (-> response :status) (-> response :body))
          [])))))

(defn content
  [request owner repo path]
  (go
    (let [response (<! (client/get (gstring/format "https://api.github.com/repos/%s/%s/contents/%s" owner repo path)
                                   {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                                              "User-Agent" "atomist"}}))]
      (if (and (= 200 (:status response))
               (= (-> response :body :type) "file"))
        (cond
          (= "base64" (-> response :body :encoding))
          {:content (-> response
                        :body
                        :content
                        (b64/decodeString))}
          :else
          {:error "file encoding only supports base64"})
        {:error "unsupported content"
         :status (:status response)
         :type (-> response :body :type)}))))

(comment
 (go (println (<! (content
                   {:token (.. js/process -env -GITHUB_TOKEN)}
                   "atomist-skills" "git-content-sync-skill" "src/atomist/main.cljs"))))
 (go (println (<! (create-issue
                   {:token (.. js/process -env -GITHUB_TOKEN)}
                   "atomist-skills" "git-content-sync-skill" {:title "test title" :body "body" :labels ["atomist:content-sync"]})))))

(defn create-issue
  [request owner repo {:keys [title body labels]}]
  (go
    (let [response (<! (client/post (gstring/format "https://api.github.com/repos/%s/%s/issues" owner repo)
                                    {:headers {"Authorization" (gstring/format "bearer %s" (:token request))
                                               "User-Agent" "atomist"}
                                     :body (json/->str {:title title
                                                        :body body
                                                        :labels labels})}))]
      response)))