(ns atomist.deps
  (:require [atomist.cljs-log :as log]
            [atomist.gitflows :as gitflow]
            [atomist.json :as json]
            [atomist.api :as api]
            [atomist.graphql :as graphql]
            [atomist.graphql-channels :as channels]
            [goog.string :as gstring]
            [goog.string.format]
            [cljs.core.async :refer [<! timeout] :as async])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn- get-param [x s]
  (->> x (filter #(= s (:name %))) first :value))

(defn- semvers? [fingerprint target]
  false)

(defn- semver<= [fingerprint target]
  true)

(defn- off-target?
  "true if fingerprints have same name but different sha
     although if both versions are semver compliant then the fingerprint must also be semantically
       earlier than the target.  If not, it's considered on target"
  [fingerprint target]
  (if
   (and (= (:name fingerprint) (:name target))
        (not (= (:sha fingerprint) (:sha target))))
    (if (semvers? fingerprint target)
      (semver<= fingerprint target)
      true)))

(defn- policy-order
  "when there are overlaps, use manual config overrides latest semver, which over-rides latest semver used"
  [v]
  (letfn [(is [x] (complement (fn [y] (= x y))))]
    (count (take-while (is v) ["latestSemVerUsed" "latestSemVerAvailable" "manualConfiguration"]))))

(defn- policy-type [configuration]
  (->> configuration :parameters (filter #(= (:name %) "policy")) first :value))

(defn- dependencies [configuration]
  (->> configuration :parameters (filter #(= (:name %) "dependencies")) first :value))

(defn- all-strings?
  [d]
  (let [data (try (cljs.reader/read-string d)
                  (catch :default _ (log/warnf "invalid:  %s" d) false))]
    (and data (coll? data) (every? string? data))))

(defn- deps-array?
  [d]
  (let [data (try (cljs.reader/read-string d) (catch :default _ (log/warnf "invalid:  %s" d)))]
    (and data (coll? data) (every? #(and (coll? %) (= 2 (count %)) (-> % second string?)) data))))

(defn- validate-policy
  "can validate latestSemVerAvailable and latestSemVerUsed uniformly as they are always string arrays
   manual configurations should also have been transformed to [[lib version]] by this point"
  [configuration]
  (case (policy-type configuration)
    "latestSemVerUsed" (if (all-strings? (dependencies configuration))
                         configuration
                         (assoc configuration :error "latestSemVerUsed dependencies must be an array of Strings"))
    "latestSemVerAvailable" (if (all-strings? (dependencies configuration))
                              configuration
                              (assoc configuration :error "latestSemVerAvailable dependencies must be an array of Strings"))
    "manualConfiguration" (if (deps-array? (dependencies configuration))
                            configuration
                            (assoc configuration :error "manualConfiguration dependencies configuration invalid"))
    configuration))

(defn- configs->policy-map
  "configuratios are {:keys [policy dependencies scope]}
     policy - one of latestSemVerUsed, latestSemVerAvailable, or manualConfiguration
     dependencies - edn string which must parse to either [lib1,lib2,...] or [[lib \"version\"],...]

     returns {library {:policy policy-type :version v}} where version is optional (might be added in later)"
  [configurations]
  (letfn [(read-dependency-string [c policy] (->> (cljs.reader/read-string (dependencies c))
                                                  (map str)
                                                  (reduce #(assoc %1 %2 {:policy policy}) {})))
          (read-manual-dependencies [c] (->> (cljs.reader/read-string (dependencies c))
                                             (reduce (fn [agg [k v]] (assoc agg (str k) {:policy :manualConfiguration
                                                                                         :version v})) {})))]
    (->> configurations
         (filter #(#{"latestSemVerUsed" "latestSemVerAvailable" "manualConfiguration"} (policy-type %)))
         (sort-by (comp policy-order policy-type))
         (map (fn [configuration] (case (policy-type configuration)
                                    "latestSemVerUsed" (read-dependency-string configuration :latestSemVerUsed)
                                    "latestSemVerAvailable" (read-dependency-string configuration :latestSemVerAvailable)
                                    "manualConfiguration" (read-manual-dependencies configuration))))
         (apply merge))))

(defn- get-latest [o type dependency]
  (go
    (try
      (-> (<! (channels/graphql->channel o graphql/query-latest-semver {:type type
                                                                        :name dependency}))
          :body
          :data
          :fingerprintAggregates)
      (catch :default ex
        (log/error "semver query " ex)))))

(defn- version-channel
  "channel to a plan element
    dictating the correct next version for a target dependency

    this assumes that dependency fingerprints use json encoded [library version] to represent their data
      - this is true for leiningen, and npm.  TODO However, maven fingerprints are not computed this way.

    policy-map is {library {:policy policy-type :version v}}
    fp is {:keys [name type data]}

    returns a plan element {:fingerprint current-fingerprint :target target-fingerprint}"
  [o policy-map fp ->library-version ->data ->sha ->name]
  (letfn [(transform-data [m] (assoc m :data (json/->obj (:data m))))]
    (go
      (let [dependency (-> fp :data ->library-version first)
            {:keys [policy] :as d} (policy-map dependency)
            plan-element {:target (case policy
                                    :latestSemVerAvailable (-> (<! (get-latest o (:type fp) (:name fp)))
                                                               :latestSemVerAvailable
                                                               (transform-data))
                                    :latestSemVerUsed (-> (<! (get-latest o (:type fp) (:name fp)))
                                                          :latestSemVerUsed
                                                          :fingerprint
                                                          (transform-data))
                                    :manualConfiguration (let [data (->data [dependency (:version d)])]
                                                           {:name (->name dependency)
                                                            :sha (->sha data)
                                                            :data data})
                                    {})
                          :fingerprint fp}]
        plan-element))))

(defn set-up-target-configuration
  "middleware to construct a manualConfiguration
     from a dependency configuration containing a '[lib version]' string"
  [handler]
  (fn [request]
    (go
      (log/infof "set up target dependency to converge on [%s]" (:dependency request))
      (<! (handler (assoc request
                          :configurations [{:parameters [{:name "policy"
                                                          :value "manualConfiguration"}
                                                         {:name "dependencies"
                                                          :value (gstring/format "[%s]" (:dependency request))}]}]))))))

(defn mw-validate-policy
  "middleware to validate an edn deps policy
    all configurations with a policy=manualConfiguration should have a dependency which is an application/json map
    all configurations with other policies use a dependency which is an array of strings"
  [handler]
  (fn [request]
    (go
      (try
        (let [configurations (->> (:configurations request)
                                  (map validate-policy))]
          (if (->> configurations
                   (filter :error)
                   (empty?))
            (<! (handler request))
            (<! (api/finish request :failure (->> configurations
                                                  (map :error)
                                                  (interpose ",")
                                                  (apply str))))))
        (catch :default ex
          (log/error ex)
          (<! (api/finish request :failure (-> (ex-data ex) :message))))))))

(defn apply-policy-targets
  "returns a channel which will emit a :complete value after applying all version policies to a Repo
     Steps are roughly
       - use configuration to construct the Policy to apply
       - emit plan elements from a version-channel (which may fetch live latest data from the graph)
       - check whether any current fingerprint are off target
       - apply plan elements that represent off-target deps inside of a PR editor

     params
       apply-target-fingerprint (project,target-fingerprint) => channel emitting logged message
   "
  [{:deps/keys [type apply-library-editor ->library-version ->sha ->data ->name]}
   {:keys [project configurations fingerprints] :as request}]

  ;; NOTE: at this point, all configurations have been transformed to [[lib version] ...] or [lib ...]
  ;; because we only care about the
  (go
    (let [policy-map (configs->policy-map configurations)
          plan (<! (->> (for [{_ :data :as fingerprint} fingerprints]
                          (version-channel
                           request
                           policy-map
                           fingerprint
                           ->library-version
                           ->data
                           ->sha
                           ->name))
                        (async/merge)
                        (async/reduce (fn [plan {:keys [fingerprint target] :as off-target}]
                                        (if (and target (off-target? fingerprint target))
                                          (conj plan off-target)
                                          plan)) [])))]
      (log/info "plan " plan)
      (let [body (->> plan
                      (map (fn [{{current-data :data} :fingerprint
                                 {target-data :data} :target :as p}]
                             (let [[current-library current-version] (->library-version current-data)
                                   [target-library target-version] (->library-version target-data)]
                               (gstring/format "off-target %s %s/%s -> %s/%s"
                                               (-> p :fingerprint :name)
                                               current-library current-version
                                               target-library target-version))))
                      (interpose "\n")
                      (apply str))
            pr-opts {:branch type
                     :target-branch (-> request :ref :branch)
                     :title (gstring/format "%s skill requesting dependency change" (-> request :skill :name))
                     :body (gstring/format "%s\n[atomist:edited]" body)}]
        ;; tries to send data from the target fingerprint to the editor
        (<! ((gitflow/with-pr
               (fn [p]
                 (go
                   (doseq [{:keys [target]} plan]
                     (log/info "applying " target " : " (<! (apply-library-editor p target))))
                   :done))
               pr-opts) project))))
    :complete))

(defn- pre-clone-status-message
  [handler]
  (fn [request]
    (go
      (api/status "pre-clone-status-message")
      (<! (api/simple-message request (gstring/format "```... syncing npm dependencies in %s/%s```"
                                                      (-> request :ref :owner)
                                                      (-> request :ref :repo))))
      (<! (api/success-status request "kick off npm sync" nil))
      (<! (handler request)))))

(defn- post-clone-status-message
  [handler]
  (fn [request]
    (go
      (api/status "post-clone-status-message")
      (<! (api/simple-message request (gstring/format "```complete sync on %s/%s complete```"
                                                      (-> request :ref :owner)
                                                      (-> request :ref :repo))))
      (<! (handler request)))))

(defn- skip-non-master [handler]
  (fn [request]
    (go
      (if (= "master" (-> request :ref :branch))
        (<! (handler request))
        (<! (api/finish request :success "skipping non-master Pushes"))))))

(defn- handle-push-event [request compute-fingerprints mw-validate-policy]
  ((-> (api/finished :message "handling Push" :success "successfully handled Push event")
       (api/send-fingerprints)
       (api/from-channel compute-fingerprints :key :results)
       (api/clone-ref)
       (mw-validate-policy)
       (api/extract-github-token)
       (skip-non-master)
       (api/create-ref-from-event)
       (api/status)) request))

(defn- fp-command-handler [request just-fingerprints]
  ((-> (api/finished :success "handling extraction CommandHandler" :message "handling extraction CommandHandler")
       (api/show-results-in-slack :result-type "fingerprints")
       (api/from-channel just-fingerprints :key :results)
       (api/clone-ref)
       (api/create-ref-from-first-linked-repo)
       (api/user-should-choose-one-linked-repo)
       (api/extract-linked-repos)
       (api/extract-github-user-token)
       (api/set-message-id)
       (api/status)) (assoc request :branch "master")))

(defn- sync-command-handler [request compute-fingerprints mw-validate-policy]
  ((-> (api/finished :success "handling sync CommandHandler" :message "handling sync CommandHandler")
       (post-clone-status-message)
       (api/from-channel compute-fingerprints :key :results)
       (api/clone-ref)
       (pre-clone-status-message)
       (api/create-ref-from-first-linked-repo)
       (api/user-should-choose-one-linked-repo)
       (api/extract-cli-parameters [[nil "--slug SLUG" "org/repo"]])
       (api/extract-linked-repos)
       (mw-validate-policy)
       (api/extract-github-user-token)
       (api/set-message-id)
       (api/status)) (assoc request :branch "master")))

(defn- update-command-handler [request compute-fingerprints validate-parameters]
  ((-> (api/finished :success "handling update CommandHandler" :message "handling update CommandHandler")
       (api/show-results-in-slack :result-type "fingerprints")
       (api/from-channel compute-fingerprints :key :results)
       (api/clone-ref)
       (pre-clone-status-message)
       (api/create-ref-from-first-linked-repo)
       (api/user-should-choose-one-linked-repo)
       (api/extract-cli-parameters [[nil "--slug SLUG" "org/repo"]])
       (api/extract-linked-repos)
       (api/extract-github-user-token)
       (validate-parameters)
       (api/set-message-id)
       (api/status)) (assoc request :branch "master")))

(defn deps-handler
  " - VIEW COMMAND setup a command to compute and show fingerprints in slack
      just-fingerprints runs on a cloned repo and just shows data in Slack - does not send fingerprints
    - SYNC COMMAND using configuration policy - setup a sync command
      compute-fingerprints will compute and send fingerprints,
        and mw-validate-policy will ensure configuration is in a good state
    - UPDATE COMMAND using cli params (does not apply cofiguration policy)
      compute-fingerprints will compute and send fingerprints
        and validate-parameters will check the command line parameters (this does not apply the policy in the configuration
          but instead sets up a new one based on the parameters
    - finally, set up a Push event handler with
      compute-fingerprints will compute and send fingerprints
        and mw-validate-policy will ensure configuration is in a good state"
  [data
   sendreponse
   & {:deps-command/keys [show sync update]
      :deps/keys [extract validate-policy validate-command-parameters] :as contract}]
  (letfn [(compute-fingerprints [request]
            (go
              (try
                (let [fingerprints (extract (:project request))]
                  ;; first create PRs for any off target deps
                  (<! (apply-policy-targets contract (assoc request :fingerprints fingerprints)))
                  ;; return the fingerprints in a form that they can be added to the graph
                  fingerprints)
                (catch :default ex
                  (log/error "unable to compute deps.edn fingerprints")
                  (log/error ex)
                  {:error ex
                   :message "unable to compute deps.edn fingerprints"}))))
          (extract-fingerprints [request]
            (go
              (try
                (let [fingerprints (extract (:project request))]
                  ;; return the fingerprints in a form that they can be added to the graph
                  fingerprints)
                (catch :default ex
                  (log/error "unable to compute deps.edn fingerprints")
                  (log/error ex)
                  {:error ex
                   :message "unable to compute deps.edn fingerprints"}))))]
    (api/make-request
     data
     sendreponse
     (fn [request]
       (cond
         ;; handle Push events
         (contains? (:data request) :Push)
         (handle-push-event request compute-fingerprints validate-policy)

         (= sync (:command request))
         (sync-command-handler request compute-fingerprints validate-policy)

         (= show (:command request))
         (fp-command-handler request extract-fingerprints)

         (= update (:command request))
         (update-command-handler request compute-fingerprints validate-command-parameters)

         :else
         (api/finish request :success "unrecognized" :visibility :hidden))))))
