(ns atomist.github
  (:require [http.client :as client]
            [goog.string :as gstring]
            [goog.string.format]
            [atomist.cljs-log :as log]
            [cljs.core.async :refer [<!]]
            [atomist.json :as json])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn github-v4 [token query variables]
  (go
    (let [response (<! (http.client/post
                        "https://api.github.com/graphql"
                        {:headers {"Authorization" (gstring/format "bearer %s" token)
                                   "Accept" "application/vnd.github.bane-preview+json"
                                   "User-Agent" "atomist"}
                         :body (json/->str {:query query
                                            :variables variables})}))]
      (log/info "response " response)
      (if (= 200 (:status response))
        (:body response)
        (log/error "status was not 200")))))

(defn pr-channel [request branch-name]
  (go
    (try
      (let [response (<! (client/get
                          (gstring/format "https://api.github.com/repos/%s/%s/pulls" (-> request :ref :owner) (-> request :ref :repo))
                          {:query-params {:state "open"
                                          :head (gstring/format "%s:%s" (-> request :ref :owner) branch-name)}
                           :headers {"User-Agent" "atomist"
                                     "Authorization" (gstring/format "Bearer %s" (:token request))}}))]
        (if (= 200 (:status response))
          (-> response :body first)
          (log/warn "no PR found")))
      (catch :default ex
        (log/error "raised exception " ex)))))

;; TODO handle paging
(defn all-labels-channel [request]
  (go
    (try
      (let [response (<! (client/get
                          (gstring/format "https://api.github.com/repos/%s/%s/labels" (-> request :ref :owner) (-> request :ref :repo))
                          {:headers {"User-Agent" "atomist"
                                     "Authorization" (gstring/format "Bearer %s" (:token request))}}))]
        (if (= 200 (:status response))
          (-> response :body)
          (log/warn "no GitHub labels found")))
      (catch :default ex
        (log/error "raised exception " ex)))))

(defn add-label [request {:keys [name description color]}]
  (go
    (try
      (let [response (<! (client/post
                          (gstring/format "https://api.github.com/repos/%s/%s/labels" (-> request :ref :owner) (-> request :ref :repo))
                          {:body (json/->str {:name name
                                              :description description
                                              :color color})
                           :headers {"User-Agent" "atomist"
                                     "Authorization" (gstring/format "Bearer %s" (:token request))}}))]
        (if (= 200 (:status response))
          (-> response :body)
          (log/warnf "status %s - %s" (:status response) (-> response :body))))
      (catch :default ex
        (log/error "raised exception " ex)))))

(defn patch-label [request {:keys [name description color]}]
  (go
    (try
      (let [response (<! (client/patch
                          (gstring/format "https://api.github.com/repos/%s/%s/labels/%s"
                                          (-> request :ref :owner) (-> request :ref :repo)
                                          name)
                          {:body (json/->str {:description description
                                              :color color})
                           :headers {"User-Agent" "atomist"
                                     "Authorization" (gstring/format "Bearer %s" (:token request))}}))]
        (if (= 200 (:status response))
          (-> response :body)
          (log/warnf "status %s - %s" (:status response) (-> response :body))))
      (catch :default ex
        (log/error "raised exception " ex)))))

(defn delete-label [request name]
  (go
    (try
      (let [response (<! (client/delete
                          (gstring/format "https://api.github.com/repos/%s/%s/labels/%s"
                                          (-> request :ref :owner) (-> request :ref :repo)
                                          name)
                          {:headers {"User-Agent" "atomist"
                                     "Authorization" (gstring/format "Bearer %s" (:token request))}}))]
        (if (= 204 (:status response))
          (-> response :body)
          (log/warn "no GitHub labels found")))
      (catch :default ex
        (log/error "raised exception " ex)))))
