(ns atomist.graphql-channels
  (:require [atomist.json :as json]
            [atomist.graphql :as graphql]
            [cljs.core.async :refer [>! <! timeout chan]]
            [cljs.pprint :refer [pprint]]
            [http.client :as client]
            [atomist.promise :as promise]
            [atomist.cljs-log :as log]
            [com.rpl.specter :as specter]
            [atomist.time]
            [goog.string :as gstring]
            [goog.string.format]
            [clojure.tools.cli :refer [parse-opts]]
            [clojure.core.async :as async]
            [clojure.string :as s]
            [atomist.time :as time])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn -js->clj+
  "For cases when built-in js->clj doesn't work. Source: https://stackoverflow.com/a/32583549/4839573"
  [x]
  (into {} (for [k (js-keys x)]
             [k (aget x k)])))

(defn env
  "Returns current env vars as a Clojure map."
  []
  (-js->clj+ (.-env js/process)))

(def graphql-endpoint (or (get (env) "GRAPHQL_ENDPOINT") "https://automation.atomist.com/graphql"))

(defn graphql->channel
  "requires an atomist://api-key, and a :team :id in the request

     returns the HTTP response for this team graphql query"
  [o query variables]
  (let [url (gstring/format "%s/team/%s" graphql-endpoint (or (-> o :team :id) (-> o :extensions :team_id)))
        auth-header (gstring/format "Bearer %s" (->> o :secrets (filter #(= "atomist://api-key" (:uri %))) first :value))]
    (if (and url auth-header)
      (client/post
       url
       {:headers {"Authorization" auth-header}
        :body (-> {:query query
                   :variables variables}
                  (json/->str))})
      (throw (ex-info "unable to extract url and token from request" {:request o
                                                                      :query query})))))

(defn head-commits->channel
  "  requires that the team id is already in the request
     returns channel emitting coll of {:keys [analysis repo]}"
  [request type name]
  (go
   (let [response (<! (graphql->channel request graphql/head-commits {:type type :name name}))]
     (if-let [head-commits (-> response
                               :body
                               :data
                               :headCommitsWithFingerprint)]
       head-commits
       (log/warnf "no head-commits for %s %s" type name)))))

(defn github-app-installation-token->channel [request owner]
  (go
   (-> (<! (graphql->channel request graphql/githubAppInstallationByOwner {:name owner}))
       :body
       :data
       :GitHubAppInstallation
       first
       :token
       :secret)))

(defn repo-refs->channel
  [request]
  (let [c (chan)]
    (go
     (let [response (<! (graphql->channel request graphql/repos {}))]
       (doseq [org (-> response :body :data :Org)]
         (let [token (or (-> org :scmProvider :credential :secret) (<! (github-app-installation-token->channel request (:owner org))))]
           (doseq [repo (:repos org)]
             (>! c (assoc request :token token
                                  :ref {:owner (:owner org)
                                        :repo (:name repo)}))))))
     (log/info "close repo-refs channel")
     (async/close! c))
    c))
