(ns atomist.sdmprojectmodel
  (:require ["@atomist/automation-client" :as ac]
            ["@atomist/automation-client/lib/operations/support/editorUtils" :as editor-utils]
            [cljs.core.async :refer [<! timeout] :as async]
            [atomist.promise :as promise]
            [atomist.cljs-log :as log]
            [goog.string :as gstring]
            [goog.string.format]
            [cljs-node-io.proc :as proc]
            [cljs-node-io.core :refer [slurp]]
            [clojure.string :as s])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(def x (. ac -PlainLogging))
(set! (.. x -console -level) "debug")

(defn enable-sdm-debug-logging []
  ((. ac -configureLogging) x))

(defn do-with-shallow-cloned-project
  "does a shallow clone of the ref in some tmp directory

    returns chan that will emit the value returned by the async project-callback"
  [project-callback token ref]
  (go
   (let [^js p (<! (promise/from-promise
                    (.cloned (.-GitCommandGitProject ac)
                             #js {:token token}
                             (.from (.-GitHubRepoRef ac) (clj->js ref))
                             #js {:alwaysDeep false})))]
     (if (= "HEAD" (.. ^js p -id -sha))
       (set! (.-sha ^js (.-id ^js p)) (:sha (<! (promise/from-promise ((. ^js p -gitStatus)))))))
     (<! (promise/from-promise ((. ^js p -setUserConfig) "atomist-bot" "bot@atomist.com")))
     (<! (project-callback p)))))

(defn commit-then-push
  "  middleware for project handling - wrap project-callback in commit, push
     project-callback returns chan that will emit either :done or :failure"
  [project-callback commit-message]
  (fn [p]
    (go
     (<! (project-callback ^js p))
     ;; TODO
     #_(<! (promise/from-promise (.call (.-createAndPushBranch editor-utils) p (clj->js {:branch "master"}))))
     (<! (promise/from-promise (.commit ^js p commit-message)))
     (<! (promise/from-promise (.push ^js p))))))

(defn remote?->chan
  [project branch]
  (go
   (let [[err stdout stderr] (<! (proc/aexec "git ls-remote --heads" {:cwd (. ^js project -baseDir)}))]
     (not (empty?
           (->> (s/split-lines (if (string? stdout) stdout (slurp stdout)))
                (map #(second (re-find #".*refs/heads/(.*)$" %)))
                (filter #(= branch %))))))))

(defn checkout-branch->chan
  "fetch a branch with depth 1, update the ref head, and then move the working copy to point at this branch
    this version not rely on any remote tracking branches, and can only push - not set up to pull, or merge"
  [project branch]
  (let [opts {:cwd (. ^js project -baseDir)}]
    (letfn [(handle-errors [[error _ stderr]]
              (if error
                (throw (ex-info stderr {:error error}))))]
      (go
       (handle-errors (<! (proc/aexec (gstring/format "git fetch origin %s:%s --depth 1" branch branch) opts)))
       (handle-errors (<! (proc/aexec (gstring/format "git checkout %s" branch) opts)))
       :done))))

(defn- hasBranch?
  "the sdm hasBranch can only check for local branches.  This function also checks for remote refs.
    returns channel that emits boolean"
  [project branch-name]
  (go
   (if
    (<! (promise/from-promise
         (.hasBranch ^js project branch-name)
         identity
         (fn [error] (throw (ex-info "sdm failure checking branch" {:failure error})))))
     true
     (<! (remote?->chan project branch-name)))))

(defn edit-inside-PR
  "generate a function to execute a project callback wrapped by an SDM PullRequest
    project-callback is (p) => channel emitting :done if successful

    - outer project callback returns a chan that emits :done or :failure

    - most of these Project api calls return the this if successful (Project can be mutated)
    - since these api calls wrap git cli commands, the Promises can fail if git cli command
      returns non-zero at any point"
  [project-callback {:keys [branch target-branch body title]}]
  (fn [p]
    (letfn [(handle-sdm-failure [{:keys [done failure]}]
              (if failure
                (throw (ex-info "sdm project operation failure" {:failure failure}))
                {:result done}))
            (promise->channel [promise] (promise/from-promise
                                         promise
                                         (fn [v] {:done v})
                                         (fn [error] {:failure error})))]
      (go
       (try
         (let [response (<! (hasBranch? p branch))]
           (if (not response)
             (handle-sdm-failure (<! (promise->channel (.createBranch ^js p branch))))
             (do
               (<! (checkout-branch->chan p branch))
               (handle-sdm-failure (<! (promise->channel (.checkout ^js p branch)))))))
         ;; by now project should represent a local git copy with HEAD ref pointed at branch
         (if (= :done (<! (project-callback p)))
           (do
             (handle-sdm-failure (<! (promise->channel (.commit ^js p "atomist commit message"))))
             (handle-sdm-failure (<! (promise->channel (.push ^js p))))
             ;; idempotent
             (handle-sdm-failure (<! (promise->channel (.raisePullRequest ^js p title body target-branch))))
             :done)
           (throw (ex-info "SDM project-callback did not finish normally" {:project p})))
         (catch :default ex
           (log/error "------")
           (log/error ex (ex-data ex))
           (log/error "------")
           :failure))))))

(defn commit-then-PR
  "middleware for project handling - wrap project-callback in commit, push
     project-callback returns chan that will emit either :done or :failure
     params
       opts - {:keys [branch targetBranch body title commit-message]}"
  [project-callback {:keys [branch target-branch body title]}]
  (fn [p]
    (log/debugf "commit then PR %s -> %s" target-branch branch)
    (go
     (<! (project-callback p))
     (let [body (gstring/format "%s\n[auto-merge-method:merge] [auto-merge:on-approve]" body)]
       (log/debug "create branch " branch)
       (<! (promise/from-promise (.createBranch ^js p branch)))
       (let [commit-result (<! (promise/from-promise (.commit ^js p body)))]
         (log/debug "commit-result " commit-result)
         (if (not (contains? commit-result :failure))
           (do
             (<! (promise/from-promise (.push ^js p)))
             (log/debugf "create PR titled %s for branch %s -> %s" title branch target-branch)
             (log/debugf "raise PR (%s)" (js->clj (<! (promise/from-promise (.raisePullRequest ^js p title body target-branch)))))
             {:message "created PR"})
           (do
             (log/warn "finished without anything to commit")
             {:message "nothing to commit"})))))))

(defn do-with-files
  "middleware for project handling - wrap project-callback in file iterator
     project-callback returns chan with an untyped value"
  [file-callback & patterns]
  (fn [p]
    (go
     (let [data (atom [])
           callbacks-over-all-patterns
           (<! (async/reduce
                conj []
                (async/merge
                 (for [pattern patterns]
                   (promise/from-promise
                    (.doWithFiles
                     (.-projectUtils ac)
                     p
                     pattern
                     (fn [f]
                       (log/debug (goog.string/format "Processing file: %s" (.-path f)))
                       (promise/chan->promise (go
                                               (let [d (<! (file-callback f))]
                                                 (swap! data conj d)
                                                 d))))))))))]
       (if @data @data :done)))))

(defn get-content
  " returns chan with file content String or {:failure error}"
  [f]
  (promise/from-promise (.getContent ^js f)))

(defn set-content
  " returns chan with the updated File #js object"
  [f content]
  (promise/from-promise (.setContent ^js f content)))
