(ns atomist.api
  (:require [atomist.json :as json]
            [atomist.graphql :as graphql]
            [atomist.json :as json]
            [atomist.meta :as meta]
            [cljs.core.async :refer [>! <! timeout chan]]
            [cljs.pprint :refer [pprint]]
            [http.client :as client]
            [goog.crypt.base64 :as b64]
            [atomist.promise :as promise]
            [cljs-node-io.core :as io :refer [slurp spit]]
            [atomist.cljs-log :as log]
            [com.rpl.specter :as specter]
            [atomist.time]
            [goog.string :as gstring]
            [goog.string.format]
            [atomist.slack :as slack]
            [clojure.tools.cli :refer [parse-opts]]
            [atomist.sdmprojectmodel :as sdm]
            [clojure.core.async :as async]
            [clojure.string :as s]
            [atomist.time :as time])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn- send-on-response-channel [o]
  (let [callback (chan)]
    (go
     (try
       (log/info "send-on-response-channel " o " with promise " (<! (promise/from-promise ((:sendreponse o) (clj->js o)))))
       (>! callback {:success true})
       (catch :default ex
         (>! callback {:error "unable to send"
                       :ex ex}))))
    callback))

(defn- default-destination [o]
  (if (or (not (:destinations o)) (empty? (:destinations o)))
    (-> o
        (update :destinations (constantly [(merge
                                            (:source o)
                                            {:user_agent "slack"})]))
        (update-in [:destinations 0] (fn [destination]
                                       (if (some? (:slack destination))
                                         (update destination :slack dissoc :user)
                                         destination))))
    o))

(defn ^:api success-status
  "on command request, send status that the invocation was successful"
  ([o status-message]
   (-> (select-keys o [:sendreponse :correlation_id :api_version :automation :team :command :source :destinations])
       (assoc :status {:code 0 :reason status-message})
       (assoc :content_type "application/x-atomist-status+json")
       (default-destination)
       (send-on-response-channel)))
  ([o]
   (success-status o "success")))

(defn ^:api failed-status
  "on command request, send status that the invocation failed"
  ([o status-message]
   (-> (select-keys o [:sendreponse :correlation_id :api_version :automation :team :command :source :destinations])
       (assoc :status {:code 1 :reason status-message})
       (assoc :content_type "application/x-atomist-status+json")
       (default-destination)
       (send-on-response-channel)))
  ([o]
   (failed-status o "failure")))

(declare send-status)

(defn ^:api finish [request & {:keys [failure success]}]
  (go
   (<!
    (if failure
      (failed-status request failure)
      (success-status request (or success "success"))))
   (>! (:done-channel request) (if failure :failed :done))))

(defn ^:api finished [& {:keys [message success send-status] :or {send-status (constantly "success")}}]
  (fn [request]
    (log/infof "----> finished %s" (or message ""))
    (finish request :success (or success (send-status request)))))

(defn ^:api snippet-message
  "send snippet as bot
    params
      o           - command request or event
      content-str - content as string
      filetype    - valid slack filetype
      title       - string title"
  [o content-str filetype title]
  (-> (select-keys o [:sendreponse :correlation_id :api_version :automation :team :source :command :destinations :id])
      (assoc :content_type "application/x-atomist-slack-file+json")
      (assoc :body (json/clj->json {:content content-str :filetype filetype :title title}))
      (default-destination)
      (send-on-response-channel)))

(defn ^:api simple-message
  "send simple message as bot
     params
       o - command request or event
       s - string message"
  [o s]
  (-> (select-keys o [:sendreponse :correlation_id :api_version :automation :team :source :command :destinations :id])
      (assoc :content_type "text/plain")
      (assoc :body s)
      (assoc :timestamp (atomist.time/now))
      (default-destination)
      (send-on-response-channel)))

(defn ^:api continue
  "continue this request with some additional parameter specs
     (all added parameters must be required)"
  [o params]
  (-> (select-keys o [:sendreponse :correlation_id :parameters :api_version :automation :team :source :command :destinations :parameter_specs :id])
      (assoc :content_type "application/x-atomist-continuation+json")
      (update :parameter_specs (fnil concat []) (->> params (map #(assoc % :required true)) (into [])))
      (default-destination)
      (send-on-response-channel)))

(defn ^:api get-secret-value
  [o secret-uri]
  (->> o :secrets (filter #(= secret-uri (:uri %))) first :value))

(defn ^:api get-parameter-value
  "search command request for parameter
     params
       o              - command request
       parameter-name - string name
     returns nil if there's no parameter"
  [o parameter-name]
  (some->> (get-in o [:parameters])
           (filter #(= parameter-name (:name %)))
           first
           :value))

(defn ^:api mapped-parameter-value
  "search command request for mapped parameter
    params
      o              - command request
      parameter-name - string name
    returns nil if there's no parameter"
  [o parameter-name]
  (some->> (get-in o [:mapped_parameters])
           (filter #(= parameter-name (:name %)))
           first
           :value))

(defn ^:api delete-message
  [o]
  (-> (select-keys o [:sendreponse :correlation_id :api_version :automation :team :source :command :destinations :id])
      (assoc :content_type "application/x-atomist-delete")
      (assoc :timestamp (.getTime (js/Date.)))
      (default-destination)
      (send-on-response-channel)))

(defn ^:api ingest
  "ingest a new custom event
     params
       o         - incoming event or command request
       x         - custom event
       channel   - name of custom event channel"
  [o x channel]
  (-> o
      (select-keys [:sendreponse :api_version :correlation_id :team :automation])
      (assoc :content_type "application/json"
             :body (json/->str x)
             :destinations [{:user_agent "ingester"
                             :ingester {:root_type channel}}])
      (send-on-response-channel)))

(defn event->request
  [request]
  (if (:extensions request)
    (log/infof "handling event %s for %s/%s - %s" (->> request :data str (take 40) (apply str)) (-> request :extensions :team_id) (-> request :extensions :team_name) (-> request :extensions :correlation_id)))
  (cond-> request
          (:extensions request) (assoc :api_version "1"
                                       :correlation_id (-> request :extensions :correlation_id)
                                       :team {:id (-> request :extensions :team_id)
                                              :name (-> request :extensions :team_name)})))

(defn -js->clj+
  "For cases when built-in js->clj doesn't work. Source: https://stackoverflow.com/a/32583549/4839573"
  [x]
  (into {} (for [k (js-keys x)]
             [k (aget x k)])))

(defn env
  "Returns current env vars as a Clojure map."
  []
  (-js->clj+ (.-env js/process)))


(def graphql-endpoint (or (get (env) "GRAPHQL_ENDPOINT") "https://automation.atomist.com/graphql"))

(defn graphql->channel
  "requires an atomist://api-key, and a :team :id in the request

     returns the HTTP response for this team graphql query"
  [o query variables]
  (client/post
   (goog.string/format "%s/team/%s" graphql-endpoint (or (-> o :team :id) (-> o :extensions :team_id)))
   {:headers {"Authorization" (goog.string/format "Bearer %s" (->> o :secrets (filter #(= "atomist://api-key" (:uri %))) first :value))}
    :body (-> {:query query
               :variables variables}
              (json/->str))}))

(defn ^:api graphql
  "requires an atomist://api-key, and a :team :id in the request

     returns the HTTP response for this team graphql query"
  [o query variables]
  (client/post
   (goog.string/format "%s/team/%s" graphql-endpoint (or (-> o :team :id) (-> o :extensions :team_id)))
   {:headers {"Authorization" (goog.string/format "Bearer %s" (->> o :secrets (filter #(= "atomist://api-key" (:uri %))) first :value))}
    :body (-> {:query query
               :variables variables}
              (json/->str))}))

(defn- add-ids-to-commands [slack message-id]
  (let [num (atom 0)]
    (specter/transform [:attachments specter/ALL :actions specter/ALL]
                       #(if (:atomist/command %)
                          (-> %
                              (assoc-in [:atomist/command :id]
                                        (str (get-in % [:atomist/command :command])
                                             "-"
                                             (swap! num inc)))
                              (update-in [:atomist/command :parameters] (fnil conj []) {:name "messageId" :value message-id})
                              (assoc-in [:atomist/command :automation]
                                        {:name "@atomisthq/stupendabot-command-handler-cljs"
                                         :version "0.1.9"}))
                          %)
                       slack)))

(defn- transform-to-slack-actions [slack]
  (specter/transform [:attachments specter/ALL :actions specter/ALL]
                     #(if (:atomist/command %)
                        (let [action-id (get-in % [:atomist/command :id])]
                          (case (:type %)
                            "button"
                            (-> %
                                (dissoc :atomist/command)
                                (assoc :name (str "automation-command::" action-id))
                                (assoc :value action-id))
                            "select"
                            (-> %
                                (dissoc :atomist/command)
                                (assoc :name (str "automation-command::" action-id)))
                            %))
                        %)
                     slack))

(defn ^:api actionable-message
  "  params
       o       - incoming command request or event payload
       slack   - slack Message data where all actions may refer to
                 other CommandHandlers"
  [o slack & [opts]]

  (let [commands-with-ids (add-ids-to-commands slack (or (:id o) (str (random-uuid))))]

    (-> (select-keys o [:sendreponse :correlation_id :api_version :automation :team :source :command :destinations :id :post_mode])

        (merge opts)
        (assoc :content_type "application/x-atomist-slack+json")
        (assoc :timestamp (atomist.time/now))
        (assoc :body (-> commands-with-ids
                         (transform-to-slack-actions)
                         (json/->str)))
        (assoc :actions (->> (:attachments commands-with-ids)
                             (mapcat :actions)
                             (filter :atomist/command)
                             (mapv :atomist/command)))
        (default-destination)
        (send-on-response-channel))))

(defn ^:api block-message
  "  params
       blocks - array of Slack blocks"
  [o blocks]
  (-> (select-keys o [:sendreponse :correlation_id :api_version :automation :team :source :command :destinations :id :post_mode])

      (assoc :content_type "application/x-atomist-slack+json")
      (assoc :timestamp (atomist.time/now))
      (assoc :body (-> {:text "fallback"
                        :blocks blocks}
                       (json/->str)))
      (default-destination)
      (send-on-response-channel)))

(defn linked-person->channel
  "query for all persons linked to a chatUserId, but filter for only Persons representing
     scmIds for a resource provider with provider-id
   channel returns
     nil when there's no linked person
     OR Person map

   if you want to extract a credential from a person use (-> person :scmId :credential :secret)"
  [ch-request chat-id]
  (go
   (let [response (<! (graphql->channel ch-request graphql/linkedScmId {:chatUserId chat-id}))]
     (if-let [persons (-> response
                          :body
                          :data
                          :ChatId)]
       (->> persons
            (filter (fn [person] (= "github_com" (-> person :person :scmId :provider :providerType))))
            first
            :person)
       (log/warnf "no persons for chatUserId %s: %s" chat-id (:status response))))))

(defn github-provider->channel [ch-request]
  (go
   (if-let [providers (-> (<! (graphql->channel ch-request graphql/scmProviders {}))
                          :body
                          :data
                          :SCMProvider)]
     (->> providers
          (filter (fn [provider] (= "github_com" (-> provider :providerType))))
          first))))

(defn linked-repos->channel
  "query for all repos linked to this channel
   channel returns
     nil when there's no linked repo
     OR an array of linked Repos"
  [ch-request channel-id]
  (go
   (if-let [linked-repos (-> (<! (graphql->channel ch-request graphql/linkedRepos {:channelId channel-id}))
                             :body
                             :data
                             :ChatChannel
                             first
                             :repos)]
     linked-repos
     (log/warnf "no linked repos for channelId %s" channel-id))))

(defn repo-query->channel [ch-request repo-name]
  (go
   (let [response (<! (graphql->channel ch-request graphql/repo {:name repo-name}))]
     (log/info "graphql query response:  " response)
     (-> response
         :body
         :data
         :Repo
         first))))

(defn validate-repo-in-graph [handler repo-name-fn]
  (fn [ch-request]
    (go
     (if-let [repo-name (repo-name-fn ch-request)]
       (let [repo (<! (repo-query->channel ch-request repo-name))]
         (if repo
           (handler (assoc ch-request :repo repo))
           (let [message (gstring/format "we were unable to find the repo %s" (first repo))]
             (<! (simple-message ch-request message))
             (finish ch-request :failure message))))
       (let [message (gstring/format "this request did not contain a repo name")]
         (<! (simple-message ch-request message))
         (finish ch-request :failure message))))))

(defn check-required-parameters
  "request middleware
     used during command handler processing to check that required parameters are present

   - if the request is a command handler request and has missing parameters then the bot will go back to the user
     to get more data
   - if the required parameters are all present then the pipeline will continue with the parameter values keywordized
     and merged into the request"
  [handler & parameters]
  (fn [ch-request]
    (let [missing (->> parameters (filter (complement #(or (get-parameter-value ch-request (:name %))
                                                           (contains? ch-request (keyword (:name %)))))))]
      (if (empty? missing)
        (handler (merge
                  ch-request
                  (reduce
                   (fn [agg p] (if-let [v (get-parameter-value ch-request (:name p))]
                                 (assoc agg (keyword (:name p)) v)
                                 agg))
                   {}
                   parameters)))
        (go
         (if-let [p (and
                     (= 1 (count missing))
                     (#{"single" "multiple"} (-> missing first :type :kind))
                     (first missing))]

           (<! (actionable-message ch-request {:attachments
                                               [{:callback_id "callbackid"
                                                 :text (gstring/format "Please Enter a value for parameter %s" (:name p))
                                                 :markdwn_in ["text"]
                                                 :actions [{:text (gstring/format "select %s" (:name p))
                                                            :type "select"
                                                            :name "rug"
                                                            :options (->> (-> p :type :options)
                                                                          (map #(assoc {} :text (:description %) :value (:value %)))
                                                                          (into []))
                                                            :atomist/command {:command (:command ch-request)
                                                                              :parameter_name (:name p)
                                                                              :parameters []}}]}]}))
           (<! (continue ch-request missing)))
         (finish ch-request :success "asking bot for additional parameters"))))))

(defn extract-cli-parameters
  "request middleware
    used during command handler processing to parse the raw bot message into cli parameters.  The parameters
    are merged into the request keywordized.

    middleware-params
    options - must contain a clojure.tools.cli option parser definition"
  [handler options]
  (fn [ch-request]
    (let [{:keys [options]} (parse-opts (-> ch-request :raw_message (clojure.string/split #" ")) options)]
      (if options
        (handler (merge ch-request options))
        (handler ch-request)))))

(defn extract-github-user-token
  "request middleware
     used during command handler processing to extract a github user token from a request

     - if this middleware finds that the chat user is linked to an authorized ScmProvider then the token will be added
         to the request
     - if the current user is not authorized then the bot will ask the user authorize and then stop the pipeline with status success."
  [handler]
  (fn [ch-request]
    (go
     (let [linked-person (<! (linked-person->channel ch-request (-> ch-request :source :slack :user :id)))]
       (if linked-person
         (handler (assoc ch-request :token (-> linked-person :scmId :credential :secret)))
         (let [github-provider (<! (github-provider->channel ch-request))]
           (if github-provider
             (<! (block-message ch-request (slack/github-auth-blocks ch-request (:id github-provider))))
             (<! (simple-message ch-request "there is no scm credential linked for this repo - please type `@atomist authorize github`")))
           (finish ch-request :success "bot should ask user to authorize github")))))))

(defn extract-linked-repos
  "request middleware
     used during command handler processing to extract the set of linked repos for this channel

     - processor will stop if there are no linked channels so this should be inserted in the pipeline only when >0
       linked channels should be found.  It will output a message if there are zero linked channels"
  [handler]
  (fn [ch-request]
    (go
     (let [linked-repos (<! (linked-repos->channel ch-request (-> ch-request :source :slack :channel :id)))]
       (log/info "linked repos " linked-repos)
       (if (not (empty? linked-repos))
         (handler (assoc ch-request :linked-repos linked-repos))
         (do
           (<! (simple-message ch-request "there are no linked repos in this channel.  Please run this from a channel with linked repos or link a repo using `@atomist repos`"))
           (finish ch-request)))))))

(defn set-message-id
  "request middleware
     used during command handler processing to extract messageIds or create new random ones

     This is useful to insert into a pipeline that will keep updating a message over a set of events."
  [handler]
  (fn [ch-request]
    (handler
     (if-let [id (or (:id ch-request) (get-parameter-value ch-request "messageId"))]
       (assoc ch-request :id id)
       (assoc ch-request :id (str (random-uuid)))))))

(defn parameter-expression [ch-request expression]
  (reduce
   (fn [agg [token param-name]]
     (clojure.string/replace agg token (or (get ch-request (keyword param-name)) (get-parameter-value ch-request param-name))))
   expression
   (re-seq #"\$\{(.*?)\}" expression)))

(defn create-ref-from-first-linked-repo
  "request middleware
    use a requests's :linked-repos to construct a git ref with {:keys [repo owner branch]}
    - the repo name is a parameter expression which can use parameter values
    - currently always using master branch (should this be a middleware parameter?)"
  [handler]
  (fn [request]
    (let [repo (first (:linked-repos request))]
      (handler (assoc request :ref {:repo (:name repo)
                                    :owner (-> repo :org :owner)
                                    :branch (:branch request)})))))

(defn show-results-in-slack
  "request middleware
    will send a slack snippet containing the results in the current request.  Mostly for debugging
    This request must already have been configured to send to a particular Slack channel or be a command handler request."
  [handler & {:keys [result-type] :or {result-type "results"}}]
  (fn [request]
    (go
     (if-let [results (:results request)]
       (<! (snippet-message request (json/->str results) "application/json" result-type))
       (<! (simple-message request (gstring/format "no %s" result-type))))
     (handler request))))

(defn extract-github-token
  "request middleware
     will extract a github token by trying a few strategies

     - first check whether any request middleware has already found a github provider with a credential secret
       This is used when we have a shared authorization for the ResourceProvider

     - second, check whether any request middleware has stored a ref with an owner org.  If so, check whether this
       org has a GitHub Installation and then fetch an Installation token for this Org.

     If we find a token store it in the request under the :token key.  If we don't find a token, continue anyway.
     It's possible a good idea to supply an option to stop processing if no token is found."
  [handler]
  (fn [request]
    (let [{:keys [ref provider]} request
          secret (-> provider :credential :secret)]
      (if secret
        (do
          (log/info "found a provider credential secret")
          (handler (assoc request :token secret)))
        (go
         (log/info "search for GitHub app Installation")
         (if-let [token (-> (<! (graphql->channel request graphql/githubAppInstallationByOwner {:name (:owner ref)}))
                            :body
                            :data
                            :GitHubAppInstallation
                            first
                            :token
                            :secret)]
           (do
             (log/info "found a GitHubAppInstallation token")
             (handler (assoc request :token token)))
           (do
             (log/warn "did not find any GitHub creds for this Push")
             (handler request))))))))

(defn create-ref-from-push-event
  "request middleware
    construct a git ref with {:keys [repo owner branch]} using data from an event rooted at Push

    this middleware only works when the root event is a Push and the
      repo, owner, provider and branch are projected"
  [handler]
  (fn [request]
    (if-let [{:keys [name] {:keys [owner scmProvider]} :org} (-> request :data :Push first :repo)]
      (handler (assoc request :ref {:repo name
                                    :owner owner
                                    :branch (-> request :data :Push first :branch)}
                              :provider scmProvider))
      (finish request :failure "this event did not contain a valid Push"))))

(defn create-ref-from-repo
  "request middleware
    construct a git ref with {:keys [repo owner branch]} using middleware parameters

   middlware params
     repo - must conform to a Push-like repo structure with name and org keys.  org map should contain an scmProvider
            (should add specs here)
     branch - must be a map with branch name {:keys [branch]}

   This is only useful if a skill is hard-coding a repo/branch for some reason.  Typically, this data
     is extracted from the request, not from middleware parameters."
  [handler repo branch]
  (fn [request]
    (if-let [{:keys [name] {:keys [owner scmProvider]} :org} repo]
      (handler (assoc request :ref {:repo name
                                    :owner owner
                                    :branch (-> branch :name)}
                              :provider scmProvider))
      (finish request :failure "this event did not contain valid Push"))))

(defn run-sdm-project-callback
  "request middleware
     will prepare an SDM project and execute a (request,project) => channel callback.

   uses the :ref and the :token in the request to checkout an SDM Project (added into the request as :project)
   therefore this middleware must come after anything that extracts tokens or decides on the Project ref

   the callback has signature (request, project) => channel - an channel should emit results that
     will, in turn, be assoced on to the request and passed to the next handler.
     Any problems cloning the project or running the callback will result in the handler chain stopping with a failure
     message.  Non-retryable exceptions in processing will cause this skill to fail."
  [handler callback]
  (fn [request]
    (go
     (let [results (<! (sdm/do-with-shallow-cloned-project (partial callback request) (:token request) (:ref request)))]
       (if (contains? results :error)
         (let [message (gstring/format "Unable to checkout ref %s:  %s" (:ref request) (:message results))]
           (log/warn message)
           (finish request :failure message))
         (handler (assoc request :results results)))))))

(defn- send-status [request status]
  (ingest request {:category "SkillLog"
                   :correlation_context {:correlation_id (-> request :correlation_id)
                                         :automation {:name "skill"
                                                      :version "0.1.0"}}
                   :level "INFO"
                   :message (json/->str {:status (name status)
                                         :event_id (:eventId request)})
                   :team_id (-> request :team :id)
                   :timestamp (time/week-date-time-no-ms-now)} "AtomistLog"))

(defn make-request
  "This is probably the only function that should be called in a cljs skill handler.  It kicks off the request handler.

    - start with the incoming PubSubMessage.data, keywordize it, and use it as the request map
    - wrap the whole request processor in a Promise to satisfy the gcf Node.js runtime
    - create a done channel in the request so that request processing pipeline can finish early
    - wrap the whole pipeline in a try catch block
    - add a callback sendreponse function to the request map to handle writing events to the response topic
    - success/fail status are normally handled in the pipeline, but exception cases handled here will result in fail status"
  [data sendreponse handler]
  (log/infof "%s %s %s (%s)" meta/module-name meta/version meta/generated-at meta/tag)
  (promise/chan->promise
   (let [done-channel (chan)
         request (-> (js->clj data :keywordize-keys true)
                     (assoc :sendreponse sendreponse
                            :done-channel done-channel)
                     (event->request))]
     (go
      (try
        (handler request)
        (catch :default ex
          (log/error ex)
          (<! (failed-status request))
          (>! done-channel :done))))
     done-channel)))

(defn send-fingerprints
  "request middleware
     that will look for fingerprints in the request and send them to Atomist"
  [handler]
  (fn [request]
    (go
     (doseq [by-type (partition-by :type (:results request))]
       (if (not (empty? by-type))
         (let [o (assoc request :team {:id (-> request :extensions :team_id)})
               repo-branch-by-id (<! (graphql->channel
                                      o
                                      graphql/repo-branch-ids
                                      {:owner (-> request :data :Push first :repo :org :owner)
                                       :repo (-> request :data :Push first :repo :name)
                                       :branch (-> request :data :Push first :branch)}))
               variables {:additions (->> by-type
                                          (map (fn [{:as fp}]
                                                 (-> fp
                                                     (assoc :data (json/->str (:data fp)))))))
                          :isDefaultBranch (=
                                            (-> request :data :Push first :repo :defaultBranch)
                                            (-> request :data :Push first :branch))
                          :type (-> by-type first :type)
                          :branchId (-> repo-branch-by-id :body :data :Repo first :branches first :id)
                          :sha (-> request :data :Push first :after :sha)
                          :repoId (-> request :data :Push first :repo :id)}
               response (<! (graphql->channel
                             o
                             graphql/send-fingerprints
                             variables))]
           (log/info "send fingerprints response: " response))))
     (handler request))))

(defn from-channel
  "request middleware
     that pauses to run an async function ->channel and then continues with the functions result merged into request

   middleware parmams
     ->channel function (request) => channel where the channel result is needed by downstream processors
     key - the key to use when merging the async result into the request map"
  [handler ->channel & {:keys [key]}]
  (fn [request]
    (go
     (let [value (<! (->channel request))]
       (if key
         (handler request)
         (handler (assoc request key value)))))))

(defn add-skill-config
  "request middleware
     that looks for configuration added for a skill with dispatchStyle equal to multiple

   if the request has a configuration then named configuration values
     are merged into the request using a keywordized map"
  [handler & ks]
  (fn [request]
    (let [configuration (-> request :configuration)]
      (log/infof "found configuration %s - %s" (:name configuration) (:parameters configuration))
      (handler (reduce
                (fn [req k] (if-let [value (->> configuration :parameters (filter #(= (name k) (:name %))) first :value)]
                              (assoc req k value)
                              req))
                request
                ks)))))

(defn skip-push-if-atomist-edited
  "request middleware
    that checks whether a Push event contains a message with [atomist:edited]

    handler chain stops if the message contains [atomist:edited] and otherwise continues"
  [handler]
  (fn [request]
    (if (and (-> request :data :Push first :after :message) (s/includes? (-> request :data :Push first :after :message) "[atomist:edited]"))
      (do
        (log/info "skipping Push because after commit was made by Atomist")
        (finish request :success "skipping Push for Atomist edited Commits."))
      (handler request))))