(ns atomist.deps
  (:require [cljs-node-io.core :as io]
            [cljs-node-io.fs :as fs]
            [atomist.cljs-log :as log]
            [atomist.sdmprojectmodel :as sdm]
            [cljs.core.async :refer [<! timeout]]
            [atomist.sha :as sha]
            [atomist.json :as json]
            [atomist.api :as api]
            [atomist.graphql :as graphql]
            [goog.string :as gstring]
            [goog.string.format])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn- get-param [x s]
  (->> x (filter #(= s (:name %))) first :value))

(defn- off-target? [fingerprint target]
  (and (= (:name fingerprint) (:name target))
       (not (= (:sha fingerprint) (:sha target)))))

(defn configs->target-map
  "take configs with name/version parameters and convert them to an FP structure like:

   {:name \"metosin::compojure-api\",\n  :version \"1.1.12\"
    :sha \"abbd49a09f20626caf3a43eba07899c9502d7e5323b3beffa6995389272efe32\",
    :data [\"metosin/compojure-api\" \"1.1.12\"],
    :library \"metosin/compojure-api\"}

    configs that do not contain name/version parameters are filtered out"
  [configs]
  (->> configs
       (filter #(and (not (empty? (:parameters %)))))
       (map :parameters)
       (map (fn [x] (assoc {} :name (get-param x "name") :version (get-param x "version"))))
       (filter #(and (:name %) (:version %)))
       (map #(let [data [(:name %) (:version %)]] (assoc % :sha (sha/sha-256 (json/->str data)) :data data)))
       (map #(assoc % :name (gstring/replaceAll (:name %) "/" "::")
                      :library (:name %)))
       (into [])))

(defn- policy-order [v]
  (letfn [(is [x] (complement (fn [y] (= x y))))]
    (count (take-while (is v) ["latestSemVerUsed" "latestSemVerAvailable" "manualConfiguration"]))))

(defn configs->policy-map [configurations]
  (letfn [(policy-type [c] (->> c :parameters (filter #(= (:name %) "policy")) first :value))
          (dependencies [c] (->> c :parameters (filter #(= (:name %) "dependencies")) first :value))
          (read-dependency-string [c policy] (->> (cljs.reader/read-string (dependencies c))
                                                  (map str)
                                                  (reduce #(assoc %1 %2 {:policy policy}) {})))
          (read-manual-dependencies [c] (->> (cljs.reader/read-string (dependencies c))
                                             (reduce (fn [agg [k v]] (assoc agg (str k) {:policy :manualConfiguration
                                                                                         :version v})) {})))]
    (->> configurations
         (filter (constantly true))                         ;; TODO only apply policies applicable to current Repo filters
         (sort-by (comp policy-order policy-type))
         (map (fn [configuration] (case (policy-type configuration)
                                    "latestSemVerUsed" (read-dependency-string configuration :latestSemVerUsed)
                                    "latestSemVerAvailable" (read-dependency-string configuration :latestSemVerAvailable)
                                    "manualConfiguration" (read-manual-dependencies configuration))))
         (apply merge))))

(defn- get-latest [o dependency]
  (go
   (-> (<! (api/graphql->channel o graphql/query-latest-semver {:type "clojure-project-deps"
                                                                :name dependency}))
       :data
       :fingerprintAggregates)))

(defn version-channel [o policy-map fp-name]
  (go
   (let [dependency (gstring/replaceAll fp-name "::" "/")
         {:keys [policy] :as d} (policy-map dependency)]
     (case policy
       :latestSemVerAvailable (-> (<! (get-latest o fp-name))
                                  :latestSemVerAvailable)
       :latestSemVerUsed (-> (<! (get-latest o fp-name))
                             :latestSemVerUsed
                             :fingerprint)
       :manualConfiguration (let [data [dependency (:version d)]]
                              {:name (gstring/replaceAll dependency "/" "::")
                               :sha (sha/sha-256 (json/->str data))
                               :data data})))))

(defn apply-policy-target
  [{:keys [project configurations fingerprints] :as request}
   apply-name-version-library-editor]
  (go
   (let [policy-map (configs->policy-map configurations)]
     (doseq [{current-data :data :as fingerprint} fingerprints]
       (let [{target-data :data :as target} (<! (version-channel request policy-map (:name fingerprint)))]
         (log/info "check target " target)
         (when (off-target? fingerprint target)
           (let [body (gstring/format "off-target %s %s/%s -> %s/%s"
                                      (:displayType fingerprint)
                                      (nth current-data 0) (nth current-data 1)
                                      (nth target-data 0) (nth target-data 1))]
             (log/info body)
             (<! (apply-name-version-library-editor
                  project
                  {:branch (:name target)
                   :target-branch (-> request :ref :branch)
                   :title (gstring/format "%s: %s skill requesting change" (:library target) (:displayType fingerprint))
                   :body body}
                  (nth target-data 0)
                  (nth target-data 1)))))))
     :complete)))

(defn apply-name-version-fingerprint-target
  "iterate over configuration targets and fingerprints and check for off-target fingerprints
    we wrap any edits in sdm/commit-then-PR so this function might create several PRs,
    depending on how many off target leiningen versions are present.  "
  [{:keys [project configurations fingerprints] :as request}
   apply-name-version-library-editor]
  (log/info "configurations " configurations)
  (go
   (let [targets (configs->target-map configurations)]
     (doseq [{current-data :data :as fingerprint} fingerprints]
       (doseq [{target-data :data :as target} targets]
         (when (off-target? fingerprint target)
           (let [body (gstring/format "off-target %s %s/%s -> %s/%s"
                                      (:displayType fingerprint)
                                      (nth current-data 0) (nth current-data 1)
                                      (nth target-data 0) (nth target-data 1))]
             (log/info body)
             (<! (apply-name-version-library-editor
                  project
                  {:branch (:library target)
                   :target-branch (-> request :ref :branch)
                   :title (gstring/format "%s: %s skill requesting change" (:library target) (:displayType fingerprint))
                   :body body}
                  (nth target-data 0)
                  (nth target-data 1)))))))
     :complete)))

