(ns atomist.editors
  (:require [atomist.sdmprojectmodel :as sdm]
            [atomist.json :as json]
            [cljs.core.async :refer [<! >! timeout chan]]
            [atomist.cljs-log :as log]
            [atomist.api :as api]
            [cljs-node-io.core :as io])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn json-editor
  "create a file editor that will converge a value at a path in an associative data structure
    returns a file callback function (f) -> chan"
  [path-to-edit editor]
  (fn [f]
    (go (<! (sdm/set-content f (-> (<! (sdm/get-content f))
                                   (json/->obj :keywordize-keys false)
                                   (update-in path-to-edit editor)
                                   (json/->str))))
        true)))

(defn conditional-json-editor
  "create a file editor that will conditionally converge a string value at a path in an associate data structure

     params
       should-edit? - (content) -> boolean which must be true for the edit to take place
       path-to-edit - path to be used in get-in/update-in/assoc-in for associative maps
       target-string - the content to converge on (at the path-to-edit) if the should-edit? matches the content
     returns a function that will search a k8 Deployment spec for an updater config matching the owner/repo and
             then updates that deployment spec"
  [path-to-edit should-edit? editor]
  (fn [f]
    (go
     (let [spec (-> (<! (sdm/get-content f))
                    (json/->obj :keywordize-keys false))]
       (when (should-edit? spec)
         (<! (sdm/set-content f (json/->str (update-in spec path-to-edit (partial editor spec)))))))
     true)))

;; edit template
(defn perform-edits
  "standard template for using the SDM to do file edits
     returns a channel that will emit a value with the editor is complete"
  [file-editor glob-pattern token ref commit-message]
  (-> file-editor
      (sdm/do-with-files glob-pattern)
      (sdm/commit-then-push commit-message)
      (sdm/do-with-shallow-cloned-project token ref)))
