(ns awsutil.core
  (:use
   [amazonica.aws.ec2 :as ec2]
   [amazonica.core :as awscore]
   )
  (:gen-class))


(defn- get-name [instance]
  (:value (first  (filter #(= (:key %) "Name") (:tags  instance)))))

(defn _get-region-names [] (map #(:region-name %) (:regions (describe-regions))))
; cache 'em
(def region-names (memoize _get-region-names))

;; some predicates to apply to an instance
(defn tag-is? [instance tag-name tag-value]
  "return true if the instance has a tag with this value"
  (some #(and (= tag-value (get % :value)) (= tag-name (get % :key)))   (:tags instance)))

(defn is-production? [instance]
  "return true if Production instance"
  (tag-is? instance "Deployment" "Production" ))

(defn state-is? [instance state]
  "return true if instance is in this state e.g running"
  (= (get-in instance [:state :name]) state))

(defn elastic-ip? [all-ips instance]
  "Return address if instance is assigned an elastic ip else nil. 
   Receives array of ips from get-elasticips to avoid hitting API every time"
  (some #(= (:instance-id %) (:instance-id instance) ) all-ips))

(defn sleepable? [eips instance]
  "Sleep instances that: have elastic ip (and are therefore restartable), and are tagged as BusinessHours"
  (and 
   (elastic-ip? eips instance) 
   (tag-is? instance "BusinessHours" "Yes")))



(defn get-ec2-instances [region]
  "Get a list of ec2 servers in an account region"
  (map #(assoc % :name (get-name %) :region region) (mapcat #(:instances %)  (:reservations (describe-instances {:region region})) )))


(defn get-elastic-ips [region]
  "Get all elastic ips in an account region"
  (:addresses (describe-addresses {:region region})))

;; is this over engineered?
(defn munge-instances 
  ([instances do-this] 
   (munge-instances do-this (fn [_] true))) ;; one arg = process all 

  ([instances do-this  if-this?]
   (map #(do-this %) 
        (filter #(if-this? %) 
                instances))))


(defn stop-these-instances [instances]
  (munge-instances 
   instances 
   (fn [i] (stop-instances {:instance-ids [(:instance-id i)]}) i)
   (fn [i] (state-is? i "running")))
  )

(defn start-these-instances [instances]
  (munge-instances 
   instances 
   (fn [i] (start-instances {:instance-ids [(:instance-id i)]}) i)
   (fn [i] (not (#{"running" "stopping" "starting"} (get-in [:state :name] i) ))))
  )


(defn wake-sleep-region [region wake-or-sleep]
  (let [eips (get-elastic-ips region)
        instances (filter #(sleepable? eips %) 
                          (get-ec2-instances region))]
    (case wake-or-sleep
      :sleep (stop-these-instances instances)
      :wake  (start-these-instances instances))))


 
;; light em up
(defn start-all-sleepable-instances []
  (for [r (region-names)] 
    (map #(assoc {} {:name (:name %)} {:region (:region %)}) 
         (wake-sleep-region r :wake))))

 
;; shut em down
(defn stop-all-sleepable-instances []
  (for [r (region-names)] 
    (map #(assoc {} {:name (:name %)} {:region (:region %)}) 
         (wake-sleep-region r :sleep))))


