(ns diglett
  (:refer-clojure :exclude [def])
  (:require [clojure.spec :as s]
            [clojure.tools.logging :as log])
  (:import [org.jsoup Jsoup]
           [org.jsoup.nodes Element Node]
           [org.jsoup.select Elements]))

(declare extract*)

(defonce ^:private extractions-ref (atom {}))

(defn select*
  [^Node node ^String css-selector]
  (let [^Elements result (.select node css-selector)]
    (if (.isEmpty result)
      nil
      result)))

(defprotocol Extractable
  (extract [x y])
  (select [x y])
  (text [x]))

(extend-protocol Extractable
  nil
  (text [_] nil)

  java.lang.String
  (text [s] s)

  Node
  (text [node] (.text node))

  Element
  (text [node] (.text node))
  (select [node s] (select* node s))

  Elements
  (select [nodes s] (select* nodes s))

  clojure.lang.Keyword
  (extract [k node]
    (extract [(@extractions-ref k)
              ((s/registry) k)]
             node))

  clojure.lang.PersistentVector
  (extract [[extr spec] node]
    (extract* node extr spec)))

(defn parse [^String html]
  (.. (Jsoup/parseBodyFragment html) (children)))

(defn vectorize [x]
  (if (sequential? x)
    (vec x)
    [x]))

(defn spec->fn [spec]
  (case (keyword (first (vectorize spec)))
    :string? text
    :and (first (filter identity (map spec->fn (rest spec))))
    nil))

(defn attr [kw]
  (fn [node]
    (and node (let [s (.get (.attributes node) (name kw))]
                (when (seq s) s)))))

(defn ->node [x]
  (cond
   (sequential? x) (first x)
   (instance? Elements x) (first x)
   :else x))

(s/def ::element (s/and (s/or :e #(instance? Elements %)
                              :e #(instance? Element %))
                        (s/conformer last)))

(s/def ::extr (s/and (s/or :e (s/and string?
                                     (s/conformer (partial hash-map :sel)))
                           :e (s/and nil?
                                     (s/conformer (constantly {})))
                           :e (s/cat :sel string?
                                     :fns (s/* fn?))
                           :e (s/cat :fns (s/+ fn?)))
                     (s/conformer last)))

(s/def ::digging (s/and (s/or :element (s/cat :element ::element
                                              :extr ::extr)
                              :string (s/cat :string string?
                                             :extr ::extr))
                        (s/conformer last)))

(s/def ::spec (s/and (s/or :s (s/and s/spec?
                                     (s/conformer s/describe))
                           :s (s/and keyword?
                                     (s/conformer #(or (some-> (s/registry)
                                                               %
                                                               s/describe)
                                                       ::s/invalid)))
                           :s symbol?
                           :s nil?)
                     (s/conformer #(some-> % last))))

(defn def [x extr]
  (swap! extractions-ref assoc x extr))

(defn pull
  ([x fns]
   ((apply comp (reverse fns)) x))
  ([x fns spec]
   (if-let [f (spec->fn (cond
                         (s/spec? spec)     (s/describe spec)
                         (keyword? spec)    (s/describe spec)
                         (symbol? spec)     spec
                         (sequential? spec) (last spec)))]
     (pull x (distinct (conj fns f)))
     (pull x fns))))

(defmulti extract* (fn [node extr spec]
                     (first (vectorize (s/conform ::spec spec)))))

(defmethod extract* 'coll-checker [node extr spec]
  (let [{:keys [sel fns]} (s/conform ::extr extr)
        spec* (last (s/conform ::spec spec))
        node (cond-> node sel (select sel))]
    (if (keyword? spec*)
      (map #(extract spec* %) node)
      (map #(extract* % (seq fns) spec*) node))))

(defmethod extract* 'keys [node extr spec]
  (let [{:keys [req opt]} (apply hash-map (rest (s/conform ::spec spec)))
        node (->node (extract* node extr nil))]
    (reduce (fn [m k]
              (let [v (extract k (cond-> node
                                   (map? node) k))]
                (if (or (.contains req k) v)
                  (assoc m k v)
                  m)))
            {}
            (concat req opt))))

(defmethod extract* ::s/invalid [node extr spec]
  (throw (ex-info "Couldn't extract node" {})))

(defmethod extract* :default [node extr spec]
  (let [{:keys [sel fns]} (s/conform ::extr extr)
        node (cond-> node sel (select sel))]
    (pull (->node node) fns spec)))
