(ns clj-faktory.net
  (:require [clojure.java.shell :as shell]
            [clojure.string :as string]
            [aleph.tcp :as tcp]
            [byte-streams]
            [cheshire.core :as cheshire]
            [crypto.random :as random]
            [gloss.core :as gloss]
            [gloss.io :as io]
            [manifold.deferred :as d]
            [manifold.stream :as s]
            [clj-faktory.protocol.redis :as redis])
  (:import [java.net InetAddress URI]))

(def ^:private response-codec
  (redis/redis-codec :utf-8))

(gloss/defcodec- command-codec
  (gloss/string :utf-8 :delimiters ["\r\n"]))

(defn- hostname []
  (or (try (.getHostName (InetAddress/getLocalHost))
           (catch Exception _))
      (some-> (shell/sh "hostname")
              :out
              string/trim)))

(defn- worker-info []
  {:wid (random/hex 12)
   :hostname (hostname)
   :v 2})

(defn- read-and-parse-response [client]
  (let [[resp-type response] @(s/take! client)]
    (case resp-type
      :single-line response
      :bulk (cheshire/parse-string response true)
      :error (throw (Exception. response)))))

(defn- command-str [[verb & segments]]
  (->> segments
       (cons (string/upper-case (name verb)))
       (map #(if (map? %)
               (cheshire/generate-string %)
               %))
       (string/join " ")))

(defn- wrap-duplex-stream [s]
  (let [out (s/stream)]
    (s/connect
     (s/map #(io/encode command-codec (command-str %)) out)
     s)
    (s/splice
     out
     (io/decode-stream s response-codec))))

(defn send-command [client command]
  (s/put! client command)
  (read-and-parse-response client))

(defn handshake [client]
  (let [worker-info (worker-info)]
    (read-and-parse-response client)
    (send-command client [:hello worker-info])
    worker-info))

(defn connect [uri]
  (let [uri (URI. uri)
        host (.getHost uri)
        port (.getPort uri)]
    @(d/chain (tcp/client {:host host
                           :port port})
              #(wrap-duplex-stream %))))
