(ns baka-http
  (:refer-clojure :exclude [get])
  (:import [java.net
            CookieManager
            CookiePolicy
            InetSocketAddress
            Proxy
            Proxy$Type]
           [okhttp3
            Authenticator
            Credentials
            FormBody$Builder
            JavaNetCookieJar
            OkHttpClient$Builder
            Request
            Request$Builder]))

(defn headers->map [headers]
  (reduce (fn [m k]
            (assoc m k (.get headers k)))
          {}
          (.names headers)))

(defn ->form-body [form-params]
  (let [builder (FormBody$Builder.)]
    (.build (reduce (fn [builder* [k v]]
                      (doto builder*
                        (.add (name k) v)))
                    builder
                    form-params))))

(defn add-proxy [client host port]
  (-> (.newBuilder client)
      (.proxy (Proxy. Proxy$Type/HTTP
                      (InetSocketAddress. host
                                          port)))
      .build))

(defn disable-redirects [builder]
  (-> builder
      (.followRedirects false)
      (.followSslRedirects false)))

(defn cookie-jar []
  (let [cookie-manager (doto (CookieManager.)
                         (.setCookiePolicy CookiePolicy/ACCEPT_ALL))]
    (JavaNetCookieJar. cookie-manager)))

(defn client
  ([{:keys [follow-redirects? cookie-jar]
     :or {follow-redirects? true}}]
   (.build
    (cond-> (OkHttpClient$Builder.)
      (not follow-redirects?)
      disable-redirects

      cookie-jar
      (.cookieJar cookie-jar))))
  ([]
   (client {})))

(defn request
  [client url method {:keys [user-agent form-params
                             proxy-host proxy-port
                             proxy-user proxy-pass]}]
  (let [client* (cond-> client
                  (and proxy-host proxy-port)
                  (add-proxy proxy-host proxy-port))
        req (.build
             (cond-> (.url (Request$Builder.) url)
               (= method :post)
               (.post (->form-body form-params))

               (and proxy-user proxy-pass)
               (.header "Proxy-Authorization"
                        (Credentials/basic proxy-user
                                           proxy-pass))
               user-agent
               (.header "User-Agent" user-agent)))
        resp (.execute (.newCall client req))
        headers (.headers resp)
        body (.body resp)
        body* (.string body)]
    (.close body)
    {:body body*
     :headers (headers->map headers)
     :status (.code resp)}))

(defn get
  ([client url opts]
   (request client url :get opts))
  ([client url]
   (get client url {})))

(defn post
  ([client url opts]
   (request client url :post opts))
  ([client url]
   (post client url {})))
