(ns landing-web.client
  (:require
    [ajax.json :as ajax-json]
    [clojure.walk :refer [postwalk]]
    [clojure.string :as str]
    [clojure.string :as str]
    [re-frame.core :as rf]))

(defn get-bucket [config]
  (let [url (-> js/window .-location .-href)]
    (-> url
        (str/replace "http://" "")
        (str/replace "https://" "")
        (str/split #"\.")
        (first))))

(defn parse-custom-fields
  [edn]
  (postwalk (fn [x]
              (cond
                (and (string? x)
                     (str/starts-with? x ":")) (keyword (subs x 1))
                (and (string? x)
                     (str/starts-with? x "#")) (uuid (subs x 1))
                :else x))
            edn))

(defn custom-json-parser
  [& params]
  (-> (apply ajax-json/read-json-native params)
      (parse-custom-fields)))

(def custom-response-format
  (ajax-json/make-json-response-format
    custom-json-parser))

(defn fetch [uri params]
  (.fetch js/window
          uri
          (clj->js params)))

(defn read-response
  [response {:keys [on-success on-failure]}]
  (let [value {:status (.-status response)
               :body   (.json response)}
        failed (not= (:status value) 200)]
    (if-not failed
      (-> (js/Promise.resolve (clj->js (:body value)))
          (.then #(js->clj % :keywordize-keys true))
          (.then #(parse-custom-fields %))
          (.then #(rf/dispatch [on-success %])))
      (rf/dispatch [on-failure]))))

(defn get-content
  [config {:keys [on-success on-failure page] :as props}]
  (let [method :get
        mode :cors
        uri (str "https://landing-repo."
                 (:PublicHostedZoneName config)
                 "/"
                 (or page
                     (get-bucket config)) "/index.json")
        request (fetch uri {:method  method
                            :mode    mode
                            :timeout 20000
                            :headers {"Content-Type"                "application/json"
                                      "Access-Control-Allow-Origin" "*"}})]
    (->
      (js/Promise.resolve (clj->js request))
      (.then #(read-response % props)))))