(ns landing-web.client
  (:require
    [ajax.json :as ajax-json]
    [clojure.walk :refer [postwalk]]
    [clojure.string :as str]
    [clojure.string :as str]
    [re-frame.core :as rf]))


(defn get-bucket [config]
  (let [url (-> js/window .-location .-hostname)
        parts (str/split url #"\.")]
    (first parts)))

(defn parse-custom-fields
  [edn]
  (postwalk (fn [x]
              (cond
                (and (string? x)
                     (str/starts-with? x ":")) (keyword (subs x 1))
                (and (string? x)
                     (str/starts-with? x "#")) (uuid (subs x 1))
                :else x))
            edn))

(defn custom-json-parser
  [& params]
  (-> (apply ajax-json/read-json-native params)
      (parse-custom-fields)))

(def custom-response-format
  (ajax-json/make-json-response-format
    custom-json-parser))

(defn fetch [uri params]
  (.fetch js/window
          uri
          (clj->js params)))

(defn read-response
  [response {:keys [on-success on-failure]}]
  (let [value {:status (.-status response)
               :body   (.json response)}
        failed (not= (:status value) 200)]
    (if-not failed
      (-> (js/Promise.resolve (clj->js (:body value)))
          (.then #(js->clj % :keywordize-keys true))
          (.then #(parse-custom-fields %))
          (.then #(rf/dispatch [on-success %])))
      (rf/dispatch [on-failure]))))


(defn content []
  [{:type    :text
    :content {:src              "<p style=\"text-align: center;\">SAMPLE <span style=\"color: rgb(255, 0, 0);\"><strong><u>TEXT</u></strong></span></p>"
              :background-color "white"}}
   {:type    :text
    :content {:src              "<p style=\"text-align: center;\"><span style=\"font-size: 24px;\"><span style=\"color: rgb(241, 241, 241);\">ФИКСАТОР</span><span style=\"color: rgb(229, 216, 92);\">&nbsp;<span style=\"font-weight: 700;\">ГОЛЕНОСТОПА</span></span></span></p>"
              :background-color "rgb(46, 55, 67)"}}
   {:type    :image
    :content {:src "images/img_1.jpg"}}
   {:type    :image-set
    :content {:background-color "rgb(46, 55, 67)"
              :images           [{:src     "images/img_2_1.png"
                                  :caption {:src "<p>\n<p style=\"text-align: center;\"><span style=\"color: rgb(241, 241, 241);\">ФИКСАЦИЯ И&nbsp;</span></p>\n<p style=\"text-align: center;\"><span style=\"font-weight: 700; color: rgb(250, 237, 125);\">ПОДДЕРЖКА</span></p>\n</p>"}}
                                 {:src     "images/img_2_2.png"
                                  :caption {:src "<p>\n<p style=\"text-align: center;\"><span style=\"color: rgb(241, 241, 241);\">ФИКСАЦИЯ И&nbsp;</span></p>\n<p style=\"text-align: center;\"><span style=\"font-weight: 700; color: rgb(250, 237, 125);\">ПОДДЕРЖКА</span></p>\n</p>"}}
                                 {:src     "images/img_2_2.png"
                                  :caption {:src "<p>\n<p style=\"text-align: center;\"><span style=\"color: rgb(241, 241, 241);\">ФИКСАЦИЯ И&nbsp;</span></p>\n<p style=\"text-align: center;\"><span style=\"font-weight: 700; color: rgb(250, 237, 125);\">ПОДДЕРЖКА</span></p>\n</p>"}}]}}
   {:type    :price
    :content {:before   "309"
              :currency "UAH"
              :after    "12"}}
   {:type    :contact
    :content {:background-color "gray"
              :fields           [{:key   :email
                                  :label "E-Mail"}
                                 {:key   :phone
                                  :label "Phone number"}
                                 {:key   :quantity
                                  :label "Quantity"}]
              :button           {:label "Buy"
                                 :props {:color   "secondary"
                                         :variant "outlined"}}}}
   {:type    :video
    :content {:src              "https://player.vimeo.com/video/514268675"
              :background-color "white"}}])

(defn get-content
  [config {:keys [on-success on-failure page] :as props}]
  (let [method :get
        mode :cors
        bucket (get-bucket config)
        uri (str "https://landing-repo."
                 (:PublicHostedZoneName config)
                 "/"
                 (or page
                     bucket) "/index.json")]
    (if (= bucket "mock")
      (rf/dispatch [on-success (content)])
      (-> (js/Promise.resolve
            (clj->js (fetch uri {:method  method
                                 :mode    mode
                                 :timeout 20000
                                 :headers {"Content-Type"                "application/json"
                                           "Access-Control-Allow-Origin" "*"}})))
          (.then #(read-response % props))))))