;; Data Access Layer - Event Store Multimethods
;;
;; Defines the DAL interface for event sourcing operations.
;; Implementations dispatch on :edd-event-store key in context.
;;
;; REALM SUPPORT:
;; All implementations MUST support multi-tenancy via realm isolation.
;; Realm is extracted from ctx [:meta :realm], defaults to :test.
;; All operations are scoped to the current realm.
;;
;; AVAILABLE IMPLEMENTATIONS:
;; - :memory (edd.memory.event-store) - In-memory, realm-partitioned, for testing
;; - :postgres (edd.postgres.*) - PostgreSQL with realm isolation
;; - :dynamodb (edd.dynamodb.*) - DynamoDB with realm partitioning

(ns edd.dal
  (:require [clojure.tools.logging :as log]))

(set! *warn-on-reflection* true)
(set! *unchecked-math* :warn-on-boxed)

(defmulti get-events
  "Retrieves events for an aggregate within the current realm.
   
   Input: ctx with :id (aggregate ID) and optional :version (start from)
   Output: Vector of events sorted by event-seq
   
   Realm-scoped: Only returns events from current realm."
  (fn [{:keys [_id] :as ctx}]
    (:edd-event-store ctx :default)))

(defmethod get-events
  :default
  [_] [])

(defmulti get-aggregate-id-by-identity
  "Resolves identity string to aggregate UUID within current realm and service.
   
   Input: ctx with :identity (string or vector of strings) and :service-name
   Output: UUID (single identity) or map {identity -> UUID} (multiple)
   
   Realm-scoped: Identity lookups isolated per realm and service."
  (fn [{:keys [_identity] :as ctx}]
    (:edd-event-store ctx)))

(defmulti store-effects
  "Stores side-effect commands generated by reg-event-fx handlers.
   
   Realm-scoped: Commands stored in current realm's command-store."
  (fn [ctx] (:edd-event-store ctx)))

(defmulti log-request
  "Logs incoming command request for auditing and debugging.
   
   Input: ctx and body (command request)
   
   Realm-scoped: Request logs isolated per realm."
  (fn [{:keys [_commands] :as ctx} _body]
    (:edd-event-store ctx)))

(defmulti log-request-error
  "Logs failed command request with error details.
   
   Input: ctx, body (command request), and error
   
   Realm-scoped: Error logs isolated per realm."
  (fn [{:keys [_commands] :as ctx} _body _error]
    (:edd-event-store ctx)))

(defmulti log-response
  "Logs command response for idempotency checking.
   
   Input: ctx with :response-summary, :request-id, :breadcrumbs
   
   Realm-scoped: Response logs enable idempotent command handling per realm."
  (fn [ctx]
    (:edd-event-store ctx)))

(defmulti get-max-event-seq
  "Gets highest event-seq number for an aggregate in current realm.
   
   Input: ctx with :id (aggregate ID)
   Output: Integer (0 if no events exist)
   
   Used for optimistic locking - next event-seq is max + 1.
   Realm-scoped: Only considers events in current realm."
  (fn [{:keys [_id] :as ctx}]
    (:edd-event-store ctx)))

(defmulti get-command-response
  "Retrieves cached response for idempotent command handling.
   
   Input: ctx with :request-id and :breadcrumbs
   Output: Previous response or nil
   
   Prevents duplicate command execution within same breadcrumb chain.
   Realm-scoped: Response cache isolated per realm."
  (fn [{:keys [_request-id _breadcrumbs] :as ctx}]
    (:edd-event-store ctx)))

(defmulti store-results
  "Transactionally persists command results (events, identities, effects).
   
   Input: ctx with :resp {:events [...] :identities [...] :effects [...]}
   
   MUST be atomic - all or nothing. Enforces optimistic locking via event-seq.
   Realm-scoped: All stores isolated per realm."
  (fn [ctx] (:edd-event-store ctx)))

(defmulti with-init
  "Initializes event store context before processing.
   
   Used for connection pooling, transaction setup, etc.
   Default implementation is a no-op."
  (fn [ctx _body-fn] (:edd-event-store ctx)))

(defmethod with-init
  :default
  [ctx body-fn]
  (log/info "Default event init")
  (body-fn ctx))

(defmulti get-records
  "Retrieves all events and effects for a given interaction-id.
   
   Input: ctx and query with :interaction-id
   Output: {:events [...] :effects [...]}
   
   Used for debugging and tracing entire request flows.
   Realm-scoped: Only returns records from current realm."
  (fn [ctx _query] (:edd-event-store ctx)))
