(ns edd.core
  (:gen-class)
  (:require [clojure.tools.logging :as log]
            [lambda.request :as request]
            [edd.el.cmd :as cmd]
            [edd.schema :as s]
            [sdk.aws.s3 :as s3]
            [edd.el.event :as event]
            [edd.el.query :as query]
            [lambda.util :as util]
            [malli.error :as me]
            [malli.core :as m]
            [edd.dal :as dal]
            [edd.search :as search]
            [edd.ctx :as edd-ctx]))

(set! *warn-on-reflection* true)
(set! *unchecked-math* :warn-on-boxed)

(def RegCmdInputSchema
  "Schema for validating reg-cmd input options (before processing).
  
  Valid keys:
  - :consumes - Malli schema for command validation (optional but recommended)
  - :deps - Dependencies map or vector (optional)
  - :id-fn - Function to derive aggregate ID (optional)"
  (m/schema
   [:map {:closed true}
    [:consumes
     {:optional true}
     :any]
    [:deps
     {:optional true}
     [:or
      [:map]
      [:vector :any]]]
    [:id-fn
     {:optional true}
     [:fn fn?]]]))

(def EddCoreRegCmd
  "Schema for validating command registration options (after processing)."
  (m/schema
   [:map
    [:handler [:fn fn?]]
    [:id-fn [:fn fn?]]
    [:deps [:or
            [:map]
            [:vector :any]]]
    [:consumes
     [:fn #(m/schema? (m/schema %))]]]))

(defn reg-cmd
  "Register a command handler.
  
  Parameters:
  - ctx: Context map
  - cmd-id: Command identifier keyword (e.g., :create-order)
  - reg-fn: Handler function (fn [ctx cmd] -> event-map or [event-maps])
  - rest: Key-value pairs of options:
    - :consumes - Malli schema for command validation (optional but recommended)
    - :deps - Dependencies map or vector (default: {})
    - :id-fn - Function to derive aggregate ID (fn [ctx cmd] -> uuid)
  
  Throws:
  - ExceptionInfo if unknown keys are provided
  
  Returns: Updated context with registered command handler."
  [ctx cmd-id reg-fn & rest]
  (log/debug "Registering cmd" cmd-id)
  (let [input-options (reduce
                       (fn [c [k v]]
                         (assoc c k v))
                       {}
                       (partition 2 rest))]

    ;; Validate input options for unknown keys
    (when-not (m/validate RegCmdInputSchema input-options)
      (throw (ex-info "Invalid command registration options"
                      {:cmd-id cmd-id
                       :explain (-> (m/explain RegCmdInputSchema input-options)
                                    (me/humanize))})))

    (let [options (-> input-options
                      (assoc :id-fn (:id-fn input-options
                                            (fn [_ _] nil))))
          options (-> options
                      (assoc :deps (:deps options {})))
          options (update options
                          :consumes
                          #(s/merge-cmd-schema % cmd-id))
          options (assoc options :handler (when reg-fn
                                            (fn [& rest]
                                              (apply reg-fn rest))))]

      (when-not (m/validate EddCoreRegCmd options)
        (throw (ex-info "Invalid command registration"
                        {:cmd-id cmd-id
                         :explain (-> (m/explain EddCoreRegCmd options)
                                      (me/humanize))})))
      (edd-ctx/put-cmd ctx
                       :cmd-id cmd-id
                       :options options))))

(defn reg-event
  "Register an event handler to apply events to aggregate state.
  
  Event handlers are pure functions that build the current aggregate state
  by applying events sequentially. They are called during event replay.
  
  Parameters:
  - ctx: Context map
  - event-id: Event identifier keyword (e.g., :order-created)
  - reg-fn: Handler function (fn [aggregate event] -> new-aggregate)
  
  Returns: Updated context with registered event handler."
  [ctx event-id reg-fn]
  (log/debug "Registering apply" event-id)
  (update ctx :def-apply
          #(assoc % event-id (when reg-fn
                               (fn [& rest]
                                 (apply reg-fn rest))))))

(defn reg-agg-filter
  [ctx reg-fn]
  (log/debug "Registering aggregate filter")
  (assoc ctx :agg-filter
         (conj
          (get ctx :agg-filter [])
          (when reg-fn
            (fn [& rest]
              (apply reg-fn rest))))))

(def RegQueryInputSchema
  "Schema for validating reg-query input options (before processing).
  
  Valid keys:
  - :consumes - Malli schema for query input validation (optional)
  - :produces - Malli schema for query output validation (optional)
  - :deps - Dependencies map or vector (optional)"
  (m/schema
   [:map {:closed true}
    [:consumes
     {:optional true}
     :any]
    [:produces
     {:optional true}
     :any]
    [:deps
     {:optional true}
     [:or
      [:map]
      [:vector :any]]]]))

(def EddCoreRegQuery
  "Schema for validating query registration options (after processing)."
  (m/schema
   [:map
    [:handler [:fn fn?]]
    [:produces
     [:fn #(m/schema? (m/schema %))]]
    [:consumes
     [:fn #(m/schema? (m/schema %))]]]))

(defn reg-query
  "Register a query handler.
  
  Parameters:
  - ctx: Context map
  - query-id: Query identifier keyword (e.g., :get-order)
  - reg-fn: Handler function (fn [ctx query] -> result)
  - rest: Key-value pairs of options:
    - :consumes - Malli schema for query input validation (optional)
    - :produces - Malli schema for query output validation (optional)
    - :deps - Dependencies map or vector (default: {})
  
  Throws:
  - ExceptionInfo if unknown keys are provided
  
  Returns: Updated context with registered query handler."
  [ctx query-id reg-fn & rest]
  (log/debug "Registering query" query-id)
  (let [input-options (reduce
                       (fn [c [k v]]
                         (assoc c k v))
                       {}
                       (partition 2 rest))]

    ;; Validate input options for unknown keys
    (when-not (m/validate RegQueryInputSchema input-options)
      (throw (ex-info "Invalid query registration options"
                      {:query-id query-id
                       :explain (-> (m/explain RegQueryInputSchema input-options)
                                    (me/humanize))})))

    (let [options (update input-options
                          :consumes
                          #(s/merge-query-consumes-schema % query-id))
          options (-> options
                      (assoc :deps (:deps options {})))
          options (update options
                          :produces
                          #(s/merge-query-produces-schema %))
          options (assoc options :handler (when reg-fn
                                            (fn [& rest]
                                              (apply reg-fn rest))))]
      (when-not (m/validate EddCoreRegQuery options)
        (throw (ex-info "Invalid query registration"
                        {:query-id query-id
                         :explain (-> (m/explain EddCoreRegQuery options)
                                      (me/humanize))})))
      (assoc-in ctx [:edd-core :queries query-id] options))))

(defn reg-fx
  [ctx reg-fn]
  (update ctx :fx
          #(conj % (when reg-fn
                     (fn [& rest]
                       (apply reg-fn rest))))))

(defn event-fx-handler
  [ctx events]
  (mapv
   (fn [event]
     (let [handler (get-in ctx [:event-fx (:event-id event)])]
       (if handler
         (apply handler [ctx event])
         [])))
   events))

(defn reg-event-fx
  "Register a side-effect handler for an event.
  
  Effect handlers execute AFTER events are persisted, triggering new commands
  in response to events. Effects are stored transactionally with events and
  executed asynchronously.
  
  Parameters:
  - ctx: Context map
  - event-id: Event identifier keyword (e.g., :order-created)
  - reg-fn: Handler function (fn [ctx event] -> command-map or [command-maps])
  
  Handler return formats:
  - Single command: {:cmd-id :send-email :id (uuid/gen) :attrs {...}}
  - Multiple commands: [{:cmd-id :cmd-1 ...} {:cmd-id :cmd-2 ...}]
  - Different service: {:service :other-svc :commands [{...}]}
  - Multiple services: [{:service :svc-1 :commands [...]} {:service :svc-2 :commands [...]}]
  
  Returns: Updated context with registered effect handler."
  [ctx event-id reg-fn]
  (let [ctx (if (:event-fx ctx)
              ctx
              (reg-fx ctx event-fx-handler))]
    (update ctx
            :event-fx
            #(assoc % event-id (fn [& rest]
                                 (apply reg-fn rest))))))

(defn reg-service-schema
  "Register a service schema that will be serialised and returned when
  requested."
  [ctx schema]
  (assoc-in ctx [:edd-core :service-schema] schema))

(defn get-meta
  [ctx item]
  (merge
   (:meta item {})
   (:meta ctx {})))

(defn- add-log-level
  [mdc ctx item]
  (if-let [level (:log-level (get-meta ctx item))]
    (assoc mdc :log-level level)
    mdc))

(defn update-mdc-for-request
  [ctx item]
  (let [meta (get-meta ctx item)]
    (swap! request/*request* update :mdc
           (fn [mdc]
             (-> mdc
                 (assoc :invocation-id  (:invocation-id ctx)
                        :realm          (:realm meta)
                        :from-service   (:from-service meta)
                        :to-service     (:to-service meta)
                        :request-id     (:request-id item)
                        :breadcrumbs    (get item :breadcrumbs [0])
                        :interaction-id (:interaction-id item))
                 (add-log-level ctx item))))))

(defn try-parse-exception
  [^Throwable e]
  (or
   (ex-message e)
   "Unable to parse exception"))

(defn dispatch-item
  [{:keys [item] :as ctx}]
  (log/debug "Dispatching" item)
  (update-mdc-for-request ctx item)
  (let [item (update item :breadcrumbs #(or % [0]))
        ctx (assoc ctx
                   :meta (get-meta ctx item)
                   :request-id (:request-id item)
                   :breadcrumbs (:breadcrumbs item)
                   :interaction-id (:interaction-id item))]
    (try
      (let [item (if (contains? item :command)
                   (-> item
                       (assoc :commands [(:command item)])
                       (dissoc :command))
                   item)

            resp (cond
                   (> (count (:breadcrumbs item)) 25)
                   (do
                     (log/error :loop-detected item)
                     {:error :loop-detected})
                   (contains? item :apply) (event/handle-event (-> ctx
                                                                   (assoc :apply (assoc
                                                                                  (:apply item)
                                                                                  :meta (get-meta ctx item)))))
                   (contains? item :query) (-> ctx
                                               (query/handle-query item))
                   (contains? item :commands) (-> ctx
                                                  (cmd/handle-commands item))
                   (contains? item :error) item
                   :else (do
                           (log/error item)
                           {:error :invalid-request}))]
        (if (:error resp)
          {:error          (:error resp)
           :invocation-id  (:invocation-id ctx)
           :request-id     (:request-id item)
           :interaction-id (:interaction-id ctx)}
          {:result         resp
           :invocation-id  (:invocation-id ctx)
           :request-id     (:request-id item)
           :interaction-id (:interaction-id ctx)}))
      (catch Exception e
        (do
          (log/error e)
          (let [data (ex-data e)]
            (cond
              (:error data) {:exception      (:error data)
                             :invocation-id  (:invocation-id ctx)
                             :request-id     (:request-id item)
                             :interaction-id (:interaction-id ctx)}

              data {:exception      data
                    :invocation-id  (:invocation-id ctx)
                    :request-id     (:request-id item)
                    :interaction-id (:interaction-id ctx)}
              :else {:exception (try-parse-exception e)
                     :invocation-id  (:invocation-id ctx)
                     :request-id     (:request-id item)
                     :interaction-id (:interaction-id ctx)})))))))

(defn dispatch-request
  [{:keys [body] :as ctx}]
  (util/d-time
   "Dispatching"
   (let [items
         body

         batch-size
         (count items)

         responses
         (loop [items items
                responses []
                position 1]
           (let [item
                 (first items)

                 {:keys [exception]
                  :as response}
                 (util/d-time
                  (format "Dispatching batch item %s/%s"
                          position batch-size)
                  (dispatch-item (assoc ctx
                                        :item item)))

                 remaining
                 (rest items)

                 responses
                 (conj responses response)]
             (cond
               exception
               (do
                 (log/infof "Got excepion from handling item. Skipping ramining %s messages"
                            (count remaining))
                 responses)

               (seq remaining)
               (recur remaining
                      responses
                      (inc position))

               :else
               responses)))]
     (assoc
      ctx
      :resp responses))))

(defn fetch-from-s3
  [ctx {:keys [s3]
        :as msg}]
  (if s3
    (-> (s3/get-object ctx msg)
        slurp
        util/to-edn)
    msg))

(defn filter-queue-request
  "If request is coming from queue we need to get out all request bodies"
  [{:keys [body] :as ctx}]
  (if (contains? body :Records)
    (let [queue-body (mapv
                      (fn [it]
                        (->> it
                             :body
                             util/to-edn
                             (fetch-from-s3 ctx)))
                      (-> body
                          (:Records)))]
      (log/infof "Request From queue with batch size %s"
                 (count queue-body))
      (-> ctx
          (assoc :body queue-body
                 :queue-request true)))

    (assoc ctx :body [body]
           :queue-request false)))

(defn prepare-response
  "Wrap non error result into :result keyword"
  [{:keys [resp] :as ctx}]
  (log/infof "Preparing response. Is from queue: %s" (:queue-request ctx))
  (if (:queue-request ctx)
    resp
    (first resp)))

(def schema
  [:and
   [:map
    [:request-id uuid?]
    [:interaction-id uuid?]]
   [:or
    [:map
     [:command [:map]]]
    [:map
     [:commands sequential?]]
    [:map
     [:apply map?]]
    [:map
     [:query map?]]]])

(defn validate-request
  [{:keys [body] :as ctx}]
  (log/info "Validating request")
  (assoc
   ctx
   body
   (mapv
    #(if (m/validate schema %)
       %
       {:error (->> body
                    (m/explain schema)
                    (me/humanize))})
    body)))

(defn with-stores
  [ctx body-fn]
  (search/with-init
    ctx
    #(dal/with-init
       % body-fn)))

(defn handler
  [ctx body]
  (log/debug "Handler body" body)
  (log/debug "Context" ctx)
  (if (:skip body)
    (do (log/info "Skipping request")
        {})
    (with-stores
      ctx
      #(-> (assoc % :body body)
           (filter-queue-request)
           (validate-request)
           (dispatch-request)
           (prepare-response)))))

(defn -main
  [& _args]
  (let [ctx {}
        cmd-schema (m/schema
                    [:map {:closed true}
                     [:cmd-id [:= :test-native-cmd]]
                     [:id uuid?]])
        get-risk-on-dep {:service :glms-dimension-svc
                         :query   (fn [deps cmd]
                                    {:query-id :get-risk-on-by-id
                                     :id       (or (-> cmd :request :attrs :risk-on-id)
                                                   (-> cmd :request :attrs :consolidation-point :risk-on-id)
                                                   (-> deps :facility :attrs :risk-on-id))})}
        ctx (-> ctx
                (reg-cmd :test-native-cmd (fn [_ _]
                                            (log/info "Native test command")
                                            {:result {:status :ok}})
                         :consumes cmd-schema
                         :deps {:risk-on get-risk-on-dep}))]
    (log/info "Native image test")
    (log/warn "CTX" ctx)))
