(ns lambda.filters
  (:require [lambda.util :as util]
            [clojure.tools.logging :as log]
            [clojure.string :as str]
            [lambda.jwt :as jwt]
            [lambda.core :as lambda-core]
            [lambda.uuid :as uuid]
            [aws.ctx :as aws-ctx])
  (:import (lambda.core LambdaFilter)))

(defn has-role?
  [user role]
  (some #(= role %)
        (get user :roles [])))

(defn get-realm
  [ctx {:keys [realm roles]} _]
  (let [default-realm (get-in ctx [:auth :default-realm])
        realm (or realm
                  default-realm)]
    (when-not realm
      (throw (ex-info (str "Realm: " realm) {:error         "Missing realm in request token"
                                             :realm         realm
                                             :roles         roles
                                             :default-realm default-realm})))
    realm))

(defn non-interactive
  [user]
  (first
   (filter
    #(= % :non-interactive)
    (:roles user))))

(defn extract-user
  [ctx claims]
  (let [groups (:cognito:groups claims)
        groups (cond
                 (vector? groups) groups
                 groups (-> groups
                            (str/split #","))
                 :else [])
        groups (filter #(str/includes? % "-") groups)
        id-claim (keyword (get-in ctx [:auth :mapping :id] :email))
        id (get claims id-claim)]
    (reduce
     (fn [user group]
       (let [group (name group)
             [prefix value] (-> group
                                (str/split #"-" 2))
             [prefix value] (if (= prefix "lime")
                              ["roles" group]
                              [prefix value])
             [prefix value] (if (= group "non-interactive")
                              ["roles" group]
                              [prefix value])]
         (if (= prefix "realm")
           (assoc user :realm (keyword value))
           (update user (keyword prefix) conj (keyword value)))))
     {:id    id
      :roles '()
      :email (:email claims)}
     groups)))

(defmulti check-user-role
  (fn [_ctx request]
    (get-in request [:requestContext :authorizer :claims :token_use]
            (get-in request [:requestContext :authorizer :token_use]))))

(defmethod check-user-role "id"
  [_ctx request]
  (get-in request [:requestContext :authorizer :claims]))

(defmethod check-user-role "m2m"
  [_ctx request]
  (get-in request [:requestContext :authorizer]))

(defmethod check-user-role :default
  [ctx request]
  (jwt/parse-token
   ctx
   (or (get-in request [:headers :x-authorization])
       (get-in request [:headers :X-Authorization]))))

(defn extract-attrs
  [user-claims]
  (reduce
   (fn [p [k v]]
     (let [key (str/replace (name k) #"department_code" "department-code")
           key (cond
                 (= key "department") key
                 (= key "department-code") key
                 (str/starts-with? key "x-") (subs key 2)
                 (str/starts-with? key "custom:x-") (subs key 9)
                 :else nil)]
       (if key
         (assoc p (keyword key) v)
         p)))
   {}
   user-claims))

(defn parse-authorizer-user
  [ctx body user-claims]
  (let [{:keys [roles realm] :as user} (extract-user ctx user-claims)
        attrs (extract-attrs user-claims)
        selected-role (-> body :user :selected-role)
        role (if (and selected-role
                      (not-any? #(= % selected-role)
                                roles)
                      (not (non-interactive user)))
               (throw (ex-info "Selecting non-existing role"
                               {:message       "Selecting non-existing role"
                                :selected-role selected-role
                                :roles         roles}))
               selected-role)
        role (or role
                 (non-interactive user)
                 (first (remove
                         #(or (= % :anonymous)
                              (str/starts-with? (name %)
                                                "realm-"))
                         roles)))
        role (or role :anonymous)

        user (cond-> (merge user {:role role})
               realm (assoc :realm realm))]
    (if (empty? attrs)
      user
      (assoc user :attrs attrs))))

(defn assign-metadata
  [ctx request body]
  (let [{:keys [meta] :or {}} request
        {:keys [error] :as user-claims} (check-user-role ctx request)
        user-claims (if error
                      (throw (ex-info "User authentication error" error))
                      user-claims)
        {:keys [role] :as user} (parse-authorizer-user ctx body user-claims)]
    (if (= :non-interactive
           role)
      (assoc ctx :meta meta)
      (assoc ctx :meta (assoc meta
                              :realm (get-realm ctx user {})
                              :user (dissoc user :realm))))))

(defn create-headers
  [content-type]
  {:Access-Control-Allow-Headers  (str/join
                                   ","
                                   ["Id" "VersionId" "X-Authorization" "Content-Type"
                                    "X-Amz-Date" "Authorization" "X-Api-Key" "X-Amz-Security-Token"])
   :Access-Control-Allow-Methods  "OPTIONS,POST,PUT,GET"
   :Access-Control-Expose-Headers "*"
   :Content-Type                  content-type
   :Access-Control-Allow-Origin   "*"})

(def default-content-type "application/json")
(def default-headers (create-headers default-content-type))

(defn request->parse-body
  [request]
  (let [{isBase64Encoded :isBase64Encoded} request
        body (:body request)
        body (if isBase64Encoded
               (util/base64decode body)
               body)
        body (util/to-edn body)]
    body))

(defn from-api-handler
  [ctx
   {:keys [content-type]
    :or {content-type default-content-type}
    :as _config}
   request
   filter-chain]
  (let [{http-method :httpMethod
         path        :path} request
        resp (cond
               (= path "/health")
               {:healthy  true
                :build-id (util/get-env "BuildId" "b0")}

               (= http-method "OPTIONS")
               {:healthy  true
                :build-id (util/get-env "BuildId" "b0")}

               :else
               (let [body (request->parse-body request)
                     ctx (assign-metadata ctx request body)]
                 (lambda-core/continue-filter filter-chain
                                              ctx
                                              body)))
        {:keys [error
                exception]
         :as resp}  resp]
    (cond-> {:statusCode 200
             :isBase64Encoded false
             :headers        (create-headers content-type)
             :body           (util/to-json resp)}
      error (assoc :statusCode 400)
      exception (assoc :statusCode 500))))

(defn aws-api-gw-filter->filter-request
  [ctx config request filter-chain]
  (if (contains? request :path)
    (from-api-handler ctx config request filter-chain)
    (do
      (log/info "Skipping AWSApiGWFilter")
      (lambda-core/continue-filter filter-chain
                                   ctx
                                   request))))

(deftype AWSApiGWFilter [config]
  LambdaFilter
  (init-filter [_this ctx]
    (-> ctx
        jwt/fetch-jwks-keys
        aws-ctx/init
        jwt/ctx->aws-user-pool))
  (do-filter [_this ctx request filter-chain]
    (aws-api-gw-filter->filter-request ctx config request filter-chain)))

