(ns edd.view-store.searchable.s3
  (:require [clojure.tools.logging :as log]
            [malli.core :as m]
            [aws.ctx :as aws-ctx]
            [lambda.ctx :as lambda-ctx]
            [lambda.util :as util]
            [edd.view-store :as view-store]
            [edd.view-store.searchable :as searchable-view-store]
            [sdk.aws.s3 :as s3]))

(defn get-bucket-name
  [ctx aggregate-store]
  (str (aws-ctx/account-id ctx)
       "-"
       (name
        (lambda-ctx/environment-name-lower ctx))
       "-"
       aggregate-store))

(defn impl->get-key
  [realm
   service-name
   id]
  (let [partition-prefix (-> (str id)
                             last
                             str
                             util/hex-str-to-bit-str)]
    (str "aggregates/"
         (name realm)
         "/latest/"
         (name service-name)
         "/"
         partition-prefix
         "/"
         id
         ".json")))

(defn impl->get-snapshot
  [ctx config id & [_version]]
  (let [storage (:storage config)
        realm (lambda-ctx/realm ctx)
        service-name (lambda-ctx/get-service-name ctx)
        object (-> storage
                   (assoc-in [:s3 :object :key]
                             (impl->get-key realm service-name id)))
        {:keys [error] :as resp} (s3/get-object ctx object)]
    (log/infof "Done fetching from S3: %s, with response: %s" object resp)
    (cond
      (= resp nil) nil
      (not error) (-> resp
                      (slurp)
                      (util/to-edn)
                      :aggregate)
      :else (throw (ex-info "Error getting snapshot"
                            {:error  error
                             :object object})))))

(defn impl->update-snapshot
  [ctx config aggregate]
  (let [storage (:storage config)
        realm (lambda-ctx/realm ctx)
        service-name (lambda-ctx/get-service-name ctx)
        id (:id aggregate)
        object (-> storage
                   (assoc-in [:s3 :object :key]
                             (impl->get-key realm service-name id)))
        {:keys [error]
         :as resp}
        (s3/put-object ctx
                       (-> object
                           (assoc-in [:s3 :object :content]
                                     (util/to-json {:aggregate aggregate
                                                    :service-name service-name
                                                    :realm realm}))))]

    (log/infof "Done updating on S3: %s, with response: %s" object resp)
    (when error
      (throw (ex-info "Error updating snapshot"
                      {:error  error
                       :object object})))))

(defn impl->advanced-search
  [ctx config query])

(defn impl->simple-search
  [ctx config query])

(def S3ConfigSchema
  (m/schema
   [:map
    [:storage
     [:map
      [:s3
       [:map
        [:bucket
         [:map
          [:name string?]]]]]]]
    [:category-fn {:description "Function that returns category for aggregate.
                              It is used later for searching. Max aggregates
                              in category is 1000"} fn?]]))

(deftype S3ViewStore [config]
  view-store/ViewStore
  (init [_this ctx]
    (log/info "Intializing S3ViewStore")
    ctx)
  (get-snapshot [_this ctx query]
    (impl->get-snapshot ctx config query))
  (save-aggregate [_this ctx aggregate]
    (impl->update-snapshot ctx config aggregate))

  searchable-view-store/SearchableViewStore
  (advanced-search [_this ctx query]
    (impl->advanced-search ctx config query))
  (simple-search [_this ctx query]
    (impl->simple-search ctx config query)))

(defn get-default-config
  [ctx]
  {:storage     {:s3
                 {:bucket
                  {:name (get-bucket-name ctx "aggregates")}}}
   :category-fn (fn [aggregate]
                  (:id aggregate))})

(defn register
  [ctx & [config]]
  (view-store/register
   ctx
   (let [{:keys [aggregate-store-bucket
                 aggregate-store-name]}
         config]
     {:implementation-class S3ViewStore
      :config (cond-> (get-default-config ctx)

                aggregate-store-bucket
                (assoc-in [:storage :s3 :bucket :name]
                          aggregate-store-bucket)

                aggregate-store-name
                (assoc-in [:storage :s3 :bucket :name]
                          (get-bucket-name ctx aggregate-store-name)))

      :config-default (get-default-config ctx)
      :config-schema S3ConfigSchema})))
