(ns edd.view-store
  (:require
   [malli.core :as m]
   [malli.error :as me]))

(defn new-instance
  [clazz config args]
  (->
   (. clazz getConstructor (into-array
                            Class
                            (concat [Object]
                                    (map (constantly Object) args))))
   (.newInstance (into-array Object (concat [config]
                                            args)))))

(defn register
  [ctx {:keys [implementation-class
               config
               config-schema
               config-default
               args]
        :or {config-schema (m/schema [:map])
             config-default {}
             args []}}]

  (let [config (merge (:config ctx)
                      config-default
                      config)]
    (when (not (m/validate config-schema config))
      (let [explain (-> (m/explain config-schema config)
                        (me/humanize))]
        (throw (ex-info (str "Error registering S3 view-store" explain)
                        {:error explain}))))
    (assoc ctx :view-store
           (new-instance
            implementation-class
            config
            args))))

(defprotocol ViewStore
  "Implementation of view store. View store needs to be stored 
  in context as :view-store key. There is `edd.view-store/register
  helper function for registration. Each implementation namespace
  must offer register function to make use easier."
  (init [this ctx]
    "This will be called when server is started.
   Inside body filter-chain should be invoked accepting ctx 
   as parameter to continue invocation. This function can be used 
   to bind atom for cache or to open DB connections pool.")
  (get-snapshot [this ctx {:keys [id version] :as query}]
    "Returns snapshot of aggregate stored in this 
   view store. Agregate is retrieved by id. I case 
   view-store supports keeping all versions, specific 
   aggregate version can be returned by using version in query.")
  (save-aggregate [this ctx aggregate]
    "Save aggregate snapshot in view-store. It will replace 
   latest version stored. If view store supports 
   keeping all version it will also keep previous  
   version of aggregate. Previous version can then 
   be retrieved by calling get-snapshot with version."))

