;; This namespace functions emulating Data Access Layer
;; by redefining Data Access Layer function and binding
;; state to thread local. Should be used only in tests.
;; Any test wanted to use emulated Data Access Layer
;; should call "defdaltest" macro.
;;
;; Example:
;;
;; (defdaltest when-store-and-load-events
;;   (dal/store-event {} {} {:id 1 :info "info"})
;;   (verify-state [{:id 1 :info "info"}] :event-store)
;;   (let [events (dal/get-events {} 1)]
;;     (is (= [{:id 1 :info "info"}]
;;            events))))
;;

(ns edd.test.fixture.dal
  (:require [clojure.tools.logging :as log]
            [clojure.test :refer [is]]
            [malli.core :as m]
            [malli.error :as me]
            [edd.el.query :as edd-client]
            [edd.response.cache :as response-cache]
            [edd.core :as edd]
            [lambda.test.fixture.core :as lambda-fixture-core]
            [lambda.util :as util]
            [lambda.core :as lambda-core]
            [lambda.runtime :as lambda-runtime]
            [edd.common :as common]
            [lambda.uuid :as uuid]
            [edd.memory.event-store :as event-store]
            [edd.view-store.searchable.memory :as memory-view-store]
            [edd.view-store.searchable :as searchable-view-store]
            [edd.view-store.common :as view-store-common]
            [lambda.test.fixture.client :as client]
            [lambda.test.fixture.state :refer [*queues*]]
            [aws.aws :as aws]
            [edd.ctx :as edd-ctx]
            [lambda.ctx :as lambda-ctx]
            [edd.memory.middleware :as memory-middleware]
            [edd.core])
  (:import (lambda.runtime MemoryRuntime PassTroughMiddleware)
           (edd.memory.middleware EddMemoryMiddleware)
           (edd.core EddHandler)
           (org.slf4j MDC)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; Data Access Layer ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn- like-cond
  "Returns function which checks if map contains key value pair
  described in condition, where value is not exact match"
  [condition]
  (log/info "like-cond" condition)
  (let [k (key (first condition))
        v (val (first condition))]
    (fn [x] (> (.indexOf (get x k) v) 0))))

(defn- equal-cond
  "Returns function which checks if map contains key value pair
  described in condition, where value is exact match"
  [condition]
  (log/info "equal-cond" condition)
  (let [k (key (first condition))
        v (val (first condition))]
    (fn [x] (= (get x k) v))))

(defn- full-search-cond
  "Returns function which checks if map contains any value which contains
  condition"
  [condition]
  (log/info "full-search-cond" condition)
  (fn [x] (some #(> (.indexOf % condition) 0) (vals x))))

(defn map-conditions [condition]
  (cond
    (:like condition) (like-cond (:like condition))
    (:equal condition) (equal-cond (:equal condition))
    (:search condition) (full-search-cond (:search condition))
    :else (fn [_] false)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;   Test Fixtures   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def service-name lambda-ctx/default-service-name)

(def ctx
  (-> {:service-name service-name
       :meta {:realm :mock}}
      (assoc-in [:edd :config :secrets-file] "files/secret-eu-west.json")
      (response-cache/register-default)
      (memory-view-store/register)
      (event-store/register)))

#_(def mock-s3-ctx
    (-> {:service-name service-name}
        (assoc-in [:edd :config :secrets-file] "files/secret-eu-west.json")
        (response-cache/register-default)
        (s3-view-store/register :implementation :mock)
        (event-store/register)))

(def current-ctx)

(defn create-identity
  [identities & [id]]
  (get identities id (uuid/gen)))

(def DepsSchema
  (m/schema
   [:vector
    [:map
     [:resp {:optional true}
      [:or [:map]
       nil?]]
     [:service [:or string? keyword?]]
     [:query
      [:map
       [:query-id keyword?]]]]]))

(defn compatible-response
  [{:keys [success]
    :as resp}]
  (if (nil? success)
    resp
    (dissoc resp :success)))

(defn prepare-dps-calls
  [deps]
  (when-not (m/validate DepsSchema deps)
    (throw (ex-info "Invcalid schema for :deps"
                    {:message "Invalid schema for :deps"
                     :validation (-> (m/explain DepsSchema deps)
                                     (me/humanize))})))
  (mapv
   #(-> {:method :post
         :url (edd-client/calc-service-query-url (:service %))
         :response {:body (util/to-json {:result (:resp %)})}
         :request {:body {:query
                          (:query %)}}})
   deps))

(defn aws-get-token
  [_ctx]
  "#mock-id-token")

(def ^:dynamic *params* (atom {}))
(def ^:dynamic *runtime*)
(def ^:dynamic *ctx*)

(defn prepare-runtime
  [default-ctx & [params]]
  (or
   (when (bound? #'*runtime*)
     *runtime*)
   (let [ctx (:ctx params default-ctx)
         params (or params {})
         ctx (merge
              ctx
              (select-keys params
                           (concat
                            [:filters
                             :middleware]
                            (keys memory-view-store/default-elastic-store)
                            (keys event-store/default-db))))
         middleware   (get ctx
                           :middleware
                           (memory-middleware/EddMemoryMiddleware. {}))
         runtime (MemoryRuntime.
                  {:handler (EddHandler. {:middleware middleware})
                   :filters (:filters ctx)}
                  (atom 0))]
     runtime)))

(defn get-runtime
  []
  (when-not (bound? #'*runtime*)
    (throw (ex-info "Cannot get runtime, runtime not bound to *runtime*"
                    {:runtime *runtime*})))
  *runtime*)

(defn init-runtime
  [ctx & [params]]
  (when-not (bound? #'*runtime*)
    (throw (ex-info "Cannot init runtime, runtime not bound to *runtime*"
                    {:runtime *runtime*})))
  (let [ctx (merge
             ctx
             (select-keys params
                          (concat
                           (keys memory-view-store/default-elastic-store)
                           (keys event-store/default-db))))
        ctx (lambda-core/init-runtime *runtime* ctx)]
    ctx))

(defmacro with-mock-dal [& body]
  (let [params (cond
                 (or (map? (first body))
                     (symbol? (first body))) (first body)
                 (= (-> body
                        first
                        first)
                    'assoc) (first body)
                 :else {})]
    `(do
       (MDC/put "prefix" " ")
       (when  lambda-fixture-core/*mocking*
         (throw (ex-info "Nested mocking" {:message "Nested mocking not allowed"})))
       (let [ctx#  (:ctx ~params ctx)]
         (with-redefs [util/get-env (partial lambda-fixture-core/get-env-mock (:env ~params))]
           (binding [lambda-fixture-core/*mocking* true
                     event-store/*event-store* (event-store/init-db ctx# ~params)
                     ; We will create new runtime based on default context. If
                     ; runtime based on custom ctx is needed it can be passed in as
                     ; :ctx to this macro
                     *runtime* (prepare-runtime ctx# ~params)
                     *params* (atom (merge {:seed (rand-int 10000000)}
                                           ~params))]
             (binding [*ctx* (init-runtime ctx# ~params)]
               (let [deps# (-> (get ~params
                                    :deps
                                    (get ~params :dps []))
                               (prepare-dps-calls)
                               (concat (get ~params :responses []))
                               (vec))
                     identities# (get ~params :identities {})
                     http-mock# (select-keys ~params [:http-mock])
                     http-ctx# ctx#
                     http-ctx# (merge http-ctx#
                                      http-mock#)]
                 (client/mock-http
                  http-ctx#
                  deps#
                  (with-redefs [aws/get-token aws-get-token
                                common/create-identity (partial create-identity identities#)]
                    (util/d-time
                     (str "with-mock-dal using seed" (:seed *queues*))
                     (do ~@body))))))))))))

(defmacro with-runtime
  "Docu"
  [params & body]
  `(do
     (MDC/put "prefix" " ")
     (when  lambda-fixture-core/*mocking*
       (throw (ex-info "Nested mocking" {:message "Nested mocking not allowed"})))
     (let [deps# (-> (get ~params
                          :deps
                          (get ~params :dps []))
                     (prepare-dps-calls)
                     (concat (get ~params :responses []))
                     (vec))
           identities# (get ~params :identities {})
                                        ; We will use default context. If
                                        ; custom ctx is needed it can be passed in as
                                        ; :ctx to this macro
           ctx# (:ctx ~params ctx)
           http-mock# (select-keys ~params [:http-mock])
           env# (:env ~params {})]
       (with-redefs [util/get-env (partial lambda-fixture-core/get-env-mock env#)
                     aws/get-token aws-get-token
                     common/create-identity (partial create-identity identities#)]
         (binding [lambda-fixture-core/*mocking* true
                   *runtime* (prepare-runtime ctx# ~params)

                   *params* (atom (merge {:seed (rand-int 10000000)}
                                         ~params))]
           (let [ctx# (init-runtime ctx# ~params)
                 ~(:ctx params) ctx#]
             (binding [*ctx* ctx#
                       event-store/*event-store* (event-store/init-db ctx# ~params)]
               (client/mock-http
                (merge ctx#
                       http-mock#)
                deps#
                (do ~@body)))))))))

(defn get-state
  [x]
  (when-not (bound? #'*runtime*)
    (throw (ex-info "State can be retrived only within runtime"
                    {:runtime-bound? (bound? #'*runtime*)})))

  (when-not (bound? #'*ctx*)
    (throw (ex-info "State can be retrived only when ctx is initialized"
                    {:runtime-bound? (bound? #'*ctx*)})))
  (let [ctx *ctx*
        state (vec
               (if (= x :aggregate-store)
                 (searchable-view-store/simple-search
                  (:view-store ctx)
                  ctx
                  {})
                 (event-store/get-db ctx x)))]
    state))

(defn process-state
  [store values]
  (case store
    :event-store (mapv
                  (fn [{:keys [meta] :as v}]
                    (if meta
                      (update v :meta #(dissoc % :realm))
                      v))
                  values)
    :command-store (mapv
                    (fn [{:keys [meta] :as v}]
                      (if (:realm meta)
                        (update v :meta #(dissoc % :realm))
                        v))
                    values)
    values))

(defmacro verify-state [x & [y]]
  `(let [key# (if (keyword? ~y) ~y ~x)
         value# (if (keyword? ~y) ~x ~y)]
     (is (= value#
            (process-state
             key#
             (get-state key#))))))

(defmacro verify-state-fn [x fn y]
  `(is (= ~y
          (mapv
           ~fn
           (get-state ~x)))))

(defn pop-state
  "Retrieves commands and removes them from the store"
  [x]
  (let [ctx *ctx*
        current-state (event-store/get-db ctx x)]
    (event-store/update-db ctx x (fn [_v] []))
    current-state))

(defn peek-state
  "Retrieves the first command without removing it from the store"
  [& [x]]
  (let [ctx *ctx*]
    (if x
      (event-store/get-db ctx x)
      (event-store/get-db ctx))))

(defn- re-parse
  "Sometimes we parse things from outside differently
  because keywordize keys is by default. If you have map of string
  keys they would be keywordized. But if we pass in to test map
  that has string we would receive string. So here we re-parse requests"
  [cmd]
  (util/fix-keys cmd))

(def ^:dynamic *responses*)

(defn remove-meta
  [resp]
  (cond-> resp

    true
    (dissoc :request-id
            :invocation-id
            :interaction-id)

    (get-in resp [:result :events])
    (update-in [:result :events]
               #(if (number? %)
                  %
                  (mapv
                   (fn [event]
                     (dissoc event
                             :request-id
                             :interaction-id
                             :meta))
                   %)))

    (get-in resp [:result :effects])
    (update-in [:result :effects]
               #(if (number? %)
                  %
                  (mapv
                   (fn [cmd]
                     (dissoc cmd
                             :request-id
                             :interaction-id
                             :meta))
                   %)))))

(defn resolve-tracking
  [ctx cmd]
  {:request-id (or (:request-id cmd)
                   (:request-id ctx)
                   (uuid/gen))
   :interaction-id (or (:interaction-id cmd
                                        (:interaction-id ctx))
                       (uuid/gen))})
(defn process-response [{:keys [include-meta]
                         :as _ctx
                         :or {include-meta false}}
                        resp]
  (if include-meta
    resp
    (if (map? resp)
      (remove-meta resp)
      (mapv remove-meta resp))))

(defn handle-cmd
  [{:keys [service-name]
    :as ctx}
   cmd]
  (try
    (let [runtime (get-runtime)
          {:keys [request-id
                  interaction-id]} (resolve-tracking
                                    ctx
                                    cmd)
          cmd (if (contains? cmd :commands)
                (re-parse cmd)
                {:commands [(re-parse cmd)]
                 :service-name service-name})
          cmd (assoc cmd
                     :request-id request-id
                     :interaction-id interaction-id)
          resp  (when (or (= service-name
                             (:service cmd))
                          (= nil
                             (:service cmd)))
                  (lambda.core/run-request runtime
                                           ctx
                                           (re-parse cmd)))
          resp (process-response ctx resp)]
      (compatible-response resp))
    (catch Exception ex
      (log/error "CMD execution ERROR" ex)
      (ex-data ex))))

(def handle-commands handle-cmd)

(defn get-commands-response
  [ctx cmd]
  (handle-cmd (assoc ctx
                     :no-summary true)
              cmd))

(defn apply-events
  [{:keys [request-id interaction-id]
    :or {request-id (uuid/gen)
         interaction-id (uuid/gen)}
    :as ctx} id]
  (let [runtime (get-runtime)
        resp (lambda-core/run-request runtime
                                      (assoc ctx
                                             :request-id request-id
                                             :interaction-id interaction-id)
                                      {:apply {:aggregate-id id}
                                       :request-id request-id
                                       :interaction-id interaction-id})]
    (-> (process-response ctx resp)
        compatible-response)))

(defn apply-cmd
  [{:keys [no-summary]
    :or {no-summary true}
    :as ctx}
   cmd]
  (log/info "apply-cmd" cmd)
  (let [ctx (assoc ctx :no-summary no-summary)
        resp (handle-cmd ctx cmd)
        resp (if (map? resp)
               resp
               (first resp))
        ids (->>  (get-in resp [:result :events])
                  (map :id)
                  distinct)
        {:keys [request-id
                interaction-id]} (resolve-tracking
                                  ctx
                                  cmd)

        items (mapv
               (fn [id]
                 {:apply {:aggregate-id id}
                  :meta (:meta cmd)
                  :request-id request-id
                  :interaction-id interaction-id})
               ids)
        items (remove
               #(not
                 (get-in % [:apply :aggregate-id]))
               items)
        items (distinct items)
        runtime (get-runtime)]
    (log/info "apply-cmd returned event ids: " ids)
    (mapv
     (fn [item]
       (lambda-core/run-request runtime
                                ctx
                                item))
     items)
    (compatible-response resp)))

(def max-depth 5)

(defn execute-cmd
  "Executes a command and applies all the side effects
   and executes also the produced commands until the
  command store is empty."
  [ctx cmd]
  (let [ctx (assoc ctx
                   :include-meta true
                   :no-summary true)
        cmds (if (seq? cmd)
               cmd
               [cmd])]
    (loop [depth 0
           cmds cmds
           responses []]
      (when (> depth max-depth)
        (throw (ex-info "Too deep man"
                        {:depth depth
                         :max max-depth})))
      (let [resp (mapv
                  #(apply-cmd ctx %)
                  cmds)
            effects (flatten
                     (map
                      #(-> %
                           :result
                           :effects)
                      resp))
            effects (remove nil? effects)]
        (if-not (empty? effects)
          (recur (inc depth)
                 effects
                 (conj responses resp))
          responses)))))

(defn execute-fx [ctx]
  (execute-cmd ctx
               (peek-state :command-store)))

(defn handle-event
  [{:keys [request-id interaction-id]
    :or {request-id (uuid/gen)
         interaction-id (uuid/gen)}
    :as ctx} event]
  (let [runtime (get-runtime)
        resp (lambda-core/run-request runtime
                                      (assoc ctx
                                             :request-id request-id
                                             :interaction-id interaction-id)
                                      (assoc
                                       event
                                       :request-id request-id
                                       :interaction-id interaction-id))]
    (-> (process-response ctx resp)
        compatible-response)))

(defn run-request
  [ctx request]
  (let [runtime (get-runtime)
        resp (lambda-core/run-request runtime
                                      ctx
                                      request)]
    (process-response ctx resp)))

(defn query
  [ctx query]
  (let [request-id (or (:request-id query)
                       (:request-id ctx)
                       (uuid/gen))
        interaction-id (or (:interaction-id query)
                           (:interaction-id ctx)
                           (uuid/gen))
        query (if (contains? query :query)
                (re-parse query)
                (re-parse {:query query}))
        query (assoc query
                     :request-id request-id
                     :interaction-id interaction-id)
        runtime (get-runtime)
        resp (lambda-core/run-request runtime
                                      (assoc ctx
                                             :request-id request-id
                                             :interaction-id interaction-id)
                                      query)
        resp (if (vector? resp)
               (first resp)
               resp)

        resp
        (compatible-response resp)]
    (process-response ctx resp)))

(def handle-query query)

(defn get-by-id
  [ctx {:keys [id]
        :as agg-id}]
  (let [ctx (-> ctx
                (edd/reg-query :mock->get-by-id common/get-by-id))
        resp (query ctx {:query-id :mock->get-by-id
                         :id (or id
                                 agg-id)})
        resp (compatible-response resp)]
    resp))
