(ns edd.memory.event-store
  (:require
   [clojure.tools.logging :as log]
   [edd.dal :refer [get-aggregate-id-by-identity get-command-response
                    get-events get-id-for-sequence-number get-max-event-seq
                    get-sequence-number-for-id log-dps log-request log-request-error log-response
                    store-results with-init]]
   [lambda.ctx :as lambda-ctx]
   [lambda.test.fixture.state :as state]
   [lambda.util :as util])
  (:import
   (clojure.lang RT)
   (java.util
    ArrayList
    Collection
    Collections
    Random)))

(defn fix-keys
  [val]
  (-> val
      (util/to-json)
      (util/to-edn)))

(def default-db
  {:event-store    []
   :identity-store []
   :sequence-store []
   :command-store  []
   :response-log   []
   :command-log    []})

(def ^:dynamic *event-store* (atom default-db))

(defn get-db
  [ctx & [store]]
  (let [service-name
        (lambda-ctx/get-service-name ctx)

        realm
        (lambda-ctx/realm ctx)]
    (if store
      (get-in @*event-store*
              [service-name realm store])
      (get-in @*event-store*
              [service-name realm]))))

(defn- prepare-data
  [params default-store]
  (merge
   default-store
   (util/fix-keys
    (select-keys
     params
     (keys default-store)))))

(defn init-db
  [ctx data]
  (let [service-name
        (lambda-ctx/get-service-name ctx)

        realm
        (lambda-ctx/realm ctx)]
    (atom
     (assoc-in {}
               [service-name realm]
               (prepare-data data
                             default-db)))))

(defn update-db
  [ctx store update-fn]
  (let [service-name
        (lambda-ctx/get-service-name ctx)

        realm
        (lambda-ctx/realm ctx)]
    (swap! *event-store*
           #(update-in %
                       [service-name realm store]
                       update-fn))))

(defn store-sequence
  "Stores sequence in memory structure.
  Raises RuntimeException if sequence is already taken"
  [ctx {:keys [id]}]
  {:pre [id]}
  (log/info "Emulated 'store-sequence' dal function")
  (let [store (get-db ctx :sequence-store)
        sequence-already-exists (some
                                 #(= (:id %) id)
                                 store)
        max-number (count store)]
    (when sequence-already-exists
      (throw (RuntimeException. "Sequence already exists")))
    (update-db
     ctx
     :sequence-store
     (fn [v] (conj v {:id    id
                      :value (inc max-number)})))))

(defn store-identity
  "Stores identity in memory structure.
  Raises RuntimeException if identity is already taken"
  [ctx identity]
  (log/info "Emulated 'store-identity' dal function")
  (let [id-fn
        (juxt :id :identity)

        id
        (id-fn identity)

        store
        (get-db ctx :identity-store)

        id-already-exists (some #(= (id-fn %) id) store)]
    (when id-already-exists
      (throw (RuntimeException. "Identity already exists")))
    (update-db
     ctx
     :identity-store
     (fn [v] (conj v (dissoc identity
                             :request-id
                             :interaction-id
                             :meta))))))

(defn deterministic-shuffle
  [^Collection coll seed]
  (let [al (ArrayList. coll)
        rng (Random. seed)]
    (Collections/shuffle al rng)
    (RT/vector (.toArray al))))

(defn enqueue [q item seed]
  (vec (deterministic-shuffle (conj (or q []) item) seed)))

(defn peek-cmd!
  []
  (let [popq (fn [q] (if (seq q) (pop q) []))
        [old _new] (swap-vals! (:command-queue state/*queues*) popq)]
    (peek old)))

(defn enqueue-cmd! [cmd]
  (swap! (:command-queue state/*queues*) enqueue cmd (:seed state/*queues*)))

(defn clean-commands
  [cmd]
  (dissoc cmd
          :request-id
          :interaction-id
          :breadcrumbs))

(defn store-command
  "Stores command in memory structure"
  [ctx cmd]
  (log/info "Emulated 'store-cmd' dal function")
  (update-db
   ctx
   :command-store
   (fn [x]
     (conj (or x [])
           (clean-commands cmd))))
  (enqueue-cmd! cmd))

(defn get-stored-commands
  [ctx]
  (get-db ctx :command-store))

(defn store-event
  "Stores event in memory structure"
  [ctx event]
  (log/info "Emulated 'store-event' dal function")
  (let [aggregate-id (:id event)]
    (when (some (fn [e]
                  (and (= (:id e)
                          aggregate-id)
                       (= (:event-seq e)
                          (:event-seq event))))
                (get-db ctx :event-store))
      (throw (ex-info "Already existing event" {:id        aggregate-id
                                                :event-seq (:event-seq event)})))
    (update-db
     ctx
     :event-store
     (fn [v]
       (sort-by
        :event-seq
        (conj v (dissoc
                 event
                 :interaction-id
                 :request-id)))))))

(defn store-events
  [_events])

(defn store-results-impl
  [ctx resp]
  (let [resp (fix-keys resp)]
    (log-response ctx)
    (doseq [i (:events resp)]
      (store-event ctx i))
    (doseq [i (:identities resp)]
      (store-identity ctx i))
    (doseq [i (:sequences resp)]
      (store-sequence ctx i))
    (doseq [i (:effects resp)]
      (store-command ctx i))
    (log/info resp)
    (log/info "Emulated 'with-transaction' dal function")
    resp))

(defmethod store-results
  :memory
  [{:keys [resp] :as ctx}]
  (store-results-impl ctx resp))

(defmethod get-events
  :memory
  [{:keys [id version]
    :as ctx}]
  {:pre [id]}
  "Reads event from vector under :event-store key"
  (log/info "Emulated 'get-events' dal function")
  (let [resp (->> (get-db ctx :event-store)
                  (filter #(and (= (:id %) id)
                                (if version (> (:event-seq %) version) true)))
                  (into [])
                  (sort-by #(:event-seq %)))]
    (log/info
     (format "Found: %s events with latest having version: %s"
             (count resp)
             (:event-seq (last resp))))
    resp))

(defmethod get-sequence-number-for-id
  :memory
  [{:keys [id] :as ctx}]
  {:pre [id]}
  (let [store (get-db ctx :sequence-store)
        entry (first (filter #(= (:id %) id)
                             store))]
    (:value entry)))

(defmethod get-command-response
  :memory
  [{:keys [request-id breadcrumbs] :as ctx}]

  (log/info "Emulating get-command-response-log" request-id breadcrumbs)
  (when (and request-id breadcrumbs)
    (let [store (get-db ctx :response-log)]
      (first
       (filter #(and (= (:request-id %) request-id)
                     (= (:breadcrumbs %) breadcrumbs))
               store)))))

(defmethod get-id-for-sequence-number
  :memory
  [{:keys [sequence] :as ctx}]
  {:pre [sequence]}
  (let [store (get-db ctx :sequence-store)
        entry (first (filter #(= (:value %) sequence)
                             store))]
    (:id entry)))

(defn get-max-event-seq-impl
  [{:keys [id] :as ctx}]
  (log/info "Emulated 'get-max-event-seq' dal function with fixed return value 0")
  (let [resp (map
              #(:event-seq %)
              (filter
               #(= (:id %) id)
               (get-db ctx :event-store)))]
    (if (> (count resp) 0)
      (apply
       max
       resp)
      0)))

(defmethod get-max-event-seq
  :memory
  [ctx]
  (get-max-event-seq-impl ctx))

(defmethod get-aggregate-id-by-identity
  :memory
  [{:keys [identity] :as ctx}]
  {:pre [identity]}
  (log/info "Emulating get-aggregate-id-by-identity" identity)
  (let [store (get-db ctx :identity-store)]
    (if (coll? identity)
      (->> store
           (filter (fn [it]
                     (some #(= %
                               (:identity it))
                           identity)))
           (reduce
            (fn [p v]
              (assoc p
                     (:identity v)
                     (:id v)))
            {}))

      (->> store
           (filter #(= (:identity %) identity))
           (first)
           :id))))

(defmethod log-request
  :memory
  [ctx body]
  (log/info "Storing mock request" body)
  (update-db ctx :command-log
             (fn [v] (conj (or v []) body))))

(defmethod log-request-error
  :memory
  [_ctx body error]
  (log/info "Should store mock request error" body error))

(defmethod log-dps
  :memory
  [{:keys [dps-resolved] :as ctx}]
  (log/debug "Storing mock dps" dps-resolved)
  (update-db ctx
             :dps-log
             (fn [v] (conj (or v []) dps-resolved)))
  ctx)

(defmethod log-response
  :memory
  [{:keys [response-summary request-id breadcrumbs] :as ctx}]
  (log/info "Storing mock response" response-summary)
  (update-db ctx
             :response-log
             (fn [v] (conj (or v [])
                           {:request-id  request-id
                            :breadcrumbs breadcrumbs
                            :data        response-summary}))))

(defmethod with-init
  :memory
  [ctx body-fn]
  (log/debug "Initializing memory event store")
  (body-fn ctx))

(defn register
  [ctx]
  (assoc ctx :edd-event-store :memory))

