(ns aws.filters.s3
  (:require [lambda.util :as util]
            [lambda.uuid :as uuid]
            [clojure.tools.logging :as log]
            [clojure.string :as string]
            [lambda.jwt :as jwt]
            [lambda.core :as lambda-core]
            [aws.ctx :as aws-ctx]
            [sdk.aws.s3 :as s3])
  (:import (lambda.core LambdaFilter)))

(defn parse-key
  [key]
  (log/info "Parsing key" key)
  (try
    (let [parts (string/split key #"/")
          realm (first parts)
          parts (rest parts)
          realm (if (= realm "upload")
                  "prod"
                  realm)
          date (if (re-matches #"[\d]{4}-[\d]{2}-[\d]{2}" (first parts))
                 (first parts)
                 (throw (ex-info "Missing date" {:parts parts
                                                 :error "Missing date"})))
          parts (rest parts)
          interaction-id (first parts)
          parts (rest parts)
          id (if (= 2 (count parts))
               (first parts)
               (-> parts
                   (first)
                   (string/split #"\.")
                   (first)))
          request-id (if (= 2 (count parts))
                       (-> parts
                           (second)
                           (string/split #"\.")
                           (first))
                       id)]
      {:request-id     (uuid/parse request-id)
       :interaction-id (uuid/parse interaction-id)
       :id             (uuid/parse (or id request-id))
       :date           date
       :realm          realm})
    (catch Exception e
      (log/error "Unable to parse key. Should be in format
                  /{{ realm }}/{{ yyyy-MM-dd }}/{{ interaction-id uuid }}/{{ request-id uuid }}.*
                  or
                 /{{ realm }}/{{ yyyy-MM-dd }}/{{ interaction-id uuid }}/{{ request-id uuid }}/{{ id uuid }}.* "
                 e)
      (throw (ex-info "Unable to parse key"
                      {:key  key
                       :data (ex-data e)})))))

(defn aws-s3-filter->valid-request?
  [request]
  (when (and (vector? request)
             (> (count request) 1))
    (throw (ex-info "Unknown constalation"
                    {:message "I dont know how you could have vector here since s3 is not batching
                                             triggers."})))
  (let [body (if (vector? request)
               (first request)
               request)]
    (and
     (contains? body :Records)
     (= (:eventSource (first (:Records body))) "aws:s3"))))

(defn is-folder-creation?
  [key]
  (string/ends-with? key "/"))

(defn aws-s3-filter->process-request
  [ctx request]
  (let [body request
        body (if (vector? body)
               (first body)
               body)
        ctx (-> ctx
                (assoc-in [:user :id] (name (:service-name ctx)))
                (assoc-in [:user :role] :non-interactive))

        record (first (:Records body))
        key (get-in record [:s3 :object :key])
        bucket (get-in record [:s3 :bucket :name])

        {:keys [request-id
                interaction-id
                date
                realm
                id] :as parsed-key}
        (parse-key key)]

    (log/info "Parsing success " parsed-key)
    {:request-id     request-id
     :interaction-id interaction-id
     :user           (name (:service-name ctx))
     :meta           {:realm (keyword realm)
                      :user  {:id    request-id
                              :email "non-interractiva@s3.amazonws.com"
                              :role  :non-interactive}}
     :commands       (if-not (is-folder-creation? key)
                       [{:cmd-id :object-uploaded
                         :id     id
                         :body   (s3/get-object ctx record)
                         :date   date
                         :bucket bucket
                         :key    key}]
                       [])}))

(defn aws-s3-filter->filter-request
  [ctx _config request filter-chain]

  (if (aws-s3-filter->valid-request? request)
    (let [processed (aws-s3-filter->process-request ctx request)]
      (lambda-core/continue-filter filter-chain
                                   ctx
                                   processed))
    (lambda-core/continue-filter filter-chain
                                 ctx
                                 request)))

(defn aws-s3-filter->init
  [ctx]
  (-> ctx
      jwt/fetch-jwks-keys
      aws-ctx/init
      jwt/ctx->aws-user-pool))

(deftype AWSS3Filter [config]
  LambdaFilter
  (init-filter [_this ctx]
    (aws-s3-filter->init ctx))
  (do-filter [_this ctx request filter-chain]
    (aws-s3-filter->filter-request ctx config request filter-chain)))
