(ns aws.aws
  (:require
   [lambda.util :as util]
   [clojure.tools.logging :as log]
   [lambda.request :as request]
   [sdk.aws.common :as common]
   [clojure.set :as clojure-set]
   [clojure.string :as string]
   [sdk.aws.cognito-idp :as cognito-idp]
   [sdk.aws.sqs :as sqs]))

(defn get-next-invocation
  [runtime-api]
  (util/http-request
   (str "http://" runtime-api "/2018-06-01/runtime/invocation/next")
   {:method :get
    :timeout 90000000}))

(defn get-next-request []
  (let [runtime-api (util/get-env "AWS_LAMBDA_RUNTIME_API")
        req (get-next-invocation runtime-api)]
    (if (-> req :body :isBase64Encoded)
      (update-in req [:body :body] util/base64decode)
      req)))

(def response
  {:statusCode 200
   :headers    {"Content-Type" "application/json"}})

(defn get-or-set
  [cache key get-fn]
  (let [current-time (util/get-current-time-ms)
        meta (get-in cache [:meta key])]
    (if (or (not (get cache key))
            (> (- current-time (get meta :time 0)) 1800000))
      (-> cache
          (assoc-in [:meta key] {:time current-time})
          (assoc key (common/retry
                      get-fn
                      3)))
      cache)))

(defn admin-auth
  [ctx]
  (let [{:keys [error] :as response} (cognito-idp/admin-initiate-auth ctx)]
    (if error
      response
      (get-in response [:AuthenticationResult :IdToken]))))

(defn get-token-from-cache
  [{:keys [application-scoped-cache] :as ctx}]
  (let [{:keys [id-token]}
        (swap! application-scoped-cache
               (fn [cache]
                 (get-or-set cache
                             :id-token
                             (fn []
                               (admin-auth ctx)))))]
    id-token))

(defn get-token
  [ctx]
  (if (:id-token ctx)
    (:id-token ctx)
    (get-token-from-cache ctx)))
